package com.amazon.discovery;

import com.amazon.discovery.testdiscoverables.AImplWithDiscoveriesDependency;
import com.amazon.discovery.testdiscoverables.AImplWithOptionalDependency;
import com.amazon.discovery.testdiscoverables.AImplWithRequiredDependency;
import com.amazon.discovery.testdiscoverables.BImpl;
import com.amazon.discovery.testdiscoverables.BImpl1;
import com.amazon.discovery.testdiscoverables.BImpl2;
import com.amazon.discovery.testdiscoverables.MyTestDiscoverableImpl1;
import com.amazon.discovery.testdiscoverables.MyTestDiscoverableImpl2;
import com.amazon.discovery.testinterfaces.A;
import com.amazon.discovery.testinterfaces.B;
import com.amazon.discovery.testinterfaces.MyTestDiscoverableContract;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;
import org.powermock.reflect.Whitebox;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * Unit tests for {@link Discovery}
 * Created by Ma, Don on 04/14/2018.
 * Copyright © 2018 Amazon.com. All Rights Reserved.
 */
@RunWith(PowerMockRunner.class)
@PrepareForTest({Discovery.class, DiscoveryProvider.class})
public class DiscoveryTest {

    @Before
    public void setup() {
        MyTestDiscoverableImpl1.initializationCount = 0;
    }

    @Test
    public void testDiscoverSingleSuccess() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(MyTestDiscoverableContract.class.getName(),
                Arrays.asList(MyTestDiscoverableImpl1.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));
        UniqueDiscovery<MyTestDiscoverableContract> discoverable =
                UniqueDiscovery.of(MyTestDiscoverableContract.class);
        Assert.assertNotNull(discoverable.value());
        Assert.assertTrue(discoverable.value() instanceof MyTestDiscoverableImpl1);
    }

    @Test
    public void testDiscoverSingleNotFound() {
        Discovery.setDiscoveryProvider(new DiscoveryProvider(Collections.EMPTY_MAP));
        UniqueDiscovery<MyTestDiscoverableContract> discoverable =
                UniqueDiscovery.of(MyTestDiscoverableContract.class);
        Assert.assertNull(discoverable.value());
    }

    @Test
    public void testDiscoverSingleZeroImpl() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(MyTestDiscoverableContract.class.getName(), Collections.EMPTY_LIST);
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));
        UniqueDiscovery<MyTestDiscoverableContract> discoverable =
                UniqueDiscovery.of(MyTestDiscoverableContract.class);
        Assert.assertNull(discoverable.value());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testDiscoverSingleMulti() {

        Collection<String> discoverables = new ArrayList<>();
        discoverables.add(MyTestDiscoverableImpl1.class.getName());
        discoverables.add(MyTestDiscoverableImpl2.class.getName());

        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(MyTestDiscoverableContract.class.getName(),
                discoverables);
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));
        UniqueDiscovery<MyTestDiscoverableContract> discoverable =
                UniqueDiscovery.of(MyTestDiscoverableContract.class);
    }

    @Test
    public void testDiscoverAllSuccess() {
        Collection<String> discoverables = new ArrayList<>();
        discoverables.add(MyTestDiscoverableImpl1.class.getName());
        discoverables.add(MyTestDiscoverableImpl2.class.getName());

        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(MyTestDiscoverableContract.class.getName(),
                discoverables);
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        Discoveries<MyTestDiscoverableContract> set = Discoveries.of(MyTestDiscoverableContract.class);
        Set<Class<? extends MyTestDiscoverableContract>> objects = new HashSet<>();
        for (MyTestDiscoverableContract discoverable : set) {
            objects.add(discoverable.getClass());
        }
        Assert.assertEquals(2, objects.size());
        Assert.assertTrue(objects.contains(MyTestDiscoverableImpl1.class));
        Assert.assertTrue(objects.contains(MyTestDiscoverableImpl2.class));
    }

    @Test
    public void testDiscoverAllFailure() {
        Discovery.setDiscoveryProvider(new DiscoveryProvider(Collections.EMPTY_MAP));
        Discoveries<MyTestDiscoverableContract> set = Discoveries.of(MyTestDiscoverableContract.class);
        Set<Class<? extends MyTestDiscoverableContract>> objects = new HashSet<>();
        for (MyTestDiscoverableContract discoverable : set) {
            objects.add(discoverable.getClass());
        }
        Assert.assertTrue(objects.isEmpty());
    }

    @Test
    public void testDiscoveryWithDefaultConfig() {
        Whitebox.setInternalState(Discovery.class, "discoveryProvider", (DiscoveryProvider)null);
        Discoveries<MyTestDiscoverableContract> discoveries = Discoveries.of(MyTestDiscoverableContract.class);
        List<MyTestDiscoverableContract> myDiscoverables = new ArrayList<>();
        for (MyTestDiscoverableContract discoverable : discoveries) {
            myDiscoverables.add(discoverable);
        }
        Assert.assertEquals(2, myDiscoverables.size());
    }

    @Test
    public void testGetDiscoveryProvider() {
        Whitebox.setInternalState(Discovery.class, "discoveryProvider", (DiscoveryProvider)null);
        DiscoveryProvider provider = Discovery.getDiscoveryProvider();
        Assert.assertNotNull(provider);
        Assert.assertEquals(provider.findTypeNames(MyTestDiscoverableContract.class.getName()).size(), 2);
    }

    @Test (expected = IllegalArgumentException.class)
    public void testGetDiscoveryProviderException() {
        Whitebox.setInternalState(Discovery.class, "discoveryProvider", (DiscoveryProvider)null);
        PowerMockito.mockStatic(DiscoveryProvider.class);
        try {
            PowerMockito.when(DiscoveryProvider.loadMappings(Mockito.any(Reader.class))).thenThrow(new IOException());
        } catch (Exception ex) {
            Assert.fail();
        }
        Discovery.getDiscoveryProvider();
    }

    @Test
    public void testLazyInitializationMulti() {
        Whitebox.setInternalState(Discovery.class, "discoveryProvider", (DiscoveryProvider)null);
        Discoveries<MyTestDiscoverableContract> discoveries = Discoveries.of(MyTestDiscoverableContract.class);
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 0);
        List<MyTestDiscoverableContract> myDiscoverables = new ArrayList<>();
        for (MyTestDiscoverableContract discoverable : discoveries) {
            myDiscoverables.add(discoverable);
        }
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 1);
    }

    @Test
    public void testLazyInitializationUnique() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(MyTestDiscoverableContract.class.getName(),
                Arrays.asList(MyTestDiscoverableImpl1.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 0);
        UniqueDiscovery<MyTestDiscoverableContract> discoverable =
                UniqueDiscovery.of(MyTestDiscoverableContract.class);
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 0);
        MyTestDiscoverableContract obj = discoverable.value();
        Assert.assertNotNull(obj);
        Assert.assertTrue(obj instanceof MyTestDiscoverableImpl1);
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 1);
        discoverable =
                UniqueDiscovery.of(MyTestDiscoverableContract.class);
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 1);
        MyTestDiscoverableContract obj2 = discoverable.value();
        Assert.assertNotNull(obj2);
        Assert.assertSame(obj, obj2);
        Assert.assertEquals(MyTestDiscoverableImpl1.initializationCount, 1);
    }

    /**
     * Note: DiscoveryTool static analysis should prevent this problematic mapping from being created.
     * This test is here for completeness only: if, for whatever reason, static analysis is not executed or
     * performing as expected, an un-satisfiable dependency on {@link RequiredUniqueDiscovery} should result
     * in a runtime exception on {@link Discoverable} creation.
     */
    @Test(expected = IllegalArgumentException.class)
    public void testDependencyOnRequiredUnique_whenZeroImplsArePresent_expectException() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithRequiredDependency.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        aDiscovery.value();
    }

    @Test
    public void testDependencyOnRequiredUnique_whenSingleImplIsPresent_expectDependencyValueExists() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithRequiredDependency.class.getName()));
        mappings.put(B.class.getName(),
                Arrays.asList(BImpl.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        A a = aDiscovery.value();
        Assert.assertTrue(a instanceof AImplWithRequiredDependency);

        UniqueDiscovery<B> bDiscovery = UniqueDiscovery.of(B.class);
        B b = bDiscovery.value();
        Assert.assertNotNull(b);
        Assert.assertEquals(b, ((AImplWithRequiredDependency)a).b.value());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testDependencyOnRequiredUnique_whenMultipleImplsArePresent_expectException() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithRequiredDependency.class.getName()));
        mappings.put(B.class.getName(),
                Arrays.asList(BImpl.class.getName(),
                        BImpl1.class.getName(),
                        BImpl2.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        aDiscovery.value();
    }

    @Test
    public void testDependencyOnOptionalUnique_whenZeroImplsArePresent_expectDependencyValueIsNull() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithOptionalDependency.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        A a = aDiscovery.value();
        Assert.assertTrue(a instanceof AImplWithOptionalDependency);

        UniqueDiscovery<B> bDiscovery = UniqueDiscovery.of(B.class);
        B b = bDiscovery.value();
        Assert.assertNull(b);
        Assert.assertNull(((AImplWithOptionalDependency)a).b.value());
    }

    @Test
    public void testDependencyOnOptionalUnique_whenSingleImplIsPresent_expectDependencyValueExists() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithOptionalDependency.class.getName()));
        mappings.put(B.class.getName(),
                Arrays.asList(BImpl.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        A a = aDiscovery.value();
        Assert.assertTrue(a instanceof AImplWithOptionalDependency);

        UniqueDiscovery<B> bDiscovery = UniqueDiscovery.of(B.class);
        B b = bDiscovery.value();
        Assert.assertNotNull(b);
        Assert.assertEquals(b, ((AImplWithOptionalDependency)a).b.value());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testDependencyOnOptionalUnique_whenMultipleImplsArePresent_expectException() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithOptionalDependency.class.getName()));
        mappings.put(B.class.getName(),
                Arrays.asList(BImpl.class.getName(),
                        BImpl1.class.getName(),
                        BImpl2.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        aDiscovery.value();
    }

    @Test
    public void testDependencyOnDiscoveries_whenZeroImplsArePresent_expectEmptyIterable() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithDiscoveriesDependency.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        A a = aDiscovery.value();
        Assert.assertTrue(a instanceof AImplWithDiscoveriesDependency);

        List<B> globalBList = listFromIterable(Discoveries.of(B.class));
        Assert.assertEquals(0, globalBList.size());
        List<B> injectedBList = listFromIterable(((AImplWithDiscoveriesDependency)a).bDiscoveries);
        Assert.assertEquals(0, injectedBList.size());
    }

    @Test
    public void testDependencyOnDiscoveries_whenSingleImplIsPresent_expectSingleItemInIterable() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithDiscoveriesDependency.class.getName()));
        mappings.put(B.class.getName(),
                Arrays.asList(BImpl.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        A a = aDiscovery.value();
        Assert.assertTrue(a instanceof AImplWithDiscoveriesDependency);

        List<B> globalBList = listFromIterable(Discoveries.of(B.class));
        Assert.assertEquals(1, globalBList.size());
        Assert.assertTrue(globalBList.get(0) instanceof BImpl);
        List<B> injectedBList = listFromIterable(((AImplWithDiscoveriesDependency)a).bDiscoveries);
        Assert.assertEquals(1, injectedBList.size());
        Assert.assertTrue(injectedBList.get(0) instanceof BImpl);
    }

    @Test
    public void testDependencyOnDiscoveries_whenMultipleImplsArePresent_expectItemsInIterable() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        mappings.put(A.class.getName(),
                Arrays.asList(AImplWithDiscoveriesDependency.class.getName()));
        mappings.put(B.class.getName(),
                Arrays.asList(BImpl.class.getName(),
                        BImpl1.class.getName(),
                        BImpl2.class.getName()));
        Discovery.setDiscoveryProvider(new DiscoveryProvider(mappings));

        UniqueDiscovery<A> aDiscovery = UniqueDiscovery.of(A.class);
        A a = aDiscovery.value();
        Assert.assertTrue(a instanceof AImplWithDiscoveriesDependency);

        List<B> globalBList = listFromIterable(Discoveries.of(B.class));
        Assert.assertEquals(3, globalBList.size());
        Assert.assertTrue(globalBList.get(0) instanceof BImpl);
        Assert.assertTrue(globalBList.get(1) instanceof BImpl1);
        Assert.assertTrue(globalBList.get(2) instanceof BImpl2);
        List<B> injectedBList = listFromIterable(((AImplWithDiscoveriesDependency)a).bDiscoveries);
        Assert.assertEquals(3, injectedBList.size());
        Assert.assertTrue(injectedBList.get(0) instanceof BImpl);
        Assert.assertTrue(injectedBList.get(1) instanceof BImpl1);
        Assert.assertTrue(injectedBList.get(2) instanceof BImpl2);
    }

    private <T> List<T> listFromIterable(Iterable<T> iterable) {
        List<T> list = new ArrayList<>();
        for (T item : iterable) {
            list.add(item);
        }
        return list;
    }
}
