package com.amazon.discovery;

import com.amazon.discovery.testdiscoverables.MyTestDiscoverableImpl1;
import com.amazon.discovery.testdiscoverables.MyTestDiscoverableImpl2;
import com.amazon.discovery.testinterfaces.MyTestDiscoverableContract;
import org.json.JSONException;
import org.junit.Assert;
import org.junit.Test;
import org.powermock.reflect.Whitebox;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

public class DiscoveryProviderTest {

    @Test
    public void testDeserialization() throws Exception {
        DiscoveryProvider provider = DiscoveryProvider.loadMappings(
                new InputStreamReader(getClass().getResourceAsStream("/discovery.json")));

        Assert.assertEquals(provider.findTypeNames(MyTestDiscoverableContract.class.getName()).size(), 2);
    }

    @Test
    public void testDerializationFailure() {
        try {
            DiscoveryProvider provider = DiscoveryProvider.loadMappings(
                    new InputStreamReader(getClass().getResourceAsStream("/discovery.jso")));
        } catch (Exception ex) {
            return;
        }
        Assert.fail();
    }

    @Test
    public void testSerialization() throws Exception {
        DiscoveryProvider provider = DiscoveryProvider.loadMappings(
                new InputStreamReader(getClass().getResourceAsStream("/discovery.json")));

        String serialized = provider.serialize();

        StringBuffer existing = new StringBuffer();
        try (BufferedReader br = new BufferedReader(
                new InputStreamReader(getClass().getResourceAsStream("/discovery.json")))) {
            boolean firstLine = true;
            String line = br.readLine();
            while (line != null) {
                if (!firstLine) {
                    existing.append('\n');
                }
                existing.append(line);
                line = br.readLine();
                firstLine = false;
            }
        }

        Assert.assertEquals(serialized, existing.toString());
    }

    @Test
    public void testFindInstance() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        DiscoveryProvider provider = new DiscoveryProvider(mappings);
        MyTestDiscoverableContract instance = provider.findInstance(MyTestDiscoverableImpl2.class.getName());
        Assert.assertNotNull(instance);

        MyTestDiscoverableContract instance2 = provider.findInstance(MyTestDiscoverableImpl2.class.getTypeName());
        Assert.assertSame(instance, instance2);
    }

    @Test
    public void testTypeLock() throws Exception {
        Map<String, Collection<String>> mappings = new HashMap<>();
        DiscoveryProvider provider = new DiscoveryProvider(mappings);
        MyTestDiscoverableContract instance = provider.findInstance(MyTestDiscoverableImpl1.class.getName());
        MyTestDiscoverableContract instance2 = provider.findInstance(MyTestDiscoverableImpl2.class.getTypeName());
        Map<String, Object>  typeLockMap =
                Whitebox.getInternalState(provider, "typeLockMap");
        Assert.assertTrue(typeLockMap.containsKey(MyTestDiscoverableImpl1.class.getName()));
        Assert.assertTrue(typeLockMap.containsKey(MyTestDiscoverableImpl2.class.getName()));
    }

    @Test (expected = IllegalArgumentException.class)
    public void testFindInstanceFailureNoPublicDefaultConstructor() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        DiscoveryProvider provider = new DiscoveryProvider(mappings);
        //DiscoveryProvider does not have a public default constructor.
        provider.findInstance(DiscoveryProvider.class.getName());
    }

    @Test (expected = IllegalArgumentException.class)
    public void testFindInstanceFailureClassNotFound() {
        Map<String, Collection<String>> mappings = new HashMap<>();
        DiscoveryProvider provider = new DiscoveryProvider(mappings);
        provider.findInstance("class.does.not.exist");
    }

    @Test
    public void testLoadMappingsWithEmptyBody() throws Exception {
        InputStreamReader configReader = new InputStreamReader(
                Discovery.class.getResourceAsStream("/empty.json"),
                StandardCharsets.UTF_8);

        DiscoveryProvider discoveryProvider = DiscoveryProvider.loadMappings(configReader);
        Map<String, Collection<String>> mappings =
                Whitebox.getInternalState(discoveryProvider, "discoverableTypes");
        Assert.assertTrue(mappings.isEmpty());
    }

    @Test
    public void testLoadMappingsWithNoImpls() throws Exception {
        InputStreamReader configReader = new InputStreamReader(
                Discovery.class.getResourceAsStream("/no_impls.json"),
                StandardCharsets.UTF_8);

        DiscoveryProvider discoveryProvider = DiscoveryProvider.loadMappings(configReader);
        Assert.assertTrue(discoveryProvider.findTypeNames(MyTestDiscoverableContract.class.getName()).isEmpty());
    }

    @Test(expected = JSONException.class)
    public void testLoadMappingsWithEmptyBindings() throws Exception {
        InputStreamReader configReader = new InputStreamReader(
                Discovery.class.getResourceAsStream("/empty_binding.json"),
                StandardCharsets.UTF_8);

        DiscoveryProvider.loadMappings(configReader);
    }

    @Test(expected = JSONException.class)
    public void testLoadMappingsWithNoTypes() throws Exception {
        InputStreamReader configReader = new InputStreamReader(
                Discovery.class.getResourceAsStream("/no_type.json"),
                StandardCharsets.UTF_8);

        DiscoveryProvider.loadMappings(configReader);
    }
}
