package com.amazon.discovery;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import java.util.Collection;

/**
 * A Lazy wrapper object for singleton {@link Discoverable} instance.
 * <p>
 * Sample usage:
 * <pre>
 *    // use the unique singleton instance.
 *    UniqueDiscovery&lt;MyContract&gt; myComponent = UniqueDiscovery.of(MyContract.class);
 *    MyContract discoverable = myComponent.value();
 *    if (discoverable != null) {
 *      //use the discoverable
 *    }
 * </pre>
 * Created by Ma, Don on 04/17/2018.<br>
 * Copyright © 2018 Amazon.com. All Rights Reserved.
 *
 * @param <T> the type of the {@link Discoverable}
 */
public final class UniqueDiscovery<T> {

    /**
     * the name of the {@link Discoverable} type.
     */
    private final String discoverableType;

    /**
     * Get an instance of {@link UniqueDiscovery} for the given type.
     * @param type the class type of the contract
     * @param <T> the type of the contract
     * @return the instance of {@link UniqueDiscovery}
     */
    @Nonnull
    public static <T> UniqueDiscovery<T> of(@Nonnull final Class<T> type) {
        return new UniqueDiscovery<>(type);
    }

    /**
     * Constructor with the given contract class type. If there are more than one
     * implementation fo the given type, an {@link IllegalArgumentException} will be thrown.
     *
     * @param contract the class type of the contract.
     */
    private UniqueDiscovery(@Nonnull final Class<T> contract) {
        Collection<String> discoverables =
                Discovery.getDiscoveryProvider().findTypeNames(contract.getName());
        if (discoverables != null) {
            if (discoverables.size() > 1) {
                String message = "More than one implementation of type " + contract + " found!";
                message += "Use the dependencies config file (-c command line) option with the DiscoveryTool";
                message += "in order to exclude implementations in shared code, or do not use UniqueDiscovery.";
                message += "See the SDDFInterface package's readme.md for more information on how to do that.";
                throw new IllegalArgumentException(message);
            } else if (discoverables.size() == 1) {
                discoverableType = discoverables.iterator().next();
            } else {
                discoverableType = null;
            }
        } else {
            discoverableType = null;
        }
    }

    /**
     * Get the discoverable instance.
     * @return the singleton instance of the discoverable if there is one and only one; null otherwise.
     */
    @CheckForNull
    public T value() {
        if (exists()) {
            return Discovery.getDiscoveryProvider().findInstance(discoverableType);
        }
        return null;
    }

    /**
     * Check if a unique discoverable exists.
     * @return true if there is one and only one implementation of the discoverable type
     */
    public boolean exists() {
        return discoverableType != null;
    }
}
