package com.amazon.discovery;

import javax.annotation.Nonnull;

/**
 * A Lazy wrapper object for singleton {@link Discoverable} instance which requires that the singleton is available at
 * runtime.
 *
 * Unlike the optional {@link UniqueDiscovery}, {@link RequiredUniqueDiscovery} is not accessible through a public
 * static accessor method. The only way to use {@link RequiredUniqueDiscovery} publicly is to declare it as an
 * initialization parameter in a {@link Discoverable}'s constructor.
 *
 * <p>
 * Sample usage:
 * <pre>{@code
 *     public class MyDiscoverable extends Discoverable, MyDiscoverableInterface {
 *         private final RequiredUniqueDiscovery<SomeService> someService;
 *         public MyDiscoverable(RequiredUniqueDiscovery<SomeService> someService) {
 *             this.someService = someService;
 *         }
 *
 *         public void doSomething() {
 *              someService.value().fetchData(...);
 *         }
 *     }
 *
 * }</pre>
 *
 * Copyright © 2018 Amazon.com. All Rights Reserved.
 *
 * @param <T> the type of the {@link Discoverable}
 */
public final class RequiredUniqueDiscovery<T> {

    /**
     * Get an instance of {@link UniqueDiscovery} for the given type.
     * @param type the class type of the contract
     * @param <T> the type of the contract
     * @return the instance of {@link UniqueDiscovery}
     */
    @Nonnull
    static <T> RequiredUniqueDiscovery<T> of(@Nonnull final Class<T> type) {
        return new RequiredUniqueDiscovery<>(type);
    }

    /**
     * The inner {@link UniqueDiscovery} instance which is ultimately used to fulfill the request for the value.
     */
    private final UniqueDiscovery<T> innerUniqueDiscovery;

    /**
     * Constructor with the given contract class type. If there are more than one
     * implementation fo the given type, an {@link IllegalArgumentException} will be thrown.
     *
     * @param contract the class type of the contract.
     */
    private RequiredUniqueDiscovery(@Nonnull final Class<T> contract) {
        innerUniqueDiscovery = UniqueDiscovery.of(contract);
        if (!innerUniqueDiscovery.exists()) {
            throw new IllegalStateException("Discoverable interface " + contract + " is required, but no "
                    + "implementation is present");
        }
    }

    /**
     * Get the discoverable instance.
     * @return the singleton instance of the discoverable.
     */
    @Nonnull
    public T value() {
        T value = innerUniqueDiscovery.value();
        // As part of initialization we have already confirmed that the innerUniqueDiscovery's value will be present.
        // However, we need to check again here to satisfy FindBugs since we're forcefully converting from @Nullable
        // to @Nonnull. The if (value == null) conditional block here will never be executed.
        if (value == null) {
            throw new IllegalStateException("No implementation is present");
        }
        return value;
    }
}
