package com.amazon.discovery;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import java.io.BufferedReader;
import java.io.Reader;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Hosts the discovery mapping.
 * <p>
 * This has package level access and should be used internally by SDDFCore
 * to set up the {@link Discovery} after loading the mappings.
 * <p>
 *
 * Created by Ma, Don on 04/18/2018.<br>
 * Copyright © 2018 Amazon.com. All Rights Reserved.
 */
class DiscoveryProvider {

    /**
     * Indentation count for the json output.
     */
    private static final int INDENTATION = 4;

    /**
     * json key for discovery.
     */
    private static final String KEY_DISCOVERY = "discovery";
    /**
     * json key for contract.
     */
    private static final String KEY_CONTRACT = "contract";
    /**
     * json key for implementations array.
     */
    private static final String KEY_IMPLEMENTATIONS = "implementations";

    /**
     * discoverable types mapping.
     */
    @Nonnull
    private final Map<String, Collection<String>> discoverableTypes;

    /**
     * cache of initialized objects.
     */
    private final Map<String, Object> objectsMap = new HashMap<>();

    /**
     * lock for each type.
     */
    private final Map<String, Object> typeLockMap = new HashMap();

    /**
     * constructor with the mappings.
     * @param mappings the mappings for discoverables.
     */
    DiscoveryProvider(@Nonnull final Map<String, Collection<String>> mappings) {
        discoverableTypes = mappings;
    }

    /**
     * find all the type names of the given contract.
     *
     * @param contractName the type name of the contract
     * @return a collection of the type names implementing the contract.
     */
    @CheckForNull
    public Collection<String> findTypeNames(@Nonnull final String contractName) {
        return discoverableTypes.get(contractName);
    }

    /**
     * get the instance of the given type name, if not found, it will try to instantiate it.
     *
     * @param typeName the type name of the implementation
     * @param <T>      the type of the implementation
     * @return the instance of the type.
     */
    @Nonnull
    public <T> T findInstance(@Nonnull final String typeName) {
        T discoverable;
        synchronized (objectsMap) {
            discoverable =  (T) objectsMap.get(typeName);
        }

        if (discoverable != null) {
            return discoverable;
        }

        Object typeLock;
        synchronized (typeLockMap) {
            typeLock = typeLockMap.get(typeName);
            if (typeLock == null) {
                typeLock = new Object();
                typeLockMap.put(typeName, typeLock);
            }
        }

        synchronized (typeLock) {
            synchronized (objectsMap) {
                discoverable = (T) objectsMap.get(typeName);
            }
            if (discoverable == null) {
                try {
                    Class<?> clazz = Class.forName(typeName);
                    Constructor<?> constructor = DiscoverableInitializationUtils.getDiscoverableConstructor(clazz);
                    List<Dependency> dependencies = DiscoverableInitializationUtils.getDependencies(constructor);
                    Object[] params = new Object[dependencies.size()];
                    for (int i = 0; i < dependencies.size(); i++) {
                        Dependency dependency = dependencies.get(i);
                        final Class interfaceClass = dependency.getRequestedClass();
                        switch (dependency.getType()) {
                            case OPTIONAL_UNIQUE:
                                params[i] = UniqueDiscovery.of(interfaceClass);
                                break;
                            case REQUIRED_UNIQUE:
                                params[i] = RequiredUniqueDiscovery.of(interfaceClass);
                                break;
                            case DISCOVERIES:
                                params[i] = Discoveries.of(interfaceClass);
                                break;
                            default:
                                break;
                        }
                    }
                    discoverable = (T) constructor.newInstance(params);
                    objectsMap.put(typeName, discoverable);
                } catch (Exception ex) {
                    throw new IllegalArgumentException("Failed to instantiate object of type " + typeName, ex);
                }
                synchronized (objectsMap) {
                    objectsMap.put(typeName, discoverable);
                }
            }
        }

        return discoverable;
    }


    /**
     * deserialize the discoverable json mappings, and register them in {@link Discovery}.
     *
     * @param mappingsReader the {@link Reader} of the serialized mapping file.
     * @throws Exception when deserialization fails.
     * @return the {@link DiscoveryProvider} instance with the mapping.
     */
    @Nonnull
    static DiscoveryProvider loadMappings(@Nonnull final Reader mappingsReader) throws Exception {
        //read the whole json configuration
        StringBuilder jsonStr = new StringBuilder();
        try (Reader reader = mappingsReader;
             BufferedReader br = new BufferedReader(reader)) {
            String line = br.readLine();
            while (line != null) {
                jsonStr.append(line);
                line = br.readLine();
            }
        }
        Map<String, Collection<String>> mappings = new HashMap<>();
        //parse json string for discoverable mappings.
        JSONObject discoverablesMap = new JSONObject(jsonStr.toString());
        JSONArray discoveries = discoverablesMap.getJSONArray(KEY_DISCOVERY);
        for (int i = 0; i < discoveries.length(); i++) {
            JSONObject binding = discoveries.getJSONObject(i);
            String type = binding.getString(KEY_CONTRACT);
            JSONArray impls = binding.getJSONArray(KEY_IMPLEMENTATIONS);
            Collection<String> types = new ArrayList<>(impls.length());
            for (int j = 0; j < impls.length(); j++) {
                types.add(impls.getString(j));
            }
            mappings.put(type, types);
         }
        return new DiscoveryProvider(Collections.unmodifiableMap(mappings));
    }

    /**
     * serialize the type mappings to json strings.
     * {
     * "discovery": [{
     * "contract": "com.amazon.kindle.DiscoverableInterfaceX",
     * "implementations": [
     * "com.amazon.kindle.DiscoverableInterfaceXImpl1",
     * "com.amazon.kindle.DiscoverableInterfaceXImpl2",
     * ...
     * ]
     * }
     * }.
     *
     * @return the serialized json string.
     * @throws JSONException thrown when the json object is not well-formatted.
     */
    @Nonnull
    String serialize() throws JSONException {
        JSONObject discoverablesMap = new JSONObject();
        JSONArray discoveries = new JSONArray();
        discoverablesMap.put(DiscoveryProvider.KEY_DISCOVERY, discoveries);
        for (Map.Entry<String, Collection<String>> entry : discoverableTypes.entrySet()) {
            JSONObject binding = new JSONObject();
            binding.put(DiscoveryProvider.KEY_CONTRACT, entry.getKey());
            JSONArray impls = new JSONArray();
            for (String type : entry.getValue()) {
                impls.put(type);
            }
            binding.put(DiscoveryProvider.KEY_IMPLEMENTATIONS, impls);
            discoveries.put(binding);
        }
        return discoverablesMap.toString(INDENTATION);
    }
}
