package com.amazon.discovery;

import java.io.InputStream;

import javax.annotation.Nonnull;

/**
 * Provides an abstraction for loading discovery mappings.
 * <p>
 * By default, this uses the class loader to load the mappings from a file called discovery.json. However,
 * this behavior can be overridden if needed by calling
 * {@link DiscoveryLoader#replaceInputStreamProvider(com.amazon.discovery.DiscoveryLoader.InputStreamProvider)}.
 */
public final class DiscoveryLoader {
    /**
     * Private constructor. This class should not be instantiated.
     */
    private DiscoveryLoader() { }

    /**
     * The resource name for the discovery mapping.
     */
    private static final String DEFAULT_DISCOVERY_MAPPING_RESOURCE = "/discovery.json";

    /**
     * Provides an interface for getting an input stream representing the discovery mappings.
     */
    public interface InputStreamProvider {
        /**
         * Gets the input stream containing the discovery mapping.
         * @return an InputStream representing the discovery mapping
         */
        InputStream getInputStream();
    }

    /**
     * Used for the default mapping stream provider. This is not final because it can be overridden
     * by a call to {@link DiscoveryLoader#replaceInputStreamProvider(DiscoveryLoader.InputStreamProvider)}.
     */
    @Nonnull
    private static InputStreamProvider inputStreamProvider = new InputStreamProvider() {
        @java.lang.Override
        public InputStream getInputStream() {
            return DiscoveryLoader.class.getResourceAsStream(DEFAULT_DISCOVERY_MAPPING_RESOURCE);
        }
    };

    /**
     * Allows setting a new input stream provider.
     * <p>
     * This can be used in situations where the default mechanism of loading the discovery mappings
     * is undesirable for some reason.
     *
     * @param newProvider the new provider (replaces existing provider)
     * @return the previous (replaced) provider
     */
    @Nonnull
    public static synchronized InputStreamProvider replaceInputStreamProvider(
            @Nonnull final InputStreamProvider newProvider) {
        final InputStreamProvider oldProvider = inputStreamProvider;
        if (newProvider == null) {
            throw new NullPointerException("A null provider was passed - must be nonnull");
        }
        inputStreamProvider = newProvider;
        return oldProvider;
    }

    /**
     * Gets the current input stream provider. Internal use (package private).
     * @return the input stream provider
     */
    @Nonnull
    static synchronized InputStreamProvider getInputStreamProvider() {
        return inputStreamProvider;
    }
}
