package com.amazon.discovery;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;

/**
 * A wrapper object for {@link DiscoveryProvider}.
 * <p>
 *
 * Used internally only.
 * <p>
 *
 * Created by Ma, Don on 04/13/2018.<br>
 * Copyright © 2018 Amazon.com. All Rights Reserved.
 */
final class Discovery {

    /**
     * The discovery host.
     */
    private static DiscoveryProvider discoveryProvider;

    /**
     * Default private constructor.
     */
    private Discovery() { }

    /**
     * Set up the {@link DiscoveryProvider}.
     *
     * @param factory the {@link DiscoveryProvider} with the mappings.
     */
    static synchronized void setDiscoveryProvider(@Nonnull final DiscoveryProvider factory) {
        if (factory == null) {
            throw new NullPointerException("A null provider was passed - must be nonnull.");
        }
        discoveryProvider = factory;
    }

    /**
     * Reset the discovery provider to its original, non-modified state.
     */
    static synchronized void resetDiscoveryProvider() {
        discoveryProvider = null;
    }

    /**
     * Return the {@link DiscoveryProvider}.
     * @return the discovery host.
     */
    @Nonnull
    static synchronized DiscoveryProvider getDiscoveryProvider() {
        if (discoveryProvider == null) {
            discoveryProvider = createDefaultDiscoveryProvider();
            if (discoveryProvider == null) {
                throw new IllegalStateException("Error creating default discovery provider, mappings do not exist");
            }
        }
        return discoveryProvider;
    }

    /**
     * Creates a default discovery provider serving mappings from
     * {@code DiscoveryLoader.getInputStreamProvider().getInputStream()}.
     * This method is exposed exclusively for testing.
     * @return a new default discovery provider; {@code null} if the corresponding mappings do not exist.
     */
    @CheckForNull
    static DiscoveryProvider createDefaultDiscoveryProvider() {
        final InputStream is = DiscoveryLoader.getInputStreamProvider().getInputStream();
        if (is != null) {
            try (InputStreamReader configReader = new InputStreamReader(is, StandardCharsets.UTF_8)) {
                return DiscoveryProvider.loadMappings(configReader);
            } catch (Exception ex) {
                throw new IllegalArgumentException("Error loading default discovery mappings", ex);
            }
        } else {
            return null;
        }
    }
}
