package com.amazon.discovery;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

/**
 * An {@link Iterable} for discoverables of the a given type.
 * Actual discoverable instances will be lazily instantiated.
 * <p>
 * Sample usage:
 * <pre>
 *    //access a set of implementations of a given contract:
 *    Discoveries&lt;MyContract&gt; discoverables =  Discoveries.of(MyContract.class);
 *    for (MyContract discoverable : discoverables) {
 *        //use the discoverable.
 *    }
 *  </pre>
 * Created by Ma, Don on 04/17/2018.<br>
 * Copyright © 2018 Amazon.com. All Rights Reserved.
 *
 * @param <T> the type of the {@link Discoverable} in this set.
 */
public final class Discoveries<T> implements Iterable<T> {

    /**
     * The Discoverable type name.
     */
    @Nonnull
    private final String typeName;

    /**
     * Gets an instance of Discoveries for the given type.
     * @param type the class type of the contract.
     * @param <T> the type of the contract
     * @return an instance of {@link Discoveries}
     */
    @Nonnull
    public static <T> Discoveries<T> of(@Nonnull final Class<T> type) {
        return new Discoveries<>(type);
    }

    /**
     * private constructor of {@link Discoveries} with the given contract class type.
     *
     * @param type - the type for the discoverable set.
     */
    private Discoveries(@Nonnull final Class<T> type) {
        typeName = type.getName();
    }

    @Override
    @Nonnull
    public Iterator<T> iterator() {
        final DiscoveryProvider host = Discovery.getDiscoveryProvider();
        if (host != null) {
            final Collection<String> typeNames = host.findTypeNames(typeName);
            if (typeNames != null) {
                return new Iterator<T>() {
                    private final Iterator<String> typeIterator = typeNames.iterator();

                    @Override
                    public boolean hasNext() {
                        return typeIterator.hasNext();
                    }

                    @Override
                    public T next() {
                        String type = typeIterator.next();
                        return host.findInstance(type);
                    }
                };
            }
        }
        return Collections.emptyIterator();
    }
}
