package com.amazon.discovery;

import javax.annotation.Nonnull;
import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

/**
 * A collection of methods to help in initializing Discoverable classes.
 */
final class DiscoverableInitializationUtils {

    /**
     * Unused private constructor (required for Utils class).
     */
    private DiscoverableInitializationUtils() { }

    /**
     * Returns the single public constructor to be used for initialization, or null if the constructor does not exist.
     * @param clz - the class object
     * @return the single public constructor, or null if not present
     */
    static Constructor<?> getDiscoverableConstructor(@Nonnull final Class<?> clz) {
        Constructor<?> constructor = null;
        for (Constructor<?> c : clz.getConstructors()) {
            if (Modifier.isPublic(c.getModifiers())) {
                if (constructor != null) {
                    throw new IllegalArgumentException("Discoverable class " + clz.getName() + " has more than one "
                            + " public constructor. All Discoverables must have only one public constructor.");
                }
                constructor = c;
            }
        }
        return constructor;
    }

    /**
     * @param constructor the constructor to inspect
     * @return a list of {@link Dependency} objects which describe parameters required by the constructor
     */
    static List<Dependency> getDependencies(@Nonnull final Constructor<?> constructor) {
        Class<?>[] parameterTypes = constructor.getParameterTypes();
        Type[] genericParameterTypes = constructor.getGenericParameterTypes();
        List<Dependency> dependencies = new ArrayList<>(parameterTypes.length);
        for (int i = 0; i < parameterTypes.length; i++) {
            Class<?> clazz = parameterTypes[i];
            final DependencyType dependencyType;
            if (!(genericParameterTypes[i] instanceof ParameterizedType)) {
                throw new IllegalArgumentException("The constructor for " + constructor.getDeclaringClass()
                        + " requires invalid type " + clazz.getName() + ". Only RequiredUniqueDiscovery<T>, "
                        + "UniqueDiscovery<T>, and Discoveries<T> are allowed.");
            }

            if (clazz.equals(RequiredUniqueDiscovery.class)) {
                dependencyType = DependencyType.REQUIRED_UNIQUE;
            } else if (clazz.equals(UniqueDiscovery.class)) {
                dependencyType = DependencyType.OPTIONAL_UNIQUE;
            } else if (clazz.equals(Discoveries.class)) {
                dependencyType = DependencyType.DISCOVERIES;
            } else {
                throw new IllegalArgumentException("The constructor for " + constructor.getDeclaringClass()
                        + " requires invalid type " + clazz.getName() + ". Only RequiredUniqueDiscovery<T>, "
                        + "UniqueDiscovery<T>, and Discoveries<T> are allowed.");
            }

            final ParameterizedType parameterizedType = (ParameterizedType) genericParameterTypes[i];
            final Type requestedType = parameterizedType.getActualTypeArguments()[0];
            if (!(requestedType instanceof Class)) {
                throw new IllegalArgumentException("The constructor for " + constructor.getDeclaringClass()
                        + " specifies invalid generic type " + requestedType + ". Only Class types are allowed.");
            }
            dependencies.add(new Dependency(dependencyType, (Class) requestedType));
        }
        return dependencies;
    }
}
