  
local apollonode = require "apolloutility.apollonode"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local venuscore = require "venuscore"
local Object = require "classic"
local AsyncImageLoader = require "apolloutility.async_image_loader.async_image_loader"
local venusjson = require "venusjson"


local PostEffectRender = Object:extend();

local function DeepCopy(v)
  local t = type(v)
  if t == "table" then
    local c = {}
    for k, d in pairs(v) do
      c[k] = DeepCopy(d)
    end
    return c
  else
    return v
  end
end

--render_type是为区分主相机和rect上的相机的后处理参数："maincamera"为主相机，"rect"为分相机

function PostEffectRender:new(rootConfig, camera)

  local image_seq_list = rootConfig.ImageSequences
  local format_string_list = rootConfig.FormatStrings
  local material_list = rootConfig.Materials;
  local track_list = rootConfig.Tracks;
  local setup_once_list = rootConfig.SetupOnce

  self.camera = camera
  self.show = true

  self.posteffect_image_seq_list = {}
  self.posteffect_format_string_closure = {}
  self.posteffect_list = {};
  self.posteffect_material_list = {};
  self.posteffect_track_list = {};
  
  self.posteffect_texture = nil;

  self:ParseImageSequences(image_seq_list)

  self:ParseFormatStrings(format_string_list)

  self:ParseMaterials(material_list)

  for i = 1, #self.posteffect_list do
    camera:AttachPostEffect(self.posteffect_list[i])
  end

  self:ParseSetupOnce(setup_once_list)

  self:ParseTracks(track_list)

end

function PostEffectRender:Release()
  LOG("PostEffectRender:Release")
  for i = 1, #self.posteffect_list do
    self.camera:DetachPostEffect(self.posteffect_list[i])
  end
end

function PostEffectRender:ParseImageSequences(image_seq_list)

  if image_seq_list == nil then
    return false
  end

  local image_seq_cnt = #image_seq_list

  for i = 1, image_seq_cnt do

    local image_seq_item = image_seq_list[i]

    local pathFormat = image_seq_item.PathFormat

    local startNumber = image_seq_item.StartNumber

    local count = image_seq_item.Count

    local imageLoader = AsyncImageLoader(
            pathFormat,
            startNumber,
            count,
            false)

    table.insert(self.posteffect_image_seq_list, imageLoader)
  end

  return true

end

function PostEffectRender:ParseFormatStrings(format_string_list)

  if format_string_list == nil then
    return false
  end

  local format_string_list_cnt = #format_string_list

  for i = 1, format_string_list_cnt do
    local item = format_string_list[i]
    local format = item.Format
    local formatData = item.FormatData
    local script = string.format(
            "return function(value,year,month,day,hour,min,sec) return string.format(\"%s\",%s) end",
            format,
            formatData)
    local f, e = load(script)
    if f then
      local closure = f()
      table.insert(self.posteffect_format_string_closure, closure)
    else
      ERROR("failed to compile string format script")
    end
  end
  return true
end

function PostEffectRender:ParseMaterials(material_list)

  if material_list == nil then
    return false
  end

  self.posteffect_material_list = DeepCopy(material_list)

  local material_cnt = #material_list;

  --处理post effect
  for i = 1, material_cnt do

    local item = material_list[i]

    local postEffect = apollonode.PostEffect();

    postEffect:Disable();

    local path = venuscore.IFileSystem:PathAssembly(item.Path);

    postEffect:CreateResource(path);

    for j = 1, #item.Parameters do
      postEffect:RegisterParameter(item.Parameters[j].Name)
    end

    table.insert(self.posteffect_list, postEffect);

  end

  return true
end


function PostEffectRender:ParseSetupOnce(setup_once_list)

  if setup_once_list == nil then
    return false
  end

  local setup_once_cnt = #setup_once_list

  for i = 1, setup_once_cnt do
    local setupOnceItem = setup_once_list[i]
    local materialIndex = setupOnceItem.MaterialIndex
    local parameterValues = setupOnceItem.ParameterValues
    local parameterValuesCnt = #parameterValues

    local postEffect = self.posteffect_list[materialIndex]
    local material = self.posteffect_material_list[materialIndex]

    if postEffect == nil or material == nil then
      ERROR("Incorrect material index")
      return false
    end

    for j = 1, parameterValuesCnt do
      local param_index = parameterValues[j].ParameterIndex
      local param_value = parameterValues[j].Value
      local param_item = material.Parameters[param_index]

      if param_item == nil then
        ERROR("Incorrect parameter index")
        return false
      end

      local param_name = param_item.Name
      local param_type = param_item.Type

      local post_effect_param_value = self:_WrapParameterValue(param_value, param_type)

      if post_effect_param_value ~= nil then
        post_effect_param_value = self:_PostWrapParameterValue(
                param_type,
                post_effect_param_value,
                parameterValues[j])
        postEffect[param_name](postEffect, post_effect_param_value)
      end
    end
  end

  return true

end

function PostEffectRender:ParseTracks(track_list)

  if track_list == nil then
    return false
  end

  self.posteffect_track_list = DeepCopy(track_list)

  return true
end


function PostEffectRender:Update(frameIndex)

  for i = 1, #self.posteffect_list do
    self.posteffect_list[i]:Disable();
  end

  if not self.show then
    return
  end

  --enable/disable posteffect && set parameters
  local frameIndex = frameIndex - 1; --设计要求配置中帧号从0开始

  local trackCnt = #self.posteffect_track_list;

  for i = 1, trackCnt do

    local trackItem = self.posteffect_track_list[i]

    if frameIndex >= trackItem.Start or frameIndex < trackItem.Start + trackItem.Duration then
      local materialIndex = trackItem.MaterialIndex
      local parameterValues = trackItem.ParameterValues
      local parameterValuesCnt = #parameterValues
      local postEffect = self.posteffect_list[materialIndex]
      local material = self.posteffect_material_list[materialIndex]

      if postEffect == nil or material == nil then
        ERROR("incorrect material index")
        return
      end

      postEffect:Enable()

      for j = 1, parameterValuesCnt do

        local parameterValue = parameterValues[j]

        self:UpdateParameterValue(frameIndex, parameterValue, postEffect, material)

      end
    end
  end
end

function PostEffectRender:UpdateParameterValue(frameIndex, parameterValue, postEffect, material)

  local paramIndex = parameterValue.ParameterIndex
  local interpolate = parameterValue.Interpolate
  local table = parameterValue.Table
  local tableSize = #table

  local paramItem = material.Parameters[paramIndex]
  if paramItem == nil then
    ERROR("incorrect parameter index")
    return
  end
  local paramName = paramItem.Name
  local paramType = paramItem.Type

  if tableSize == 0 or frameIndex < table[1].Time then
    return
  end

  for i = 1, tableSize - 1 do
    local tableItem1 = table[i]
    local tableItem2 = table[i+1]
    local time1 = tableItem1.Time
    local time2 = tableItem2.Time
    if frameIndex >= time1 and frameIndex < time2 then
      if not interpolate then
        local val1 = self:_WrapParameterValue(tableItem1.Value, paramType)
        if val1 == nil then
          ERROR("failed to wrap value")
        else
          val1 = self:_PostWrapParameterValue(paramType, val1, parameterValue)
          postEffect[paramName](postEffect, val1)
        end
      else
        local val1 = self:_WrapParameterValue(tableItem1.Value, paramType)
        local val2 = self:_WrapParameterValue(tableItem2.Value, paramType)
        if val1 == nil or val2 == nil then
          ERROR("failed to wrap value")
        else
          local ratio = (frameIndex - time1) / (time2 - time1)
          local val3 = val1 + (val2 - val1) * ratio
          val3 = self:_PostWrapParameterValue(paramType, val3, parameterValue)
          postEffect[paramName](postEffect, val3)
        end
      end
      return
    end
  end

  if frameIndex >= table[tableSize].Time then
    local val = self:_WrapParameterValue(table[tableSize].Value, paramType)
    if val == nil then
      ERROR("failed to wrap value")
    else
      val = self:_PostWrapParameterValue(paramType, val, parameterValue)
      postEffect[paramName](postEffect, val)
    end
    return
  end
end

function PostEffectRender:_WrapParameterValue(param_value, param_type)

  local post_effect_param_value = nil

  if param_type == "immediate" then

    post_effect_param_value = param_value

  elseif param_type == "vector" then

    local param_value_len = #param_value
    if param_value_len == 1 then
      post_effect_param_value = mathfunction.vector1(
              param_value[1])
    elseif param_value_len == 2 then
      post_effect_param_value = mathfunction.vector2(
              param_value[1],
              param_value[2])
    elseif param_value_len == 3 then
      post_effect_param_value = mathfunction.vector3(
              param_value[1],
              param_value[2],
              param_value[3])
    elseif param_value_len == 4 then
      post_effect_param_value = mathfunction.vector4(
              param_value[1],
              param_value[2],
              param_value[3],
              param_value[4])
    end
  elseif param_type == "json" then

    post_effect_param_value = venusjson.StoreJsonString(param_value)

  elseif param_type == "imageIndex" then

    post_effect_param_value = param_value

  elseif param_type == "formatString" then

    post_effect_param_value = param_value

  end

  return post_effect_param_value
end

function PostEffectRender:_PostWrapParameterValue(paramType, paramValue, paramValueItem)
  if paramType == "imageIndex" then
    local imageSequenceIndex = paramValueItem.ImageSequenceIndex
    local imageLoader = self.posteffect_image_seq_list[imageSequenceIndex]
    return imageLoader:Load(paramValue)
  elseif paramType == "formatString" then
    local formatStringIndex = paramValueItem.FormatStringIndex
    local closure = self.posteffect_format_string_closure[formatStringIndex]
    local dateTable = os.date("*t", os.time())
    return closure(
            paramValue,
            dateTable.year,
            dateTable.month,
            dateTable.day,
            dateTable.hour,
            dateTable.min,
            dateTable.sec)
  else
    return paramValue
  end
end

function PostEffectRender:SetShow(show)
  if not show then
    for i = 1, #self.posteffect_list do
      self.posteffect_list[i]:Disable();
    end
  end
  self.show = show
end

return PostEffectRender
