--@author      : Chen Wei
--@date        : 2019-06-25
--@description : lua script for LineEffect
--@version     : 1.0

require "utility"

local Object                = require "classic"
local apolloengine          = require "apolloengine"
local mathfunction          = require "mathfunction"
local apollonode            = require "apolloutility.apollonode"
local defined               = require "apolloutility.defiend"
local renderqueue           = require "apolloutility.renderqueue"
local HistoryFrameManager   = require "apolloutility.basehistoricalframe"
local LineStripRender       = require "apolloutility.linestriprender"
local LineSystem            = require "lineeffect"

local LineEffect = Object:extend();

function LineEffect:new(main_camera, sequence, line_num, vertex_num_per_line, line_system_num, is_vertical)
    -- create line system
    self.line_num            = line_num;
    self.vertex_num_per_line = vertex_num_per_line;
    self.line_system_num     = line_system_num;
    self.is_vertical         = is_vertical;

    self.line_system         = LineSystem.LineEffect(line_num, vertex_num_per_line, line_system_num, is_vertical);

    -- set history frame manger
    self.history_frame_manager = HistoryFrameManager(main_camera, 2, sequence, defined.blit_material_path);

    -- set motion detect camera
    local fbo_size = mathfunction.vector2(vertex_num_per_line, line_num);
    self.motion_detect_camera = apollonode.CameraNode(fbo_size);
    self.motion_detect_camera:Activate();
    self.motion_detect_camera:SetSequence(sequence);

    -- create render target & tex & tex_stream
    self.rt, self.tex = self:_CreateRenderTarget();
    self.tex_stream = nil;

    -- attach render target to camera
    self.motion_detect_camera:AttachRenderTarget(self.rt);

    -- create uniform slots
    apolloengine.ShaderEntity.UNIFORM_CUR_FRAME = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
                                                                                                "UNIFORM_CUR_FRAME");
    apolloengine.ShaderEntity.UNIFORM_PRE_FRAME = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
                                                                                                "UNIFORM_PRE_FRAME");
    apolloengine.ShaderEntity.UNIFORM_TEX_SIZE  = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
                                                                                                "UNIFORM_TEX_SIZE");
    apolloengine.ShaderEntity.UNIFORM_IS_HORIZONTAL_LINE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
                                                                                                         "UNIFORM_IS_HORIZONTAL_LINE");

    -- get resolution
    self.resolution = main_camera:GetCameraResolution();

    -- set quad node to draw fbo(camera)
    local material = "docs:shadereffect/material/line.material"

    self.quad_node = apollonode.QuadNode();
    self.quad_node:SetShow(true);
    self.quad_node:CreateResource(material, true);
    self.quad_node:SetSequence(sequence);
    

    self.line_strip_render_queue = {};
    
    for i = 1, line_system_num, 1 do
        --NOTE(Chen Wei): need modify line_num / line_system_num
        local line_strip_render = LineStripRender(line_num / line_system_num,
                                                  vertex_num_per_line,
                                                  mathfunction.vector4(1.0, 1.0, 1.0, 1.0));
        table.insert(self.line_strip_render_queue, line_strip_render);
    end
    

end

function LineEffect:_CreateRenderTarget()
    local clear_color = mathfunction.vector4(0.0, 0.0, 0.0, 0.0);
    local resolution  = self.motion_detect_camera:GetCameraResolution();
    
    local rt = apolloengine.RenderTargetEntity();
    rt:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
                                                      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
                                                      clear_color,
                                                      mathfunction.vector4(0, 0, resolution:x(), resolution:y()),
                                                      resolution)
                    );

    local tex = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
    tex:PushMetadata(apolloengine.TextureBufferMetadata(resolution,
                                                        apolloengine.TextureEntity.TT_TEXTURE2D,
                                                        apolloengine.TextureEntity.TU_READ,
                                                        apolloengine.TextureEntity.PF_R8G8B8A8,
                                                        1, false,
                                                        apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                        apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                        apolloengine.TextureEntity.TF_LINEAR,
                                                        apolloengine.TextureEntity.TF_LINEAR));
                    
    
    rt:CreateResource();

  return rt, tex;
end

function LineEffect:Update()
    -- update history frame manager
    self.history_frame_manager:SetShow(true);
    self.history_frame_manager:Update();

    -- draw to calculate motion info

    local tex_queue = self.history_frame_manager:GetFrame();
    self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_CUR_FRAME, tex_queue[1]);
    self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_PRE_FRAME, tex_queue[2]);

    local tex_size = mathfunction.vector2(self.resolution:x(), self.resolution:y());
    self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_TEX_SIZE, tex_size);

    if self.is_vertical then
        self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_IS_HORIZONTAL_LINE, mathfunction.vector1(0.0));
    else
        self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_IS_HORIZONTAL_LINE, mathfunction.vector1(1.0));
    end
    
    
    -- read motion data from texture to texture stream
    self.tex_stream = self.rt:GetAttachmentStream(apolloengine.RenderTargetEntity.TA_COLOR_0);

    -- add force based on motion info
    self.line_system:SetForceFromVerticesMotionInfo(self.tex_stream);

    -- simulate line effect system
    -- for i = 1, self.line_system_num, 1 do
    --     self.line_system:Step(i - 1, 0.005, 6);
    -- end

    self.line_system:StepAll(0.005, 6);

    -- draw lines
    for i = 1, self.line_system_num, 1 do
        local vertex_stream = self.line_system:VertexPositionStream(i - 1);
        self.line_strip_render_queue[i]:Update(vertex_stream);
    end
    
end

return LineEffect