--@author      : Zhu Fei
--@date        : 2019-06-27
--@description : lua script to create patch image for BlingEffect
--@version     : 1.0

local Object       = require "classic"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local apollonode   = require "apolloutility.apollonode"
local venuscore    = require "venuscore"

--[[
    Generate a subsampled image of current video frame by splitting it into non-overlapping patches. Pixel intensity
    of subsampled image is computed as mean intensity of pixels in corresponding patch of input image.
    All channels of the output patch image store the same value, i.e., the computed intensity value.

    Usage:
    1. Initialize a PatchImageCreator.
    2. Call CreatePatchImage() for multiple video frames, and use PatchImageCreator.patch_image_cpu to get the patch image.
    3. Destroy PatchImageCreator when it's no longer needed.
]]
local PatchImageCreator = Object:extend();

function PatchImageCreator:new(frame_width, frame_height, patch_size, render_sequence)
    local patch_img_width  = math.ceil(frame_width / patch_size);
    local patch_img_height = math.ceil(frame_height / patch_size);
    
    -- setup fbo
    local fbo_size      = mathfunction.vector2(patch_img_width, patch_img_height);
    local clear_color   = mathfunction.vector4(0.0, 0.0, 0.0, 0.0);
    self.render_target  = apolloengine.RenderTargetEntity();
    self.render_target:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
                                                                      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
                                                                      clear_color,
                                                                      mathfunction.vector4(0, 0, patch_img_width, patch_img_height),
                                                                      fbo_size));
    self.patch_image_gpu = self.render_target:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
    self.patch_image_gpu:PushMetadata(apolloengine.TextureBufferMetadata(fbo_size,
                                                                         apolloengine.TextureEntity.TT_TEXTURE2D,
                                                                         apolloengine.TextureEntity.TU_READ,
                                                                         apolloengine.TextureEntity.PF_R8G8B8A8,
                                                                         1, false,
                                                                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                                         apolloengine.TextureEntity.TF_LINEAR,
                                                                         apolloengine.TextureEntity.TF_LINEAR));
    self.render_target:CreateResource();
    
    -- attach fbo to camera
    self.camera = apollonode.CameraNode(fbo_size);
    self.camera:Activate();
    self.camera:SetSequence(render_sequence);
    self.camera:AttachRenderTarget(self.render_target);

    -- create shader parameter slot
    apolloengine.ShaderEntity.PATCH_SIZE       = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "PATCH_SIZE");
    apolloengine.ShaderEntity.FRAME_IMG        = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "FRAME_IMG");
    apolloengine.ShaderEntity.TEX_COORD_OFFSET = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEX_COORD_OFFSET");

    -- create quad node to draw fbo(camera)
    local material = venuscore.IFileSystem:PathAssembly("docs:blingbling/material/create_patch_img.material")
    self.quad_node = apollonode.QuadNode();
    self.quad_node:SetShow(true);
    self.quad_node:CreateResource(material, true);
    self.quad_node:SetSequence(render_sequence);
    self.quad_node:SetParameter(apolloengine.ShaderEntity.PATCH_SIZE, mathfunction.vector1(patch_size));
    self.quad_node:SetParameter(apolloengine.ShaderEntity.TEX_COORD_OFFSET, mathfunction.vector2(1.0 / frame_width, 1.0 / frame_height));

    -- output patch image
    self.patch_image_cpu = nil;

end

function PatchImageCreator:CreatePatchImage(frame_texture)
    -- set shader parameters
    self.quad_node:SetParameter(apolloengine.ShaderEntity.FRAME_IMG, frame_texture);

    -- read render result to texture stream
    self.patch_image_cpu = self.render_target:GetAttachmentStream(apolloengine.RenderTargetEntity.TA_COLOR_0);
    return self.patch_image_cpu;
end

return PatchImageCreator;