--@author      : Zhu Fei
--@date        : 2019-06-27
--@description : lua script for BlingEffect
--@version     : 1.0

local Object        = require "classic"
local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local rendernode    = require "apolloutility.apollonode.rendernode"
local venusjson     = require "venusjson"
local mathfunction  = require "mathfunction"
local videodecet    = require "videodecet"
local bling         = require "blingeffect"
local patch_creator = require "blingbling.patch_img_creator"
local venuscore     = require "venuscore"
local renderqueue   = require "apolloutility.renderqueue"

--[[
    Render Blings as billboards on background frame.
    Usage:
    1. Create a BlingEffect object, make sure patch_creator_render_sequence is less than that of the main camera.
    2. call Update() in rendering Update()
]]

local BlingEffect = {}
local patch_creator_render_sequence = -4000

function BlingEffect:Initialize(cameraLayer)
    self.camera = renderqueue:GetCamera(cameraLayer)
    self.initialized = false
    self.showing = true
end

function BlingEffect:LoadConfig(config_path)
    local rootconfig   = venusjson.LaodJsonFile(config_path);
    if rootconfig == nil then
        return false
    end
    if rootconfig.bling == nil then
        return false
    end
    local config       = rootconfig.bling.config;
    if config == nil then
        return false
    end
    local texturePath = rootconfig.bling.texture
    if texturePath == nil then
        return false
    end

    local rootDir = string.match(config_path, "(.+)/[^/]*%.%w+$")
    venuscore.IFileSystem:SetResourcePath(rootDir .. "/")

    local blingconfig  = bling.BlingEffectConfig();
    local resolution   = apolloengine.Framework:GetResolution();
    blingconfig:SetFrameWidth(resolution:x());
    blingconfig:SetFrameHeight(resolution:y());
    blingconfig:SetPatchSize(config.patch_size);
    blingconfig:SetMaxBlingSize(config.max_bling_size);
    blingconfig:SetBlingSizeCurveOrder(config.bling_size_curve_order);
    blingconfig:SetMaxBlingNum(config.max_bling_num);
    blingconfig:SetIntensityThreshold(config.intensity_threshold);
    blingconfig:SetIntensityDeltaThreshold(config.intensity_delta_threshold);
    blingconfig:SetNeighborCountThreshold(config.neighbor_count_threshold);

    self.patch_creator = patch_creator(blingconfig:GetFrameWidth(), blingconfig:GetFrameHeight(), blingconfig:GetPatchSize(), patch_creator_render_sequence);
    self.bling_creator = bling.BlingEffect(blingconfig);
    self.rendernode    = rendernode();
    --TODO: set camera sequence to rendernode

    -- create shader parameter slot
    apolloengine.ShaderEntity.POINT_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "POINT_TEXTURE");
    apolloengine.ShaderEntity.FRAME_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "FRAME_TEXTURE");

    -- load bling texture
    texturePath = venuscore.IFileSystem:PathAssembly(texturePath)
    self.rendernode:PushTextureMetadata(apolloengine.ShaderEntity.POINT_TEXTURE, texturePath);
    
    -- initialize render node, render with main camera
    self.dummy_stream = apolloengine.VertexStream();
    self.dummy_stream:ReserveBuffer(blingconfig:GetMaxBlingNum() * 6);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.rendernode:SetShow(false);
    self.rendernode:CreateResource(venuscore.IFileSystem:PathAssembly("docs:blingbling/material/render_bling.material"),
                                   apolloengine.RenderObjectEntity.RM_TRIANGLES,
                                   self.dummy_stream);
    self.initialized = true
    return true
end

function BlingEffect:Update()
    if not self.initialized then
        return
    end
    --TODO: get previous texture instead of input texture??
    local frame_texture = videodecet:GetVideoTexture();
    local patch_image   = self.patch_creator:CreatePatchImage(frame_texture);
    local bling_stream  = self.bling_creator:GetBlingBillboard(patch_image);
    if bling_stream then
        self.rendernode:SetShow(self.showing);
        -- update to render
        self.rendernode:SetParameter(apolloengine.ShaderEntity.FRAME_TEXTURE, frame_texture);
        self.rendernode.render:ChangeVertexBuffer(bling_stream);
        local vert_num = self.bling_creator:GetBlingNum() * 6;
        self.rendernode:SetDrawCount(vert_num);
    else
        self.rendernode:SetShow(false);
    end
end

function BlingEffect:ReleaseResource()
    self.rendernode = nil
    self.dummy_stream = nil
    self.patch_creator = nil
    self.bling_creator = nil
    self.initialized = false
    self.showing = true
end

function BlingEffect:SetShow(show)
    if not self.initialized then
        return
    end
    self.showing = show
end

return BlingEffect;