local mathfunction = require "mathfunction"
local utility = require "blendshape.bsutility"
local model = require "blendshape.model"

local tongue = model:extend();

local ANI_CLIP_TYPE = 
{
  NONE = "none";
  TONGUEOUT = "tongue_out";
  TONGUEIN = "tongue_in";
};

function tongue:new()
  tongue.super.new(self);
  self.is_animated = false;
  self.is_tongue_out = false;

  self.is_transition_start = false;
  self.is_transition_end = false;
  
  self.is_initialized_param = false;
  self.jawopen_weight = 0;
  self.jawopen_weight_limit = 0;
  
  self.mouth_params = {mouthPuckerName, mouthSmileName, mouthLeftName, mouthRightName};
  -- store current mouth weight dict
  self.mouth_weights_dict = {};
  -- the weight dict stores the restired mouth params when playing animation 
  self.mouth_weights_dict_limit = {};
  
  self.cur_play_clip = ANI_CLIP_TYPE.NONE;
  self.is_playing = false;
  
  self.animation_speed_correct = 1.0; -- animation speed correct
  self.adjust_step = 0.1; -- adjust step before and play animation
end

function tongue:CreateModel(tongue_config, attach_to_model, boundingbox, render_before)
  if (tongue_config ~= nil) then
    if (tongue_config.is_animated == 1) then
      local animationConfig = tongue_config.animation;   
      self.model = utility.CreateAnimationModel(animationConfig.mesh, tongue_config.pos, false, false);
    else
      self.model = utility.CreateModel(tongue_config.mesh, tongue_config.render, tongue_config.texture, tongue_config.normal, tongue_config.pbr_amr,  tongue_config.pos, tongue_config.outline);
    end
    
    if (self.model ~= nil) then
      self.is_animated = tongue_config.is_animated == 1;
      self.tongue_trans_left = tongue_config.trans_jaw_left.weight;
      self.tongue_trans_right = tongue_config.trans_jaw_left.weight;
      self.tongue_trans_forward = tongue_config.trans_jaw_forward.weight;
      self.tongue_rot = tongue_config.rot_angle.weight;      
      self.trans_tongue = self.model:GetLocalPosition();
      if (self.is_animated) then
        self.start_tongue_out_limit = tongue_config.start_tongue_out;
        self.start_tongue_in_limit = tongue_config.start_tongue_in;
        if (tongue_config.limit ~= nil) then
          self.jawopen_weight_limit = tongue_config.limit[jawOpenName];
          for i = 1, #self.mouth_params do
            local k = self.mouth_params[i];
            self.mouth_weights_dict_limit[k] = tongue_config.limit[k];
          end
        end
       
        if (tongue_config.correct ~= nil) then         
          if (tongue_config.correct.animation_speed ~= nil) then
            self.animation_speed_correct = tongue_config.correct.animation_speed;
          end 
          if (tongue_config.correct.adjust_step ~= nil) then
            self.adjust_step = tongue_config.correct.adjust_step;
          end
        end
      end
      
      self:AttachToModel(attach_to_model);  
      self:SetBindBox(boundingbox);    
      self:SetRenderQueue(render_before, self.is_animated);
    end
  end
end

function tongue:IsAnimated()
  return self.is_animated;
end

function tongue:IsTongueOut()
  return self.is_tongue_out;
end

function tongue:AdjustWeight(masqueradeResult)
  if (not self.is_animated) then
    return;
  end

  local weights_dict = masqueradeResult.blendshapeWeight;
  if (not self.is_initialized_param) then
    self:CopyMouthWeights(weights_dict, self.mouth_weights_dict, false);
    self.is_initialized_param = true;
  end;
  
  if (self:IsIntransition()) then 
    self:TransitionPlayAnimation(weights_dict);    
    -- update the weights_dict
    self:CopyMouthWeights(self.mouth_weights_dict, weights_dict, false);
    weights_dict[jawOpenName] = self.jawopen_weight;
  else
    if (self.is_tongue_out) then 
      self:CopyMouthWeights(self.mouth_weights_dict, weights_dict, true);
      if (weights_dict[jawOpenName] < self.jawopen_weight_limit) then
        weights_dict[jawOpenName] = self.jawopen_weight_limit;
      end
    else 
      self:CopyMouthWeights(weights_dict, self.mouth_weights_dict, false);
    end
    self.jawopen_weight = weights_dict[jawOpenName];
  end  
  
  -- get the tongue weight
  self.weight = masqueradeResult.tongue;
end

function tongue:CopyMouthWeights(from, to, is_upperbound)
  for i = 1, #self.mouth_params do
    local k = self.mouth_params[i];
    if (not is_upperbound or to[k] > from[k]) then
      to[k] = from[k];
    end
  end
end

-- if the mouth hasn't opened, then force it open
-- return whether need adjust
function tongue:TransitionJawOpenToWeight(from, target, is_lowerbound)
  local need_adjust = true;
  local adjusted = from;
  
  -- no need adjust
  if (is_lowerbound) then 
    if (from > target) then
      need_adjust = false;
    end
  else
    if (math.abs(from - target) < EPSILON) then
      adjusted = target;   
      need_adjust = false;
    end
  end

  if (need_adjust) then
    local step = utility.GetStep(from, target, self.adjust_step);  
    local next_val = from + step;
    
    -- in case of adjust too much
    if(utility.GetStep(target, from, 1) * utility.GetStep(target, next_val, 1) <= 0) then
      adjusted = target;
      need_adjust = false;
    else
      adjusted = next_val;
      need_adjust = true;
    end
  end
  
  self.jawopen_weight = adjusted;
  return need_adjust;  
end

-- adjust the weight to make sure the facial animation is smooth before and after the animation play
function tongue:TransitionMouthParamsToWeighs(from, to, is_upperbound)
  local need_adjust = false;
  local res = true;
  
  local adjusted = {};
  for i = 1, #self.mouth_params do
    local k = self.mouth_params[i];
    local cur = from[k];
    
    local target = to[k];
    
    local more_adjust = true;    
    local step = utility.GetStep(cur, target, self.adjust_step);    
    
    if(is_upperbound) then
      if (cur < target) then
        adjusted[k] = cur;
        more_adjust = false;
      else 
        if(cur - target < EPSILON) then
          adjusted[k] = target;
          more_adjust = false;
        end
      end  
    else
      if (math.abs(cur, target) < EPSILON) then
        adjusted[k] = target;
        more_adjust = false;
      end
    end
    
    if (more_adjust) then
      local next_val = cur + step;
      -- if adjust too much
      if(utility.GetStep(target, cur, 1) * utility.GetStep(target, next_val, 1) <= 0) then
        adjusted[k] = target;
      else
        adjusted[k] = next_val;
        need_adjust = true;
      end  
    end
  end
  
  self.mouth_weights_dict = adjusted;
  return need_adjust;
end

function tongue:TransitionPlayAnimation(weights_dict)
  local adjust_mouthparamw = false;
  local adjust_jawopen = false;
  
  if (self.cur_play_clip == ANI_CLIP_TYPE.TONGUEOUT) then
    adjust_mouthparams = self:TransitionMouthParamsToWeighs(self.mouth_weights_dict, self.mouth_weights_dict_limit, true);     
    adjust_jawopen = self:TransitionJawOpenToWeight(self.jawopen_weight, self.jawopen_weight_limit, true);    
  else
    if (self.cur_play_clip == ANI_CLIP_TYPE.TONGUEIN) then
      adjust_mouthparams = self:TransitionMouthParamsToWeighs(self.mouth_weights_dict, weights_dict, false);              
      adjust_jawopen = self:TransitionJawOpenToWeight(self.jawopen_weight, weights_dict[jawOpenName], false);
    end
  end
    
  local need_adjust = adjust_mouthparams or adjust_jawopen;
  if (not need_adjust) then
    self.is_transition_start = false;
    self.is_transition_end = true;
  end  
end

function tongue:IsIntransition()
  local is_intransition = false;
  if (self.is_transition_start and not self.is_transition_end) then
    is_intransition = true;
  end
  return is_intransition;
end

function tongue:UpdatePosition(weights_dict)
  if (self.model ~= nil) then
    local jawLeftWeight = weights_dict[jawLeftName] or 0;
    local jawRightWeight = weights_dict[jawRightName] or 0;
    local jawForwardWeight = weights_dict[jawForwardName] or 0;
    local jawOpenWeight = weights_dict[jawOpenName] or 0;      

    local quat2 = mathfunction.Mathutility:YawPitchRoll(math.rad(0.0),math.rad(self.tongue_rot*jawOpenWeight),math.rad(0.0));   

    local dx = 0;
    local dz = 0;
    if (jawLeftWeight ~= nil) then
      dx = -self.tongue_trans_left*jawLeftWeight;
    end
    if (jawRightWeight~= nil) then
      dx = dx + self.tongue_trans_right*jawRightWeight;
    end
    if (jawForwardWeight ~= nil) then
      dz = self.tongue_trans_forward*jawForwardWeight;
    end
    local trans = mathfunction.vector3(dx, 0.0, dz);
    self.model:SetLocalPosition(self.trans_tongue + trans);  
    
    local quat2 = mathfunction.Mathutility:YawPitchRoll(math.rad(0.0),math.rad(self.tongue_rot*jawOpenWeight),math.rad(0.0));
    self.model:SetLocalRotation(quat2);
  end
end

function tongue:CanPlayTongueOut()
  -- not play yet and the tongue is inside
  local res = true;
  if (not self.is_tongue_out) then
    if (self.jawopen_weight < self.jawopen_weight_limit) then
      res = false;
    end
  end  
  return res;
end

function tongue:PlayClip(clip)
  if (not self.is_playing) then
    local frameCount = self.model:GetFrameCount();
    local halfFrameCount = 0.5 * frameCount;
  
    if (clip == ANI_CLIP_TYPE.TONGUEOUT) then
      self.model:SetAnimationInterval(0, halfFrameCount);
      self.model:Reset(0);
    else 
      if (clip == ANI_CLIP_TYPE.TONGUEIN) then
        self.model:SetAnimationInterval(halfFrameCount, frameCount);   
      end
    end
    
    self.cur_play_clip = clip;
    self.is_playing = true;
    self.model:Play();
  end
end

function tongue:UpdateModel(def)
  if (not self:IsAnimated() or self:IsIntransition()) then
    return;
  end
  
  local currentWeight = self.weight or 0;
  
  --LOG("currentWeight"..currentWeight);
  
  local def_correct = self.animation_speed_correct * def;
  local animation_status = self.model:UpdateAnimation(def_correct);

  -- last animation finished
  if (self.is_playing and animation_status == 2) then
    self.is_playing = false;
    
    -- played the tongue in animation just now 
    if (self.cur_play_clip == ANI_CLIP_TYPE.TONGUEIN) then
      self.is_tongue_out = false;
      self.is_transition_start = true;
      self.is_transition_end = false;
      return;
    end       
  end
  
  -- animation not started yet
  if (animation_status ~= 1) then
    -- play tongue out animation
    if (self.is_transition_end and not self.is_tongue_out and self.cur_play_clip == ANI_CLIP_TYPE.TONGUEOUT) then
      self.is_tongue_out = true;
      self:PlayClip(ANI_CLIP_TYPE.TONGUEOUT);
      return;
    end  
    
    if (currentWeight > self.start_tongue_out_limit and not self.is_tongue_out) then 
      --if (self:CanPlayTongueOut()) then      
      self.is_transition_start = true;
      self.is_transition_end = false;
      self.cur_play_clip = ANI_CLIP_TYPE.TONGUEOUT;
    else
      if (currentWeight < self.start_tongue_in_limit and self.is_tongue_out) then
        -- play tongue in animation        
        self:PlayClip(ANI_CLIP_TYPE.TONGUEIN);
      end
    end
  end  
end
  
return tongue;