require "utility"
local ae = require "apolloengine"
local vc = require "venuscore"
local mf = require "mathfunction"
local vj = require "venusjson"
local Object = require "classic"

local Rect = Object:extend()

function Rect:new()
    self.x1 = 0
    self.y1 = 0
    self.x2 = 0
    self.y2 = 0
end

function Rect:AddPoint(x, y)
    self.x1 = math.min(self.x1, x)
    self.y1 = math.min(self.y1, y)
    self.x2 = math.max(self.x2, x)
    self.y2 = math.max(self.y2, y)
end

function Rect:AddRect(rect)
    self.x1 = math.min(self.x1, rect.x1)
    self.y1 = math.min(self.y1, rect.y1)
    self.x2 = math.max(self.x2, rect.x2)
    self.y2 = math.max(self.y2, rect.y2)
end

function Rect:OffsetBy(x, y)
    self.x1 = self.x1 + x
    self.y1 = self.y1 + y
    self.x2 = self.x2 + x
    self.y2 = self.y2 + y
end

local obj = {}


function obj:Initialize(host, size)

    self.host = host
    self.targetSize = { size:x(), size:y() }
    self.order = 100
    self.refresh = false

    self.GLYPH_TRANSFORM = ae.IMaterialSystem:NewParameterSlot(
            ae.ShaderEntity.UNIFORM,
            "GLYPH_TRANSFORM");

    self.identityRenderer = host:CreateRenderObject(
            "comm:documents/material/identity.material")
    self.glyphRenderer = host:CreateRenderObject(
            "comm:documents/material/glyph.material",
            ae.VertexBufferEntity.MU_DYNAMIC,
            true)

    local m1 = mf.Matrix44:CreateScaleMatrix(mf.vector3(2.0, -2.0, 1.0))
    local m2 = mf.Matrix44:CreateTranslateMatrix(mf.vector3(-1, 1, 0))
    local m = m1 * m2

    self.glyphRenderer:SetParameter(self.GLYPH_TRANSFORM, m)

    self:_InitializeParameters()

    self:_UpdateParameters()

    return self.order
end

function obj:Resizeview(size)
    self.targetSize = { size:x(), size:y() }
    self.refresh = true
    self:_UpdateParameters()
end

function obj:Process(pipeline, Original, Scene, Output)

    self:_UpdateParameters()

    Output:PushRenderTarget()
    Output:ClearBuffer(ae.RenderTargetEntity.CF_COLOR)

    self.identityRenderer:SetParameter(
            ae.ShaderEntity.TEXTURE_DIFFUSE,
            Scene:GetAttachment(ae.RenderTargetEntity.TA_COLOR_0))

    self.identityRenderer:Draw(pipeline)

    self.glyphRenderer:Draw(pipeline)

end

local function _CreateEmptyTexture()
    local tex = ae.TextureEntity()
    tex:PushMetadata(ae.TextureBufferMetadata(mf.vector2(4, 4)))
    tex:SetJobType(vc.IJob.JT_SYNCHRONOUS)
    tex:CreateResource()
    return tex
end

local function _CreateDefaultMapping()
    local json = {
        row = 1,
        col = 1,
        map = " "
    }
    return vj.StoreJsonString(json)
end

local function _CreateDefaultSetting()
    local json = {
        {
            size = {
                "1.0",
                "1.0"
            },
            spacing = {
                "1.0",
                "0"
            },
            anchor = {
                "0",
                "0"
            },
            offset = {
                "0",
                "0"
            }
        }
    }
    return vj.StoreJsonString(json)
end

local function split(text, s)
    local res = {}
    for k in string.gmatch(text, "[^" .. s .. "]+") do
        table.insert(res, k)
    end
    return res
end

function obj:_InitializeParameters()
    self.glyphTexture = _CreateEmptyTexture()
    self.lastGlyphTexture = nil

    self.glyphMapping = _CreateDefaultMapping()
    self.lastGlyphMapping = nil

    self.glyphSetting = _CreateDefaultSetting()
    self.lastGlyphSetting = nil

    self.glyphText = " "
    self.lastGlyphText = nil

    self.map = {}
end

local function DeepCopy(v)
    local t = type(v)
    if t == "table" then
        local c = {}
        for k, d in pairs(v) do
            c[k] = DeepCopy(d)
        end
        return c
    else
        return v
    end
end

function obj:_UpdateTexture()
    if self.lastGlyphTexture ~= self.glyphTexture then
        self.glyphRenderer:SetParameter(
                ae.ShaderEntity.TEXTURE_DIFFUSE,
                self.glyphTexture)
        self.lastGlyphTexture = self.glyphTexture
    end
end

function obj:_UpdateMapping()
    if self.lastGlyphMapping ~= self.glyphMapping then
        local config = vj.LoadJsonString(self.glyphMapping)
        local row = config.row
        local col = config.col
        local map = config.map
        local len = math.min(string.len(map), row * col)
        self.map = {}
        for i = 1, len do
            local s = string.sub(map, i, i)
            local r = math.floor((i - 1) / col)
            local c = math.floor((i - 1) % col)
            self.map[s] = { r, c }
        end
        self.config = DeepCopy(config)
        self.lastGlyphMapping = self.glyphMapping
        self.refresh = true
    end
end

function obj:_UpdateSetting()
    if self.lastGlyphSetting ~= self.glyphSetting then
        local content = vj.LoadJsonString(self.glyphSetting)
        self.content = DeepCopy(content)
        self.lastGlyphSetting = self.glyphSetting
        self.refresh = true
    end
end

function obj:_UpdateText()
    if self.lastGlyphText ~= self.glyphText then
        self.text = split(self.glyphText, "\n")
        self.lastGlyphText = self.glyphText
        self.refresh = true
    end
end

function obj:_RefreshContent()
    self.vertices = ae.VertexStream()
    self.indices = ae.IndicesStream()
    self.vertices:SetVertexType(
            ae.ShaderEntity.ATTRIBUTE_POSITION,
            ae.VertexBufferEntity.DT_FLOAT,
            ae.VertexBufferEntity.DT_HALF_FLOAT,
            4);
    self.vertices:SetVertexType(
            ae.ShaderEntity.ATTRIBUTE_COORDNATE0,
            ae.VertexBufferEntity.DT_FLOAT,
            ae.VertexBufferEntity.DT_HALF_FLOAT,
            2);
    self.indices:SetIndicesType(
            ae.IndicesBufferEntity.IT_UINT16);

    local len = math.min(#self.content, #self.text)
    for i = 1, len do
        local contentItem = self.content[i]
        local textItem = self.text[i]
        local parsed = self:_ParseContentItem(contentItem, textItem)
        self:_GenerateVertices(parsed)
    end

    self.glyphRenderer:ChangeVertices(self.vertices)
    self.glyphRenderer:ChangeIndices(self.indices)
end

local function _GetEvalClosure(v)
    local script = string.format("return function(width, height, pixel) return %s end", tostring(v))
    local f, e = load(script)
    if f == nil then
        return nil
    end
    return f()
end

local function _ParseCoordinateX(targetSize, value)
    local closure = _GetEvalClosure(value)
    if closure == nil then
        return nil
    end
    local width = 1
    local height = targetSize[2] / targetSize[1]
    local pixel = 1.0 / targetSize[1]
    local result = closure(width, height, pixel)
    return result
end

local function _ParseCoordinateY(targetSize, value)
    local closure = _GetEvalClosure(value)
    if closure == nil then
        return nil
    end
    local width = targetSize[1] / targetSize[2]
    local height = 1
    local pixel = 1.0 / targetSize[2]
    local result = closure(width, height, pixel)
    return result
end

function obj:_ParseContentItem(contentItem, textItem)
    local contentRect = Rect()
    local charMappings = {}
    local charRects = {}
    local textLen = string.len(textItem)
    local charSizeX = _ParseCoordinateX(self.targetSize, contentItem.size[1])
    local charSizeY = _ParseCoordinateY(self.targetSize, contentItem.size[2])
    local charSpacingX = _ParseCoordinateX(self.targetSize, contentItem.spacing[1])
    local charSpacingY = _ParseCoordinateY(self.targetSize, contentItem.spacing[2])
    local anchorX = _ParseCoordinateX(self.targetSize, contentItem.anchor[1])
    local anchorY = _ParseCoordinateY(self.targetSize, contentItem.anchor[2])
    local offsetX = _ParseCoordinateX(self.targetSize, contentItem.offset[1])
    local offsetY = _ParseCoordinateY(self.targetSize, contentItem.offset[2])
    local lineX = 0
    local lineY = 0
    local x = 0
    local y = 0
    for i = 1, textLen do
        local char = string.sub(textItem, i, i)
        if self.map[char] ~= nil then
            table.insert(charMappings, self.map[char])
            local charRect = Rect()
            charRect:AddPoint(charSizeX, charSizeY)
            charRect:OffsetBy(x, y)
            contentRect:AddRect(charRect)
            table.insert(charRects, charRect)
            x = x + charSpacingX
            y = y + charSpacingY
        end
    end

    local srcOffsetX = (contentRect.x2 - contentRect.x1) * anchorX - contentRect.x1
    local srcOffsetY = (contentRect.y2 - contentRect.y1) * anchorY - contentRect.y1

    offsetX = offsetX - srcOffsetX
    offsetY = offsetY - srcOffsetY

    local rectLen = #charRects

    for i = 1, rectLen do
        local charRect = charRects[i]
        charRect:OffsetBy(offsetX, offsetY)
    end
    local parsed = {
        mappings = charMappings,
        rects = charRects,
    }
    return parsed
end

--in shape opposite 'N'
local function _AppendRectPositionCoord(vertices, rect)
    local attr = ae.ShaderEntity.ATTRIBUTE_POSITION
    local v1 = mf.vector4(rect.x1, rect.y1, 0, 1)
    local v2 = mf.vector4(rect.x1, rect.y2, 0, 1)
    local v3 = mf.vector4(rect.x2, rect.y1, 0, 1)
    local v4 = mf.vector4(rect.x2, rect.y2, 0, 1)
    --LOG("Rect " .. string.format("%f %f %f %f", rect.x1, rect.y1, rect.x2, rect.y2))
    --LOG("PustVertex " .. string.format("%f %f", v1:x(), v1:y()))
    --LOG("PustVertex " .. string.format("%f %f", v2:x(), v2:y()))
    --LOG("PustVertex " .. string.format("%f %f", v3:x(), v3:y()))
    --LOG("PustVertex " .. string.format("%f %f", v4:x(), v4:y()))
    vertices:PushVertexData(attr, v1)
    vertices:PushVertexData(attr, v2)
    vertices:PushVertexData(attr, v3)
    vertices:PushVertexData(attr, v4)
end

--in shape opposite 'N'
local function _AppendRectTextureCoord(vertices, mapping, wh)
    local attr = ae.ShaderEntity.ATTRIBUTE_COORDNATE0
    local r = mapping[1]
    local c = mapping[2]
    local v1 = mf.vector2(c, r) * wh
    local v2 = mf.vector2(c, r+1) * wh
    local v3 = mf.vector2(c+1, r) * wh
    local v4 = mf.vector2(c+1, r+1) * wh
    --LOG("PustTex " .. string.format("%f %f", v1:x(), v1:y()))
    --LOG("PustTex " .. string.format("%f %f", v2:x(), v2:y()))
    --LOG("PustTex " .. string.format("%f %f", v3:x(), v3:y()))
    --LOG("PustTex " .. string.format("%f %f", v4:x(), v4:y()))
    vertices:PushVertexData(attr, v1)
    vertices:PushVertexData(attr, v2)
    vertices:PushVertexData(attr, v3)
    vertices:PushVertexData(attr, v4)
end

--in shape opposite 'N'
local function _AppendRectVertexIndex(indices, start)
    indices:PushIndicesData(start + 0)
    indices:PushIndicesData(start + 1)
    indices:PushIndicesData(start + 2)
    indices:PushIndicesData(start + 2)
    indices:PushIndicesData(start + 1)
    indices:PushIndicesData(start + 3)
end

function obj:_GenerateVertices(parsed)
    local wh = mf.vector2(1.0 / self.config.col, 1.0 / self.config.row)
    local rectLen = #parsed.rects
    --LOG("Begin Push")
    for i = 1, rectLen do
        _AppendRectPositionCoord(self.vertices, parsed.rects[i])
        _AppendRectTextureCoord(self.vertices, parsed.mappings[i], wh)
        _AppendRectVertexIndex(self.indices, (i-1) * 4)
    end
    --LOG("End Push")
end

function obj:_UpdateParameters()

    self:_UpdateTexture()
    self:_UpdateMapping()
    self:_UpdateSetting()
    self:_UpdateText()

    if self.refresh then
        self.refresh = false
        self:_RefreshContent()
    end
end

return obj