#extension GL_OES_EGL_image_external : require
precision highp float;

/** The varying interpolating for the texture coordinates. */
varying vec2 texCoordinate;
/** The varying interpolating for the mask position coordinates. */
varying vec2 maskCoordinate;
/** The standard 2D sampler for the main texture. */
uniform sampler2D imageSampler;
/** The sampler for the main texture if the texture is an external OES texture. */
uniform samplerExternalOES imageSamplerOes;
/** Flag indicating whether the main texture is an external OES texture. */
uniform bool imageIsOES;
/** The standard 2D sampler for the LUT texture. */
uniform sampler2D lutSampler;
/** The pixel size of the input image. */
uniform vec2 imageSize;
/** The time variable used by time-dependent effects. */
uniform float time;

/** The intensity of the sharpening filter to be applied, in the [0-1] range. */
uniform float sharpen;
/** Intensity with which the lookup table is applied. */
uniform float lutIntensity;
/** The contrast multiplier to be applied. Non-negative value, 1 is the neutral value. */
uniform float contrast;
/**
 * The number of stops of exposure to apply. This uniform doesn't have a limited range, 0 is the
 * neutral value.
 */
uniform float exposure;
/**
 * The amount of saturation to be applied to the image. Non-negative value. 1 is the neutral value.
 */
uniform float saturation;
/** The color multiplier to be applied to apply the temperature regulation. */
uniform vec3 temperatureColor;
/**
 * Flag indicating whether the temperature regulation as defined by [temperatureColor] should be
 * applied or not.
 */
uniform bool useTemperature;
/** The normalized hue angle shift to be applied. A value of 1 means 360 degrees angle. */
uniform float hueShiftAngle;
/** The target key color used to apply the chroma key effect. */
uniform vec3 chromaColor;
/**
 * The threshold for the hue distance used to apply the chroma key effect. Must be in the [0,1]
 * range.
 */
uniform float chromaHueThreshold;
/**
 * The threshold for the s and v distances used to apply the chroma key effect. Must be in the [0,1]
 * range.
 */
uniform float chromaShadowThreshold;
/** Flag indicating whether the chroma key filter should be applied or not. */
uniform bool useChromaKey;
/** The rectangular mask radius used to generate the mask */
uniform float maskRectCornerRadius;
/** Flag indicating whether the applied mask should be inverted or not. */
uniform bool maskInvert;
/** Smooth factor applied to soften the edges of the applied mask. Must be non-negavite. */
uniform float maskSmoothness;
/** Multiplier to be applied on each axis to the distance used to generate the mask. */
uniform vec2 maskDistanceMultiplier;
/**
 * Defines which effect should be applied among the choices defined in E_* or 0 if no effect should
 * be applied.
 */
uniform int effect;
/**
 * Defines which mask effect should be applied among the choices defined in M_* or 0 if no mask
 * should be applied.
 */
uniform int mask;
/** Defines an value multiplier to be applied to the color alpha channel. */
uniform float opacity;

/** Returns the main texture color at the coordinates [uv]. */
vec4 imageColor(vec2 uv) {
    return imageIsOES ? texture2D(imageSamplerOes, uv) : texture2D(imageSampler, uv);
}

/** Returns the LUT texture color at the coordinates [uv]. */
vec4 lutColor(vec2 uv) {
    return texture2D(lutSampler, uv);
}

// Utility functions section

/** Returns a random value for the given [seed] in the [0-1) range. */
float rand(float seed) {
    // Magic number from https://www.shadertoy.com/view/MtXBDs
    return fract(sin(seed) * 43758.5453);
}

/**
 * Returns a random value for a vec2 seed in the [0-1) range.
 *
 * @param seed The seed used to generate the random value.
 */
float rand(vec2 seed) {
    // Magic numbers from https://www.shadertoy.com/view/MtXBDs
    return rand(dot(seed, vec2(12.9898, 4.1414)));
}

/**
 * Returns a random value for a vec2 seed in the [min-max) range.
 *
 * @param seed The seed used to generate the random value.
 * @param min The lower bound for the output value.
 * @param max The high bound for the output value.
 */
float randInRange(vec2 seed, float min, float max) {
    return min + rand(seed) * (max - min);
}

/** Returns a noise value for the given [seed] in the [0-1) range. */
float noise(float seed) {
    float fl = floor(seed);
    float fc = fract(seed);
    return mix(rand(fl), rand(fl + 1.0), fc);
}

/**
 * Returns a random RGB noise (values in range [0, 1)) for a given coordinate.
 *
 * @param uv The coordinate for which the random value is returned.
 * @param seed The seed used to generate the random value.
 */
vec3 rgbNoise(vec2 uv, float seed) {
    vec2 p = uv + seed + 0.1742;
    return vec3(rand(p), rand(p + 0.3333), rand(p + 0.6666));
}

/** Returns 1 if [bottom] <= [value] < [top], 0 otherwise. */
float isInRange(float v, float bottom, float top) {
    return step(bottom, v) - step(top, v);
}

/** Returns the HSV value for the given RGB [color]. */
vec3 rgb2hsv(vec3 color) {
    vec3 c = color;
    vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
    vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
    vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));

    float d = q.x - min(q.w, q.y);
    float e = 1.0e-10;
    return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}

/** Returns the RGB color for the given HSV [value]. */
vec3 hsv2rgb(vec3 value) {
    vec3 c = value;
    vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
    vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
    return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

// Kernel utils section

/**
 * Returns the color corresponding to the application of a pixel-wise 3 by 3 kernel on the main
 * texture, at the given [uv] coordinate.
 *
 * @param uv The coordinate indicating the texture position on which the kernel is centered.
 * @param kernel3 The kernel matrix to apply.
 */
vec4 withKernel3(vec2 uv, mat3 kernel3) {
    vec2 pixelOffset = 1. / imageSize;
    float offX = pixelOffset.x;
    float offY = pixelOffset.y;
    return
        imageColor(uv + vec2(-offX, offY)) * kernel3[0][0] +
        imageColor(uv + vec2(0.0, offY)) * kernel3[0][1] +
        imageColor(uv + vec2(offX, offY)) * kernel3[0][2] +
        imageColor(uv + vec2(-offX, 0.0)) * kernel3[1][0] +
        imageColor(uv + vec2(0.0, 0.0)) * kernel3[1][1] +
        imageColor(uv + vec2(offX, 0.0)) * kernel3[1][2] +
        imageColor(uv + vec2(-offX, -offY)) * kernel3[2][0] +
        imageColor(uv + vec2(0.0, -offY)) * kernel3[2][1] +
        imageColor(uv + vec2(offX, -offY)) * kernel3[2][2];
}

/**
 * EFFECTS.
 * Ported from iOS.
 */

// Glitch01 section

/**
 * Returns the color of the main image at the [uv] coordinates, with the Glitch01 filter applied.
 */
vec4 glitch01(vec2 uv) {
    vec2 coordinate = uv * imageSize;
    float blockSize = coordinate.x / uv.x / 8.0;
    // assign this coordinate to a specific block made of 16 pixels
    vec2 block = floor(coordinate.xy / vec2(blockSize));
    vec2 uv_noise = block / vec2(blockSize);
    uv_noise += floor(vec2(time) * vec2(1234.0, 3543.0)) / vec2(blockSize);
    uv_noise = fract(uv_noise);

    // increase the last number to have more blocks
    float blocE_thresh = pow(fract(time * 1236.0453), 2.0) * 0.1;
    // increase the last number to have more lines
    float line_thresh = pow(fract(time * 2236.0453), 3.0) * 0.3;
    vec2 uv_r = uv, uv_g = uv, uv_b = uv;

    vec3 rgbNoiseAtUv = rgbNoise(uv_noise, time);
    vec3 rgbNoiseAtY = rgbNoise(vec2(uv_noise.y, 0.0), time);

    // glitch some blocks and lines
    if (rgbNoiseAtUv.r < blocE_thresh || rgbNoiseAtY.g < line_thresh) {
        vec2 dist = (uv_noise - 0.5) * 0.3;
        uv_r += dist * 0.1;
        uv_g += dist * 0.2;
        uv_b += dist * 0.125;
    }

    vec4 color = vec4(1);
    color.r = imageColor(uv_r).r;
    color.g = imageColor(uv_g).g;
    color.b = imageColor(uv_b).b;

    // loose luma for some blocks
    if (rgbNoiseAtUv.g < blocE_thresh)
        color.rgb = color.ggg;

    // discolor block lines
    if (rgbNoiseAtY.b * 3.5 < line_thresh)
        color.rgb = vec3(0.0, dot(color.rgb, vec3(1.0)), 0.0);

    // interleave lines in some blocks
    if (rgbNoiseAtUv.g * 1.5 < blocE_thresh || rgbNoiseAtY.g * 2.5 < line_thresh) {
        float line = fract(coordinate.y / 12.0);
        vec3 mask = vec3(3.0, 0.0, 0.0);
        if (line > 0.333)
            mask = vec3(0.0, 3.0, 0.0);
        if (line > 0.666)
            mask = vec3(0.0, 0.0, 3.0);

        color.xyz *= mask;
        color = vec4(min(color.x, 1.0), min(color.y, 1.0), min(color.z, 1.0), 1.0);
    }

    return color;
}

// Glitch02 section
float blockyNoise(vec2 uv, float threshold, float scale, float seed, float t) {
    float scroll = floor(t + sin(11.0 * t) + sin(t)) * 0.77;
    vec2 noiseUV = fract(uv.yy / scale + scroll);
    float noise2 = imageColor(noiseUV).r;

    float id = floor(noise2 * 20.0);
    id = noise(id + seed) - 0.5;

    if (abs(id) > threshold) {
        id = 0.0;
    }

    return id;
}

/**
 * Returns the color of the main image at the [uv] coordinates, with the Glitch02 filter applied.
 */
vec4 glitch02(vec2 uv) {
    float rgbIntensity = 0.1 + 0.1 * sin(time * 3.7);
    float displaceIntensity = 0.2 + 0.3 * pow(sin(time * 1.2), 5.0);
    float interlaceIntensity = 0.01;
    float dropoutIntensity = 0.1;
    vec2 coordinate = uv * imageSize;

    float displace = blockyNoise(uv + vec2(uv.y, 0.0), displaceIntensity, 25.0, 66.6, time);
    displace *= blockyNoise(uv.yx + vec2(0.0, uv.x), displaceIntensity, 111.0, 13.7, time);
    uv.x += displace;
    vec2 offs = 0.1 * vec2(blockyNoise(uv.xy + vec2(uv.y, 0.0), rgbIntensity, 65.0, 341.0, time), 0.0);

    float colr = imageColor(uv - offs).r;
    float colg = imageColor(uv).g;
    float colb = imageColor(uv + offs).b;

    float line = fract(coordinate.y / 3.0);
    vec3 mask = vec3(3.0, 0.0, 0.0);
    if (line > 0.333)
        mask = vec3(0.0, 3.0, 0.0);
    if (line > 0.666)
        mask = vec3(0.0, 0.0, 3.0);

    mask = vec3(min(mask.x, 1.0), min(mask.y, 1.0), min(mask.z, 1.0));

    float maskNoise = blockyNoise(uv, interlaceIntensity, 90.0, time, time) * max(displace, offs.x);
    maskNoise = 1.0 - maskNoise;
    if (maskNoise == 1.0) mask = vec3(1.0);

    float dropout = blockyNoise(uv, dropoutIntensity, 11.0, time, time) * blockyNoise(uv.yx, dropoutIntensity, 90.0, time, time);
    mask *= (1.0 - 5.0 * dropout);

    vec3 color = vec3(colr, colg, colb);
    return vec4(mask * color, 1.0);
}

// Glitch03 section

/**
 * Returns the color of the main image at the [uv] coordinates, with the Glitch03 filter applied.
 */
vec4 glitch03(vec2 uv) {
    const float AMT = 0.2; // 0 - 1 glitch amount
    float SPEED = 0.1; // 0 - 1 speed

    float g_time = floor(time * SPEED * 60.0);

    //copy orig
    vec3 outCol = imageColor(uv).rgb;

    //randomly offset slices horizontally
    float maxOffset = AMT / 2.0;
    for (float i = 0.0; i < 10.0 * AMT; i += 1.0) {
        float sliceY = rand(vec2(g_time, 2345.0 + float(i)));
        float sliceH = rand(vec2(g_time, 9035.0 + float(i))) * 0.25;
        float hOffset = randInRange(vec2(g_time, 9625.0 + float(i)), -maxOffset, maxOffset);
        vec2 uvOff = uv;
        uvOff.x += hOffset;
        if (isInRange(uv.y, sliceY, fract(sliceY + sliceH)) == 1.0) {
            outCol = imageColor(uvOff).rgb;
        }
    }

    //do slight offset on one entire channel
    float maxColOffset = AMT / 6.0;
    float rnd = rand(vec2(g_time, 9545.0));
    vec2 colOffset = vec2(randInRange(vec2(g_time, 9545.0), -maxColOffset, maxColOffset),
        randInRange(vec2(g_time, 7205.0), -maxColOffset, maxColOffset));
    if (rnd < 0.33) {
        outCol.r = imageColor(uv + colOffset).r;
    } else if (rnd < 0.66) {
        outCol.g = imageColor(uv + colOffset).g;
    } else {
        outCol.b = imageColor(uv + colOffset).b;
    }

    return vec4(outCol, 1.0);
}

// Vintage01 section
vec2 curve(vec2 uv) {
    uv = (uv - 0.5) * 2.0;
    uv *= 1.1;
    uv.x *= 1.0 + pow((abs(uv.y) / 5.0), 2.0);
    uv.y *= 1.0 + pow((abs(uv.x) / 4.0), 2.0);
    uv = (uv / 2.0) + 0.5;
    uv = uv * 0.92 + 0.04;
    return uv;
}

/**
 * Returns the color of the main image at the [uv] coordinates, with the Vintage01 filter applied.
 */
vec4 vintage01(vec2 uv) {
    vec2 coordinate = uv * imageSize;
    vec2 cuv = curve(uv);

    vec3 col;
    float x = sin(0.3 * time + cuv.y * 21.0) * sin(0.7 * time + cuv.y * 29.0) * sin(0.3 + 0.33 * time + cuv.y * 31.0) * 0.0017;

    col.r = imageColor(vec2(x + cuv.x + 0.001, cuv.y + 0.001)).x + 0.05;
    col.g = imageColor(vec2(x + cuv.x + 0.000, cuv.y - 0.002)).y + 0.05;
    col.b = imageColor(vec2(x + cuv.x - 0.002, cuv.y + 0.000)).z + 0.05;

    col.r += 0.08 * imageColor(0.75 * vec2(x + 0.025, -0.027) + vec2(cuv.x + 0.001, cuv.y + 0.001)).x;
    col.g += 0.05 * imageColor(0.75 * vec2(x + -0.022, -0.02) + vec2(cuv.x + 0.000, cuv.y - 0.002)).y;
    col.b += 0.08 * imageColor(0.75 * vec2(x + -0.02, -0.018) + vec2(cuv.x - 0.002, cuv.y + 0.000)).z;

    col = clamp(col * 0.6 + 0.4 * col * col * 1.0, 0.0, 1.0);

    float vig = (0.0 + 1.0 * 16.0 * cuv.x * cuv.y * (1.0 - cuv.x) * (1.0 - cuv.y));
    col *= vec3(pow(vig, 0.3));

    col *= vec3(0.95, 1.05, 0.95);
    col *= 2.8;
    float height = coordinate.y / uv.y;
    float scans = clamp(0.35 + 0.35 * sin(3.5 * time + cuv.y * height * 1.5), 0.0, 1.0);

    float s = pow(scans, 1.7);
    col = col * vec3(0.4 + 0.7 * s);

    col *= 1.0 + 0.01 * sin(110.0 * time);
    if (cuv.x < 0.0 || cuv.x > 1.0)
        col *= 0.0;
    if (cuv.y < 0.0 || cuv.y > 1.0)
        col *= 0.0;

    col *= 1.0 - 0.65 * vec3(clamp((mod(coordinate.x, 2.0) - 1.0) * 2.0, 0.0, 1.0));

    // float comp = smoothstep(0.1, 0.9, sin(time));
    // Remove the next line to stop cross-fade between original and postprocess
    //  col = mix( col, oricol, comp );
    return vec4(col, 1.0);
}

// Vintage generic section
const float vertJerkOpt = 1.0;
const float vertMovementOpt = 1.0;
const float vintageMod = 289.;

vec3 permute(vec3 x) {
    return mod(((x * 34.0) + 1.0) * x, vintageMod);
}

float snoise(vec2 v) {
    const vec4 C = vec4(0.211324865405187, // (3.0-sqrt(3.0))/6.0
        0.366025403784439, // 0.5*(sqrt(3.0)-1.0)
        - 0.577350269189626, // -1.0 + 2.0 * C.x
        0.024390243902439); // 1.0 / 41.0
    // First corner
    vec2 i = floor(v + dot(v, C.yy));
    vec2 x0 = v - i + dot(i, C.xx);

    // Other corners
    vec2 i1;
    //i1.x = step( x0.y, x0.x ); // x0.x > x0.y ? 1.0 : 0.0
    //i1.y = 1.0 - i1.x;
    i1 = (x0.x > x0.y) ? vec2(1.0, 0.0) : vec2(0.0, 1.0);
    // x0 = x0 - 0.0 + 0.0 * C.xx ;
    // x1 = x0 - i1 + 1.0 * C.xx ;
    // x2 = x0 - 1.0 + 2.0 * C.xx ;
    vec4 x12 = x0.xyxy + C.xxzz;
    x12.xy -= i1;

    // Permutations
    i = mod(i, vintageMod); // Avoid truncation effects in permutation
    vec3 p = permute(permute(i.y + vec3(0.0, i1.y, 1.0))
        + i.x + vec3(0.0, i1.x, 1.0));

    vec3 m = max(0.5 - vec3(dot(x0, x0), dot(x12.xy, x12.xy), dot(x12.zw, x12.zw)), 0.0);
    m = m * m;
    m = m * m;

    // Gradients: 41 points uniformly over a line, mapped onto a diamond.
    // The ring size 17*17 = 289 is close to a multiple of 41 (41*7 = 287)

    vec3 x = 2.0 * fract(p * C.www) - 1.0;
    vec3 h = abs(x) - 0.5;
    vec3 ox = floor(x + 0.5);
    vec3 a0 = x - ox;

    // Normalise gradients implicitly by scaling m
    // Approximation of: m *= inversesqrt( a0*a0 + h*h );
    m *= 1.79284291400159 - 0.85373472095314 * (a0 * a0 + h * h);

    // Compute final noise value at P
    vec3 g;
    g.x = a0.x * x0.x + h.x * x0.y;
    g.yz = a0.yz * x12.xz + h.yz * x12.yw;
    return 130.0 * dot(m, g);
}

float fnoise(vec2 seed) {
    return rand(seed) * 0.55;
}

float staticV(vec2 uv, float t) {
    float staticHeight = fnoise(vec2(9.0, t * 1.2 + 3.0)) * 0.3 + 5.0;
    float staticAmount = fnoise(vec2(1.0, t * 1.2 - 6.0)) * 0.1 + 0.3;
    float staticStrength = fnoise(vec2(-9.75, t * 0.6 - 3.0)) * 2.0 + 2.0;
    return (1.0 - step(fnoise(vec2(5.0 * pow(t, 2.0) + pow(uv.x * 7.0, 1.2), pow((mod(t, 100.0) + 100.0) * (1.0 - uv.y) * 0.3 + 3.0, staticHeight))), staticAmount)) * staticStrength;
}

/**
 * Returns the color of the main image at the [uv] coordinates, with the Vintage generic filter
 * applied.
 */
vec4 vintageGeneric(
    vec2 uv,
    float bottomStaticOpt,
    float scalinesOpt,
    float scalinesFreq,
    float rgbOffsetOpt,
    float horzFuzzOpt
) {
    //float jerkOffset = (1.0 - step(snoise(float2(time * 1.3, 5.0)), 0.8)) * 0.05;

    float fuzzOffset = fnoise(vec2(time * 15.0, uv.y * 80.0)) * 0.003;
    float largeFuzzOffset = fnoise(vec2(time * 1.0, uv.y * 25.0)) * 0.004;

    float yOffset = 0.0;
    // given that the snoise function is expensive, we are using it only if really needed
    if (vertMovementOpt != 0.0) {
        float vertMovementOn = (1.0 - step(snoise(vec2(time * 0.2, 8.0)), 0.2)) * vertMovementOpt;
        float vertJerk = (1.0 - step(fnoise(vec2(time * 1.5, 5.0)), 0.6)) * vertJerkOpt;
        float vertJerk2 = (1.0 - step(fnoise(vec2(time * 5.5, 5.0)), 0.2)) * vertJerkOpt;
        yOffset = abs(sin(time) * 4.0) * vertMovementOn + vertJerk * vertJerk2 * 0.3;
    }

    float y = mod(uv.y + yOffset, 1.0);

    float xOffset = (fuzzOffset + largeFuzzOffset) * horzFuzzOpt;

    float staticVal = 0.0;

    for (float y = -1.0; y <= 1.0; y += 1.0) {
        float maxDist = 5.0 / 200.0;
        float dist = y / 200.0;
        staticVal += staticV(vec2(uv.x, 1.0 - uv.y + dist), time) * (maxDist - abs(dist)) * 1.5;
    }

    staticVal *= bottomStaticOpt;

    float red = imageColor(vec2(uv.x + xOffset - 0.01 * rgbOffsetOpt, y)).r + staticVal;
    float green = imageColor(vec2(uv.x + xOffset, y)).g + staticVal;
    float blue = imageColor(vec2(uv.x + xOffset + 0.01 * rgbOffsetOpt, y)).b + staticVal;

    vec3 color = vec3(red, green, blue);
    float scanline = sin(uv.y * scalinesFreq) * 0.04 * scalinesOpt;
    color -= scanline;

    return vec4(color, 1.0);
}

/**
 * Returns the color of the main image at the [uv] coordinates, with the Vintage02 filter applied.
 */
vec4 vintage02(vec2 uv) {
    return vintageGeneric(uv, 1., 2., 1500., 1., 4.);
}

/**
 * Returns the color of the main image at the [uv] coordinates, with the Vintage03 filter applied.
 */
vec4 vintage03(vec2 uv) {
    return vintageGeneric(uv, 0.5, 4., 2000., 0.5, 3.);
}

// Chroma 01 section
vec2 brownConradyDistortion(vec2 uv, float scalar) {
    uv = (uv - 0.5) * 2.0;

    // positive values of K1 give barrel distortion, negative give pincushion
    float barrelDistortion1 = -0.02 * scalar; // K1 in text books
    float barrelDistortion2 = 0.0 * scalar; // K2 in text books

    float r2 = dot(uv, uv);
    uv *= 1.0 + barrelDistortion1 * r2 + barrelDistortion2 * r2 * r2;

    // tangential distortion (due to off center lens elements)
    // is not modeled in this function, but if it was, the terms would go here
    //    return uv * 0.5 + 0.5;
    return (uv / 2.0) + 0.5;
}

/**
 * Returns the color of the main image at the [uv] coordinates, with the Chroma01 filter applied.
 */
vec4 chroma01(vec2 uv) {
    float maxDistort = 4.0;
    float scalar = 1.0 * maxDistort;
    vec4 colourScalar = vec4(700.0, 560.0, 490.0, 1.0); // Based on the true wavelengths of red, green, blue light.
    colourScalar /= max(max(colourScalar.x, colourScalar.y), colourScalar.z);
    colourScalar *= 2.0;

    colourScalar *= scalar;

    const float numTaps = 2.0;
    vec4 fragColor = vec4(0.0, 0.0, 0.0, 1.0);

    for (float tap = 0.0; tap < numTaps; tap += 1.0) {
        fragColor.r += imageColor(brownConradyDistortion(uv, colourScalar.r)).r;
        fragColor.g += imageColor(brownConradyDistortion(uv, colourScalar.g)).g;
        fragColor.b += imageColor(brownConradyDistortion(uv, colourScalar.b)).b;

        colourScalar *= 0.99;
    }

    fragColor /= numTaps;
    fragColor.a = 1.0;
    return fragColor;
}

// Chroma 02 section

/**
 * Returns the color of the main image at the [uv] coordinates, with the Chroma02 filter applied.
 */
vec4 chroma02(vec2 uv) {
    float amount = 0.0;
    amount = (1.0 + sin(time * 6.0)) * 0.8;
    amount *= (1.0 + sin(time * 16.0)) * 0.8;
    amount *= (1.0 + sin(time * 19.0)) * 0.8;
    amount *= (1.0 + sin(time * 27.0)) * 0.8;
    amount = pow(amount, 3.0);
    amount *= 0.05;

    vec4 color;
    color.r = imageColor(vec2(uv.x + amount, uv.y)).r;
    color.g = imageColor(uv).g;
    color.b = imageColor(vec2(uv.x - amount, uv.y)).b;

    color *= (1.0 - amount * 0.5);
    color.a = 1.0;
    return color;
}

// Chroma 03 section

/**
 * Returns the color of the main image at the [uv] coordinates, with the Chroma03 filter applied.
 */
vec4 chroma03(vec2 uv) {
    float redShift = 0.16;
    float greenShift = 0.08;
    float blueShift = 0.023;
    float aberrationStrength = 1.0;

    float uvXOffset = uv.x * 2.0 - 1.0;
    float customXOffset = 0.0;
    float uvXFromCenter = uvXOffset - customXOffset;
    float finalUVX = uvXFromCenter * abs(uvXFromCenter) * aberrationStrength;

    float redChannel = imageColor(vec2(uv.x - (finalUVX * redShift), uv.y)).r;
    float greenChannel = imageColor(vec2(uv.x - (finalUVX * greenShift), uv.y)).g;
    float blueChannel = imageColor(vec2(uv.x - (finalUVX * blueShift), uv.y)).b;

    return vec4(redChannel, greenChannel, blueChannel, 1.0);
}

// Chroma 04 section

/**
 * Returns the color of the main image at the [uv] coordinates, with the Chroma04 filter applied.
 */
vec4 chroma04(vec2 uv) {
    float separation = 0.01;
    float sinWave = sin(time) * separation;
    float cosWave = cos(time) * separation;

    vec2 clockwise = vec2(sinWave, cosWave);
    vec2 counterclockwise = vec2(-sinWave, cosWave);

    vec2 offsetRed = clockwise;
    vec2 offsetGreen = counterclockwise;
    vec2 offsetBlue = vec2(1, 1) * cosWave;

    float red = (imageColor(uv - offsetRed)).r;
    float green = (imageColor(uv - offsetGreen)).g;
    float blue = (imageColor(uv - offsetBlue)).b;
    return vec4(red, green, blue, 1.0);
}

// Sharpening section

// The standard sharpening kernel.
// TODO: move this to app side for performance.
const mat3 sharpenKernel = mat3(
    -1., -1., -1.,
    -1., 9., -1.,
    -1., -1., -1.
);

/**
 * Returns the color corresponding to main texture image with the sharpening filter applied.
 * The amount of sharpening applied is defined by [sharpen].
 */
vec4 sharpened(vec2 uv) {
    vec3 sharpened = withKernel3(uv, sharpenKernel).rgb;
    vec4 originalColor = imageColor(uv);
    vec4 outputColor = vec4(sharpened.rgb, originalColor.a);
    return mix(originalColor, outputColor, sharpen);
}

/*
 * COLOR REGULATION SECTION
 */

// LUT section

/**
 * Returns the LUT-mapped color for the given input [color].
 * Applies the intensity defined by [lutIntensity] by applying a linear mixing with the original
 * color.
 */
vec4 withLUTTransformation(vec4 color) {
    float blueColor = color.b * 63.0;

    vec2 quad1;
    quad1.y = floor(floor(blueColor) / 8.0);
    quad1.x = floor(blueColor) - (quad1.y * 8.0);

    vec2 quad2;
    quad2.y = floor(ceil(blueColor) / 8.0);
    quad2.x = ceil(blueColor) - (quad2.y * 8.0);

    vec2 texPos1;
    texPos1.x = (quad1.x * 0.125) + 0.5 / 512.0 + ((0.125 - 1.0 / 512.0) * color.r);
    texPos1.y = (quad1.y * 0.125) + 0.5 / 512.0 + ((0.125 - 1.0 / 512.0) * color.g);

    vec2 texPos2;
    texPos2.x = (quad2.x * 0.125) + 0.5 / 512.0 + ((0.125 - 1.0 / 512.0) * color.r);
    texPos2.y = (quad2.y * 0.125) + 0.5 / 512.0 + ((0.125 - 1.0 / 512.0) * color.g);

    vec4 newColor1 = lutColor(texPos1);
    vec4 newColor2 = lutColor(texPos2);
    vec4 newColor = mix(newColor1, newColor2, fract(blueColor));
    vec4 finalColor = mix(color, newColor, lutIntensity);
    // Preserve original alpha
    finalColor.a = color.a;
    return finalColor;
}

// Contrast section

/**
 * Returns the color contrast matrix for the given contrast value.
 *
 * TODO: Move the matrix definition on the app side for performance.
 */
mat4 contrastMatrix(float contrastValue) {
    float t = (1.0 - contrastValue) / 2.0;
    return mat4(contrastValue, 0, 0, 0,
        0, contrastValue, 0, 0,
        0, 0, contrastValue, 0,
        t, t, t, 1);
}

/** Returns the given [color] with the application of [contrast]. */
vec4 withContrast(vec4 color) {
    return contrastMatrix(contrast) * color;
}

// Exposure section

/** Returns the given [color] with the application of [exposure]. */
vec4 withExposure(vec4 color) {
    color.rgb *= pow(2.0, exposure);
    return color;
}

// Saturation section

// TODO: Consider the possibility of using a concatenatable color matrix instead.
/** Returns the given [color] with the application of [saturation]. */
vec4 withSaturation(vec4 color) {
    float grayscale = ((0.3 * color.r) + (0.59 * color.g) + (0.11 * color.b));
    vec3 intensity = vec3(grayscale, grayscale, grayscale);
    vec3 finalColor = mix(intensity, color.rgb, saturation);
    return vec4(finalColor, color.a);
}

// Temperature section

/** Returns the given [color] with the application of [temperatureColor] */
vec4 withTemperature(vec4 color) {
    color = clamp(color, 0., 1.);
    const vec3 lumReference = vec3(0.2126, 0.7152, 0.0722);
    // Apply temperature adjustment
    vec3 fullTemperaturedColor = color.rgb * temperatureColor;
    vec3 outColor = fullTemperaturedColor;
    float originalLumCoeff = dot(outColor, lumReference);
    float outputLumCoeff = dot(color.rgb, lumReference);
    float lumMultiplier = outputLumCoeff / max(originalLumCoeff, 1e-5);
    outColor *= lumMultiplier;
    return vec4(outColor.rgb, color.a);
}

/** Returns the given [color] with the application of [hueShiftAngle] */
vec4 withHueShift(vec4 color) {
    // TODO: Investigate potential optimization via color matrix
    //   https://stackoverflow.com/questions/8507885/shift-hue-of-an-rgb-color
    vec3 originalHSV = rgb2hsv(color.rgb);
    float newH = mod((originalHSV.x + hueShiftAngle), 1.);
    vec3 newHSV = vec3(newH, originalHSV.yz);
    vec3 outputColor = hsv2rgb(newHSV);
    return vec4(outputColor.rgb, color.a);
}

// Chroma key section

/** 
 * Returns the channel-wise distance between two HSV colors, considering a periodicity of 1 for H.
 */
vec3 hsvDistance(vec3 color1, vec3 color2) {
    vec3 dist = abs(color1 - color2);
    dist.x = min(dist.x, 1. - dist.x);
    return dist;
}

/** Returns the given [color] with the application of the  */
vec4 withChromaKey(vec4 color) {
    const float chromaBorderSoftness = 0.05; // it may be exposed as a uniform
    vec3 threshold = vec3(chromaHueThreshold, chromaShadowThreshold, chromaShadowThreshold);
    vec3 imageHSV = rgb2hsv(color.rgb);
    // TODO: Potentially calculated once from app (PERFORMANCE)
    vec3 keyHSV = rgb2hsv(chromaColor.rgb);

    // compute the distance between the current pixel color and the target color (key)
    vec3 dist = hsvDistance(imageHSV, keyHSV);

    // soften the border smoothing between two thresholds
    vec3 lowThres = threshold;
    vec3 highThres = threshold * (1. + chromaBorderSoftness);
    // the normalized distance is a value between 0 and 1
    vec3 normDist = smoothstep(lowThres, highThres, dist);

    // Close points will have 0 for all dimensions of normDist, far point will have 1 at least in
    // one of those dimensions. Summ all demensions and cap the result to 1.0.
    color.a *= min(1.0, normDist.x + normDist.y + normDist.z);
    return color;
}

/*
 * MASKS
 * The following masks are non-parametric (except for the rectangle corner size).
 * Any parameter (such as width/height/rotation/position) should be achieved by
 * applying the corresponding affine transformation to the input point coordinate.
 */

// Rectangular mask section

/**
 * Returns the distance between [point] and the rectangle with vertices in
 * `[(0, 0), (0, 1), (1, 1), (1, 0)]`.
 * The distance is adjusted to the corner radius considering the given [maskRectCornerRadius].
 * The returned distance is positive if [point] is outside the rectangle, zero if [point] is on the
 * perimeter of the rectangle and negative if [point] is inside the rectangle.
 */
float maskDistanceRect(vec2 point) {
    float cr = maskRectCornerRadius / 2.;
    vec2 q = (abs(point - 0.5) - 0.5) * maskDistanceMultiplier + cr;
    return min(max(q.x, q.y), 0.0) + length(max(q, 0.0)) - cr;
}
// Ellipsoidal mask section

/**
 * Returns the distance between [point] and the ellipse centered in `(0.5, 0.5)` and with radius
 * `(0.5, 0.5)`.
 * The returned distance is positive if [point] is outside the ellipse, zero if [point] is on the
 * perimeter of the ellipse and negative if [point] is inside the ellipse.
 */
float maskDistanceEllipse(vec2 point) {
    // TODO: Calculate a distance that is independent from the scaling applied to the system,
    //  see maskDistanceMultiplier.
    return distance(point, vec2(0.5)) - 0.5;
}

// Line mask section

/**
 * Returns the distance between [point] and the horizontal line defined by `y = 0.5`
 * The returned distance is positive if [point].y < 0.5, zero if [point] is on the
 * line and negative if [point].y > 0.5.
 */
float maskDistanceLine(vec2 point) {
    return (0.5 - point.y) * maskDistanceMultiplier.y;
}

/**
 * Returns the minimum distance between [point] and the two horizontal lines defined by `y = 0`
 * and `y = 1`. The returned distance is positive if [point].y < 0 or [point].y > 1, zero if [point]
 * is on one of the lines, and negative if [point].y < 1 and [point].y > 0.
 */
float maskDistanceBand(vec2 point) {
    return max(point.y - 1., 0. - point.y) * maskDistanceMultiplier.y;
}

// Generic geometrical mask section

/**
 * Returns the alpha multiplier to be applied to the color for a given mask distance.
 * It applies [maskInvert] and [maskSmoothness] parameters.
 *
 * @param dist The mask distance for which the alpha multiplier should be calculated.
 */
float maskAlphaMultiplierFor(float dist) {
    float s = maskSmoothness / 2.;
    float m = smoothstep(s, -s, dist);
    m = maskInvert ? 1. - m : m;
    return m;
}

// Premulitiplication utility

/** Returns the unpremultiplied [color].*/
vec4 unpremultiplied(vec4 color) {
    if (color.a == 0.) {
        return vec4(0.);
    }
    color.rgb /= color.a;
    return color;
}

/** Returns the premultiplied [color].*/
vec4 premultiplied(vec4 color) {
    color.rgb *= color.a;
    return color;
}

// Const section

#define E_CHROMA01 1
#define E_CHROMA02 2
#define E_CHROMA03 3
#define E_CHROMA04 4
#define E_GLITCH01 5
#define E_GLITCH02 6
#define E_GLITCH03 7
#define E_VINTAGE01 8
#define E_VINTAGE02 9
#define E_VINTAGE03 10
#define E_SHARPEN 11

#define M_RECT 1
#define M_LINE 2
#define M_ELLIPSE 3
#define M_BAND 4

// Core section

/**
 * Returns the color corresponding to the shader configuration at the texture coordinate [uv] and
 * mask coordinate [maskUv].
 */
vec4 processedColor(vec2 uv, vec2 maskUv) {
    vec4 color;
    if (effect == E_CHROMA01) {
        color = chroma01(uv);
    } else if (effect == E_CHROMA02) {
        color = chroma02(uv);
    } else if (effect == E_CHROMA03) {
        color = chroma03(uv);
    } else if (effect == E_CHROMA04) {
        color = chroma04(uv);
    } else if (effect == E_GLITCH01) {
        color = glitch01(uv);
    } else if (effect == E_GLITCH02) {
        color = glitch02(uv);
    } else if (effect == E_GLITCH03) {
        color = glitch03(uv);
    } else if (effect == E_VINTAGE01) {
        color = vintage01(uv);
    } else if (effect == E_VINTAGE02) {
        color = vintage02(uv);
    } else if (effect == E_VINTAGE03) {
        color = vintage03(uv);
    } else if (effect == E_SHARPEN) {
        color = sharpened(uv);
    } else {
        color = imageColor(uv);
    }
    color = unpremultiplied(color);
    
    color = clamp(color, 0., 1.);

    if (lutIntensity != 0.) {
        color = withLUTTransformation(color);
    }
    if (contrast != 1.) {
        color = withContrast(color);
    }
    if (saturation != 1.) {
        color = withSaturation(color);
    }
    if (exposure != 0.) {
        color = withExposure(color);
    }
    if (useTemperature) {
        color = withTemperature(color);
    }
    if (hueShiftAngle != 0.) {
        color = withHueShift(color);
    }
    if (useChromaKey) {
        color = withChromaKey(color);
    }
    if (mask != 0) {
        float maskAlphaMultiplier = 1.;
        if (mask == M_RECT) {
            maskAlphaMultiplier = maskAlphaMultiplierFor(maskDistanceRect(maskUv));
        } else if (mask == M_LINE) {
            maskAlphaMultiplier = maskAlphaMultiplierFor(maskDistanceLine(maskUv));
        } else if (mask == M_ELLIPSE) {
            maskAlphaMultiplier = maskAlphaMultiplierFor(maskDistanceEllipse(maskUv));
        } else if (mask == M_BAND) {
            maskAlphaMultiplier = maskAlphaMultiplierFor(maskDistanceBand(maskUv));
        }
        color.a *= maskAlphaMultiplier;
    }
    color.a *= opacity;
    return premultiplied(color);
}

void main() {
    gl_FragColor = processedColor(texCoordinate, maskCoordinate);
}
