


local venuscore = require "venuscore"
local CommandBase = require "venuscore.command.command"
local ContentManager = require "apolloengine.content.contentmanager"
local BundleSystem = require "venuscore.bundle.bundlesystem"
local stringsplit = require "stringsplit"
local apollocore = require "apollocore"

--[[
  bundlepath = venuscore.IFileSystem:PathAssembly(bundlepath);
  local file = io.open(bundlepath, "rb");
  local str = file:read("*a");
  return BundleSystem:DeSerialize(str, false, deserializeMode);
--]]

local DeserializeCmd = CommandBase:extend("DeserializeCmd");

function DeserializeCmd:new(filePath,deserializeMode)
  DeserializeCmd.super.new(self);
  self._bundlePath = venuscore.IFileSystem:PathAssembly(filePath);
  self._deserializeMode = deserializeMode or BundleSystem.DeserializeMode.Prefab;
  self._toCacheIDs = true;  --对于prefab的反序列化，为了保持多个场景中Node和Component的ID一致，需要特殊处理
  self._cachedIDList = {}; --对于prefab的反序列化，缓存的初次反序列化的得到的UUID,用来和其他的场景同步
end

function DeserializeCmd:DoIt()
  local nodeList = {};
  local scene = self:GetScene();
  if scene then
    local file = io.open(self._bundlePath, "rb");
    local str = file:read("*a");
    nodeList = scene:Deserialize(str,self._deserializeMode);
    if self._deserializeMode == BundleSystem.DeserializeMode.Prefab then
      if self._toCacheIDs == true then
        self:_CachedIDs(nodeList);
        self._toCacheIDs = false;
      else
        self:_SyncIDs(nodeList);
      end
    elseif self._deserializeMode == BundleSystem.DeserializeMode.Scene then
      if self._toCacheIDs == true then
        self:_CachedIDsScene(nodeList);
        self._toCacheIDs = false;
      else
        self:_SyncIDsScene(nodeList);
      end
    end
  end
  return nodeList;
end

function DeserializeCmd:_SyncIDs(nodeList)
  for i = 1, #nodeList do
    local node = nodeList[i];
    node:SetStaticID(self._cachedIDList[i][1]);
    local comIDs = {};
    local compTypeList = node:GetComponentTypes();
    for j = 1, #compTypeList do
      local comType = compTypeList[j];
      local comp = node:GetComponent(comType);
      if comp then
        comp:SetStaticID(self._cachedIDList[i][2][comType]);
      end
    end
  end
end


function DeserializeCmd:_CachedIDs(nodeList)
  for i = 1, #nodeList do
    local node = nodeList[i];
    local comIDs = {};
    local compTypeList = node:GetComponentTypes();
    for j = 1, #compTypeList do
      local comType = compTypeList[j];
      local comp = node:GetComponent(comType);
      if comp then
        comIDs[comType] = comp:GetStaticID();
      end
    end
    table.insert(self._cachedIDList,{node:GetStaticID(),comIDs});
  end
end

function DeserializeCmd:_SyncIDsScene(nodeList)
  for i = 1, #nodeList do
    local node = nodeList[i];
    node:SetStaticID(self._cachedIDList[i][1]);
    local comIDs = {};
    local compTypeList = node:GetComponentTypes();
    for j = 1, #compTypeList do
      local comType = compTypeList[j];
      local comp = node:GetComponent(comType);
      if comp then
        if comType == apollocore.Node.CT_SCRIPT then
            local instance = comp.Instances;
            for key,value in pairs(instance) do
              value._hostID =self._cachedIDList[i][2][comType];
            end
            local idPathList = stringsplit(self._cachedIDList[i][2][comType], "/");
            comp:SetStaticID(idPathList[2]);
        else
            comp:SetStaticID(self._cachedIDList[i][2][comType]);
        end
      end
    end
  end
end

function DeserializeCmd:_CachedIDsScene(nodeList)
  for i = 1, #nodeList do
    local node = nodeList[i];
    local comIDs = {};
    local compTypeList = node:GetComponentTypes();
    for j = 1, #compTypeList do
      local comType = compTypeList[j];
      local comp = node:GetComponent(comType);
      if comp then
        if comType == apollocore.Node.CT_SCRIPT then
          local instance = comp.Instances;
          for key,value in pairs(instance) do
            local idPathList = stringsplit(value._hostID, "/");
            comIDs[comType] = node:GetStaticID().."/"..idPathList[2];
            break;
          end
          if comIDs[comType] == nil then
            comIDs[comType] = comp:GetContentPath();
          end
          for key,value in pairs(instance) do
            value._hostID = comIDs[comType];
          end
        else
          comIDs[comType] = comp:GetStaticID();
        end
      end
    end
    table.insert(self._cachedIDList,{node:GetStaticID(),comIDs});
  end
end

--反序列化的命令怎么处理？
function DeserializeCmd:Undo()
  local scene = self:GetScene();
  if scene then
    for i = 1, #self._cachedIDList do
      scene:DestroyNodeByUUID(self._cachedIDList[i][1]);
    end
  end
end

return DeserializeCmd;
