local ltdiff = require "luatablediff.ltdiff"
local ltutils = require "luatablediff.utils"
local venusjson = require "venusjson"
local SerializationTool = require "venuscore.bundle.serialization.serializationtool"
local venuscore = require "libvenuscore"
local Serializer = require "venuscore.bundle.fserializer.fserializer"

local PrefabTool = {};

PrefabTool.PrefabInfoPool = {};

function PrefabTool:PatchDiff(prefabInfo, diffInfo)
  local newInfo = ltdiff.patch(prefabInfo, diffInfo);
  return newInfo;
end

function PrefabTool:ReadPrefabInfo(path)
  local info = nil;
  if self.PrefabInfoPool[path] ~= nil then
    info = self.PrefabInfoPool[path];
  else
    if path ~= nil and venuscore.IFileSystem:isFileExist(path) then
      local file = io.open(venuscore.IFileSystem:PathAssembly(path), "rb");
      local str = file:read("*a");
      local decoder = SerializationTool:GetDecoder(str);
      info = SerializationTool:GetAttribute(decoder);
      if info.__typename == "PrefabNode" then
        info = self:ReadPrefabInfo(info.PrefabPath);
        local diff = self:UnserializeTable(info.Diff);
        info = ltdiff.patch(info, diff);
      end
      self.PrefabInfoPool[path] = info;
    else
      ERROR(string.format("prefab create failed, %s is not exist", path));
      info = nil;
    end
  end
  return table.deepcopy(info);
end

function PrefabTool:GetPrefabDependcy(path, dependcy)
  if path ~= nil and venuscore.IFileSystem:isFileExist(path) then
    table.insert(dependcy, path);
    local file = io.open(venuscore.IFileSystem:PathAssembly(path), "rb");
    local str = file:read("*a");
    local decoder = SerializationTool:GetDecoder(str);
    local info = SerializationTool:GetAttribute(decoder);
    if info.__typename == "PrefabNode" then
      self:GetPrefabDependcy(info.PrefabPath, dependcy);
    end
  end
end

function PrefabTool:GenerateSerializationInfo(prefabPath, info)
  --原prefab信息
  local prefabInfo = self:ReadPrefabInfo(prefabPath);
  if prefabInfo == nil then
    ERROR("origin prefab path is not exist, generate new prefab");
    return info;
  end
  --当前node信息
  local serializer = Serializer();
  local encoder = serializer:GetEncoder();
  encoder:SetFieldInt("version", 1);
  local childEncoder = encoder:GetChild();
  SerializationTool:_SerializeCls(info, childEncoder);
  encoder:SetFieldClass("Data", childEncoder);
  local bufferAsString = serializer:GetBuffer();
  local decoder = serializer:GetDecoderFromBuffer(bufferAsString);
  info = SerializationTool:GetAttribute(decoder);
  --生成diff
  local diff = ltdiff.diff(prefabInfo, info);
  local diffStr = self:SerializeTable(diff);
  local prefabAttributes = {
    typename = "PrefabNode",
    fieldname = prefabPath,
    members = {
      {
        typename = "string",
        fieldname = "PrefabPath",
        members = {},
        value = prefabPath
      },
      {
        typename = "string",
        fieldname = "Diff",
        members = {},
        value = diffStr
      }
    }
  };
  return prefabAttributes;
end

function PrefabTool:SerializeTable(obj)
  local lua = ""
  local t = type(obj)
  if t == "number" then
    lua = lua .. obj
  elseif t == "boolean" then
    lua = lua .. tostring(obj)
  elseif t == "string" then
    lua = lua .. string.format("%q", obj)
  elseif t == "table" then
    lua = lua .. "{\n"
    for k, v in pairs(obj) do
      lua = lua .. "[" .. self:SerializeTable(k) .. "]=" .. self:SerializeTable(v) .. ",\n"
    end
    local metatable = getmetatable(obj)
    if metatable ~= nil and type(metatable.__index) == "table" then
      for k, v in pairs(metatable.__index) do
        lua = lua .. "[" .. self:SerializeTable(k) .. "]=" .. self:SerializeTable(v) .. ",\n"
      end
    end
    lua = lua .. "}"
  elseif t == "nil" then
    return nil
  else
    error("can not serialize a " .. t .. " type.")
  end
  return lua
end

function PrefabTool:UnserializeTable(lua)
  local t = type(lua)
  if t == "nil" or lua == "" then
    return nil
  elseif t == "number" or t == "string" or t == "boolean" then
    lua = tostring(lua)
  else
    error("can not unserialize a " .. t .. " type.")
  end
  lua = "return " .. lua
  local func = load(lua)
  if func == nil then
    return nil
  end
  return func()
end

return PrefabTool;