require "utility"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local venuscore = require "venuscore"
local framedisp = {}

framedisp.queue = 100;

function framedisp:Initialize(host, size)

    self.SourceTex     = apolloengine.TextureEntity();
    self.LastSourceTex = self.SourceTex;

    host:RegisterScriptParameter(self, "SourceTex");
    host:RegisterScriptParameter(self, "Source");

    self:_InitGreyPass(host, size);
    self:_InitBlurPass(host, size);
    self:_InitDispPass(host, size);
    self:_InitBlendPass(host, size);
    return self.queue;
end

function framedisp:_InitGreyPass(host, size)
    self.Brightness = mathfunction.vector1(1.0);
    self.Saturation = mathfunction.vector1(1.0);
    self.Contrast   = mathfunction.vector1(1.0);

    self.GreyRO = host:CreateRenderObject("comm:documents/shaders/posteffect/greydisp.material");
    self.GreyRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_A, size );

    host:RegisterScriptParameter(self, "Brightness");
    host:RegisterScriptParameter(self, "Saturation");
    host:RegisterScriptParameter(self, "Contrast");
end

function framedisp:_InitDispPass(host, size)
    self.VerticalOffset = mathfunction.vector1(0.0);
    self.HorizontalOffset = mathfunction.vector1(0.0);
    self.FrameTex = apolloengine.TextureEntity();
    self.DistRO = host:CreateRenderObject("comm:documents/shaders/posteffect/framedisp.material");
    self.DistRT =  host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_D, size );

    host:RegisterScriptParameter(self, "FrameTex");
    host:RegisterScriptParameter(self, "VerticalOffset");
    host:RegisterScriptParameter(self, "HorizontalOffset");
end

function framedisp:_InitBlurPass(host, size)
    self.beginstep = 1;
    self.Blurriness = mathfunction.vector1(1.0);
    self.Stepsize = mathfunction.vector2(1.0, 1.0);
    local GuassStep = self.Stepsize / size;

    self.GAUSSIAN_STEP = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "GAUSSIAN_STEP");

    self.RoGuassX = host:CreateRenderObject( "comm:documents/material/gaussianblur.material" );
	  self.RoGuassX:SetParameter( self.GAUSSIAN_STEP, mathfunction.vector2( GuassStep:x(), 0 ) );
	
	  self.RoGuassY = host:CreateRenderObject( "comm:documents/material/gaussianblur.material" );
    self.RoGuassY:SetParameter( self.GAUSSIAN_STEP, mathfunction.vector2( 0, GuassStep:y() ) );
    
    self.GuassXRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_B, size );
    self.GuassYRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_C, size );

    host:RegisterScriptParameter(self, "Blurriness");
end

function framedisp:_InitBlendPass(host, size)
    self.BlendMode = mathfunction.vector1(2.0);
    self.BlendOpacity = mathfunction.vector1(2.0);
    self.BlendRO = host:CreateRenderObject("comm:documents/shaders/posteffect/blendmode.material");

    host:RegisterScriptParameter(self, "BlendMode");
    host:RegisterScriptParameter(self, "BlendOpacity");
end

-- Grey Pass Begin
function framedisp:_DrawGreyPass(pipeline, Original, Scene, Output)
    self.GreyRT:PushRenderTarget();
    self.GreyRT:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR);

    local materials = self.GreyRO.MaterialEntities;
    local material = materials[1];

    if self.LastSourceTex ~= self.SourceTex then
        material:SetParameter("_MainTex", self.SourceTex);
    else
        material:SetParameter("_MainTex", Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
    end

    
    material:SetParameter("_Brightness", self.Brightness);
    material:SetParameter("_Saturation", self.Saturation);
    material:SetParameter("_Contrast", self.Contrast);

    self.GreyRO:Draw(pipeline);
end
-- Grey Pass End

-- Displacement Pass Begin
function framedisp:_DrawDisplacementPass(pipeline, Original, Scene, Output)
    self.DistRT:PushRenderTarget();
    self.DistRT:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR);

    local materials = self.DistRO.MaterialEntities;
    local material = materials[1];
    material:SetParameter("_GreyTex", self.GuassYRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
    material:SetParameter("_FrameTex", self.FrameTex);
    material:SetParameter("_VerticalOffset", self.VerticalOffset);
    material:SetParameter("_HorizontalOffset", self.HorizontalOffset);

    self.DistRO:Draw(pipeline);
end
-- Displacement Pass End

-- Blur Pass Begin
function framedisp:_GetCount(newblurriness)
    local icount = math.ceil(newblurriness);
    icount = math.max(icount,1);
    return icount;
end

function framedisp:_DrawBlurPass(pipeline, Original, Scene, Output)  
    local last = self.GreyRT;
    local BLURRINESS_SCALER = 0.25;
    local newblurriness;
    if type(self.Blurriness) == "number" then
        newblurriness = self.Blurriness * BLURRINESS_SCALER;
    else
        newblurriness = self.Blurriness:x() * BLURRINESS_SCALER;
     end  

    local icount = self:_GetCount(newblurriness);
   
    for i = 1, icount do
      self:_UpdateStepsize(newblurriness, i);
      self.GuassXRT:PushRenderTarget();
      self.GuassXRT:ClearBuffer( apolloengine.RenderTargetEntity.CF_COLOR );
      self.RoGuassX:SetParameter(
        apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
        last:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
      self.RoGuassX:Draw(pipeline);	
      
      self.GuassYRT:PushRenderTarget();
      self.GuassYRT:ClearBuffer( apolloengine.RenderTargetEntity.CF_COLOR );
      self.RoGuassY:SetParameter(
        apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
        self.GuassXRT:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
      self.RoGuassY:Draw(pipeline);
      last = self.GuassYRT;
    end
end

function framedisp:_UpdateStepsize(newblurriness, loop)
    local step = newblurriness - loop + 1;
    step = math.max(step, 0) * 4;
    if step ~= self.beginstep then
      self.beginstep = step;
      self.Stepsize:Set(step, step);
      local GuassStep = self.Stepsize / self.fbosize;
      self.RoGuassX:SetParameter( self.GAUSSIAN_STEP, mathfunction.vector2( GuassStep:x(), 0 ) );
      self.RoGuassY:SetParameter( self.GAUSSIAN_STEP, mathfunction.vector2( 0, GuassStep:y() ) );
    end
end
--Blur Pass End

--Blend Pass Begin
function framedisp:_DrawBlendPass(pipeline, Original, Scene, Output)
    Output:PushRenderTarget();
    Output:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR);

    local materials = self.BlendRO.MaterialEntities;
    local material = materials[1];
    material:SetParameter("_MainTex", Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
    material:SetParameter("_BlendTex", self.DistRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
    material:SetParameter("_BlendMode", self.BlendMode);
    material:SetParameter("_BlendOpacity", self.BlendOpacity);

    self.BlendRO:Draw(pipeline);
end
--Blend Pass End

function framedisp:Resizeview(size)
    self.fbosize = mathfunction.vector2(size:x(), size:y());
    local GuassStep = self.Stepsize / size;
  
    self.RoGuassX:SetParameter( self.GAUSSIAN_STEP, mathfunction.vector2( GuassStep:x(), 0 ) );
    self.RoGuassY:SetParameter( self.GAUSSIAN_STEP, mathfunction.vector2( 0, GuassStep:y() ) );
end

function framedisp:Process(pipeline, Original, Scene, Output)
    self:_DrawGreyPass(pipeline, Original, Scene, Output);
    self:_DrawBlurPass(pipeline, Original, Scene, Output);
    self:_DrawDisplacementPass(pipeline, Original, Scene, Output);
    self:_DrawBlendPass(pipeline, Original, Scene, Output);
end

return framedisp;