local apollonode = require "apolloutility.apollonode"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local libmasquerade = require "libmasquerade"
local defined = require "liveportrait.defined"

local mouthrig = {}
local emission = mathfunction.vector3(0, 0, 0);
local ambient = mathfunction.vector3(0.06, 0.06, 0.06);
local diffuse = mathfunction.vector3(0.4, 0.4, 0.4);
local specular = mathfunction.vector3(0.5, 0.5, 0.5);
local shininess = mathfunction.vector1(1);
local lightposn = mathfunction.vector4(0.3, 0.3, 1, 0)
local lightcolor = mathfunction.vector4(1, 1, 1, 1)

function mouthrig:Initialize(lowerteeth_path, upperteeth_path, tongue_path)
  self.mouth_model = libmasquerade.MouthRig()
  self.mesh = {}
  self.normal = {}
  self:InitShaderParams()
  self:LoadMesh(lowerteeth_path, upperteeth_path, tongue_path)
end

function mouthrig:LoadMesh(lowerteeth_path, upperteeth_path, tongue_path)
  local material_path  = "docs:liveportrait/material/mouth.material";
  local lowerteeth = apollonode.ModelNode()
  lowerteeth:CreateResource(lowerteeth_path, material_path)
  self.mesh["lowerteeth"] = lowerteeth
  
  local upperteeth = apollonode.ModelNode()
  upperteeth:CreateResource(upperteeth_path, material_path)
  self.mesh["upperteeth"] = upperteeth
  
  local tongue = apollonode.ModelNode()
  tongue:CreateResource(tongue_path, material_path)
  self.mesh["tongue"] = tongue
  
  local lowerteeth_bindbox = lowerteeth:GetBindBox()
  local lmin = lowerteeth_bindbox:GetMin()
  local lmax = lowerteeth_bindbox:GetMax()
  local upperteeth_bindbox = upperteeth:GetBindBox()
  local umin = upperteeth_bindbox:GetMin()
  local umax = upperteeth_bindbox:GetMax()
  
  local min_x = math.min(lmin:x(), umin:x())
  local min_y = math.min(lmin:y(), umin:y())
  local min_z = math.min(lmin:z(), umin:z())
  local max_x = math.max(lmax:x(), umax:x())
  local max_y = math.max(lmax:y(), umax:y())
  local max_z = math.max(lmax:z(), umax:z())
  local teeth_center = mathfunction.vector3((min_x + max_x) / 2, (min_y + max_y) / 2, (min_z + max_z) / 2)
  
  local tongue_bindbox = tongue:GetBindBox()
  local tmin = tongue_bindbox:GetMin()
  local tmax = tongue_bindbox:GetMax()
  local pivot = mathfunction.vector3((tmin:x() + tmax:x()) / 2, (tmin:y() + tmax:y()) / 2, tmin:z())
  
  self:InitModel(teeth_center, pivot)
end

function mouthrig:InitShaderParams()
  self.projSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "PROJECTION");
  self.viewSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "MODELVIEW");
  self.invViewSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "INVMODELVIEW");
  self.enableTexSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ENABLETEXTURE");
  self.enableNormalSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ENABLENORMAL");
  self.ambientSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "AMBIENT");
  self.specularSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "SPECULAR");
  self.emissionSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "EMISSION");
  self.diffuseSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "DIFFUSE");
  self.shininessSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "SHININESS");
  self.lightposnSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "LIGHTPOSN");
  self.lightcolorSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "LIGHTCOLOR");
end

function mouthrig:SetShaderParams(width, height)
  
  local projection = mathfunction.Matrix44(
    mathfunction.vector4( 2 / width, 0, 0, 0),
    mathfunction.vector4(0, 2 / height, 0, 0),
    mathfunction.vector4(0, 0, -2 / 4000, 0),
    mathfunction.vector4(-1, -1, 0, 1)
  );
  local modelview = mathfunction.Matrix44();
  modelview:SetIdentity()
  
  for k, v in pairs(self.mesh) do
    for key, node in pairs(v.nodes) do 
      node.render:EraseRenderProperty(apolloengine.RenderComponent.RP_CULL);
      node:SetParameter(self.projSlot, projection)
      node:SetParameter(self.viewSlot, modelview)
      node:SetParameter(self.invViewSlot, modelview)
      node:SetParameter(self.enableTexSlot, mathfunction.vector1(1))
      if k == "tongue" then
        node:SetParameter(self.enableNormalSlot, mathfunction.vector1(0))
      else
        node:SetParameter(self.enableNormalSlot, mathfunction.vector1(1))
      end
      
      node:SetParameter(self.ambientSlot, ambient)
      node:SetParameter(self.specularSlot, specular)
      node:SetParameter(self.emissionSlot, emission)
      node:SetParameter(self.diffuseSlot, diffuse)
      node:SetParameter(self.shininessSlot, shininess)
      node:SetParameter(self.lightposnSlot, lightposn)
      node:SetParameter(self.lightcolorSlot, lightcolor)
      node:SetSequence(defined.SEQUENCE);
      node:SetShow(false)
    end
  end
  
end

function mouthrig:InitModel(teeth_center, pivot)
  self.mouth_model:Init(teeth_center, pivot)
end

function mouthrig:SetRestPose(up, low)
  self.mouth_model:SetRestpose(up, low)
end

function mouthrig:SetScale(scale)
  self.mouth_model:SetScale(scale)
end

function mouthrig:Update(up, low, stacked_modelview)
  self.mouth_model:UpdateModelview(up, low, stacked_modelview)
  local modelview = {}
  local trans_inv_modelview = {}
  modelview["lowerteeth"] = self.mouth_model:GetLowerModelview()
  modelview["upperteeth"] = self.mouth_model:GetUpperModelview()
  modelview["tongue"] = modelview["lowerteeth"]
  
  trans_inv_modelview["lowerteeth"] = self.mouth_model:GetTransInvLowerModelview()
  trans_inv_modelview["upperteeth"] = self.mouth_model:GetTransInvUpperModelview()
  trans_inv_modelview["tongue"] = trans_inv_modelview["lowerteeth"]
  
  for k, v in pairs(self.mesh) do
    for key, node in pairs(v.nodes) do 
      node:SetParameter(self.viewSlot, modelview[k])
      node:SetParameter(self.invViewSlot, trans_inv_modelview[k])
      node:SetShow(true)
    end
  end
  
end

function mouthrig:Reset()
  if self.mesh ~= nil then
    for k, v in pairs(self.mesh) do
        v:SetShow(false)
    end
  end
end

function mouthrig:Clear()
  --目前node需要自己删除
  if self.mesh ~= nil then
      for k, v in pairs(self.mesh) do
         v:Destroy();
      end
  end
  
  self.mesh = nil
  self.mouth_model = nil

end


return mouthrig
