local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local videodecet = require "videodecet"
local venusjson = require "venusjson"
local venuscore = require "venuscore"
local libmasquerade = require "libmasquerade"
local Render =  require "liveportrait.portraitrender"
local ML = require "liveportrait.portraitml"
local Anim = require "liveportrait.portraitanim"
local Mouth = require "liveportrait.mouthrig"
local defined = require "liveportrait.defined"
local likeapp = require "likeapp"

local liveportrait = {}

function liveportrait:Initialize(start_time, last_time)
  self.start_time = start_time
  self.last_time = last_time
  self.face_model = nil
  self.render = Render()
  self.portrait = nil
  self.anim = Anim
  self.ml = nil
  self.mouth = Mouth
  self.timer = 0
  self.windows_platform = false
  
  if self.start_time == nil then
    self.start_time = 0
  end
  
  if self.last_time == nil then
    self.last_time = 10000
  end
end

function liveportrait:LoadConfig(path)
  self.windows_platform = true
  local rootconfig = venusjson.LaodJsonFile(path)
  if rootconfig == nil then
    return false
  end
  
  if rootconfig.anim_file == nil or rootconfig.img == nil then 
    return false
  end
  
  self.portrait_tex = self:LoadTexture(rootconfig.img)
  self.portrait = self.portrait_tex:GetSourceStream();
  -- record here, load later
  local pathDir= string.match(path, "(.+)/[^/]*%.%w+$");
  self.anim_path = venuscore.IFileSystem:PathAssembly(pathDir.. "/" .. rootconfig.anim_file);
  
  self:LoadWrinkleResource(rootconfig.wrinkle_config, 
                           rootconfig.wrinkle, 
                           rootconfig.forehead_mask, 
                           rootconfig.ldecree_mask, 
                           rootconfig.rdecree_mask);   
 
  local size = self.portrait:GetSize()
  self.w = size:x()
  self.h = size:y()
  self:_InitML(self.w, self.h)
  
  --videodecet:SetShow(false);
  return true;
end

function liveportrait:_InitML(w, h)
  if self.ml == nil then
    self.ml = ML
  end
  
  if self.ml:Initialize(w, h) then
    return true
  else
    self.ml:Clear()
    return false
  end
end

function liveportrait:InitMasquerade(morphable_path, face_mesh_path, template_path, blendshape_path, config_path)
  self.face_model = libmasquerade.LivePortrait()
  if self.face_model:Init(
    self.w,
    self.h,
    morphable_path,
    face_mesh_path,
    template_path,
    blendshape_path,
    "",
    config_path) then
    return true
  else
    LOG("Muglife: Masquerade init failed")
    return false
  end
end

function liveportrait:InitMouth(lowerteeth_path, upperteeth_path, tongue_path)
  self.mouth:Initialize(lowerteeth_path, upperteeth_path, tongue_path)
  local upIdx = self.face_model:GetUpperTeethAttach()
  local lowIdx = self.face_model:GetLowerTeethAttach()
  local up = self.face_model:GetFacePoint(upIdx, true)
  local low = self.face_model:GetFacePoint(lowIdx, true)
  local scale = self.face_model:GetMouthScale()
  self.mouth:SetRestPose(up, low)
  self.mouth:SetScale(scale)
  self.mouth:SetShaderParams(self.w, self.h)
end

function liveportrait:InitAnimeController(anim_path)
  local channel_map = {}
  local bs_num = self.face_model:GetBSNumber();
  for i = 1, bs_num do
    channel_map[self.face_model:GetBSName(i - 1)] = i 
  end
  channel_map["pitch"] = bs_num + 1
  channel_map["yaw"] = bs_num + 2
  channel_map["roll"] = bs_num + 3
  channel_map["l_iris_x"] = bs_num + 4
  channel_map["l_iris_y"] = bs_num + 5
  channel_map["r_iris_x"] = bs_num + 6
  channel_map["r_iris_y"] = bs_num + 7
  channel_map["scale"] = bs_num + 8
  self.anim:Initialize(channel_map)
  local res = self.anim:LoadAnim(anim_path)
  if res ~= true then
    LOG("Muglife: Animation file load failed")
  end
  
  return res
end

function liveportrait:LoadWrinkleResource(wrinkle_config, wrinkle_tex, forehead_mask, ldecree_mask, rdecree_mask)
  if wrinkle_config ~= nil then
    local wrinkleStr = venuscore.IFileSystem:ReadFile(wrinkle_config)
    local wrinkleNum = {}
    for num in wrinkleStr:gmatch("%S+") do
      table.insert(wrinkleNum, tonumber(num))
    end
    self.wrinkleCoords = mathfunction.vector2array();
    for i=1, #wrinkleNum, 2 do
      self.wrinkleCoords:PushBack(mathfunction.vector2(wrinkleNum[i], wrinkleNum[i+1]));
      end
  end
  
  if wrinkle_tex ~= nil then
    self.wrinkleTex = self:LoadTexture(wrinkle_tex)
  end
  
  if forehead_mask ~= nil then
    self.foreheadMask = self:LoadTexture(forehead_mask);
  end
  
  if ldecree_mask ~= nil then
    self.ldecreeMask = self:LoadTexture(ldecree_mask);
  end
  
  if rdecree_mask ~= nil then
    self.rdecreeMask = self:LoadTexture(rdecree_mask);
  end
end

function liveportrait:LoadResources()  
  if self.windows_platform then
    if self.ml:GetInitStatus() ~= true or self.portrait == nil then 
      return false
    end  
    
    self.ml:Run(self.portrait);
    local landmarks = self.ml.landmarks;
    self.hair_mask = self.ml.hair_mask;
    self:ParseLandmarks(landmarks)    

    self.lowerteeth_path = defined.mesh_dir .. defined.lowerteeth_path
    self.upperteeth_path = defined.mesh_dir .. defined.upperteeth_path
    self.tongue_path = defined.mesh_dir .. defined.tongue_path  
    
    self.morphable_path = "config_face_morphable_3.json"
    self.face_mesh_path = "3dmm_60.bin"
    self.template_path = "config_face_template_3.json"
    self.blendshape_path = "blendshape_41_3.bin"
    self.config_path = "config_track.json"
  else    
    self.portrait = likeapp.AI:GetMuglifeImageTexture(0)      
    local wrinkle_config = likeapp.AI:GetMuglifeResourcePath("wrinkle_coords.config")
    local wrinkle_tex = likeapp.AI:GetMuglifeResourcePath("wrinkle.jpg")
    local forehead_mask = likeapp.AI:GetMuglifeResourcePath("forehead_mask.jpg")
    local ldecree_mask = likeapp.AI:GetMuglifeResourcePath("ldecree_mask.jpg")
    local rdecree_mask = likeapp.AI:GetMuglifeResourcePath("rdecree_mask.jpg")
    
    self:LoadWrinkleResource(wrinkle_config, wrinkle_tex, forehead_mask, ldecree_mask, rdecree_mask);    
    
    self.lowerteeth_path = likeapp.AI:GetMuglifeResourcePath("lower_teeth")
    self.upperteeth_path = likeapp.AI:GetMuglifeResourcePath("upper_teeth")
    self.tongue_path = likeapp.AI:GetMuglifeResourcePath("tongue")
  
    local size = self.portrait:GetSize()
    self.w = size:x()
    self.h = size:y()
    local landmarks = likeapp.AI:GetMuglifeFaceData(0)
    self.hair_mask = likeapp.AI:GetMuglifeMaskTexture(0)

    self:ParseLandmarks(landmarks)    
      
    self.morphable_path = likeapp.AI:GetMuglifeResourcePath("config_face_morphable_3.json")
    self.face_mesh_path = likeapp.AI:GetMuglifeResourcePath("3dmm_60.bin")
    self.template_path = likeapp.AI:GetMuglifeResourcePath("config_face_template_3.json")
    self.blendshape_path = likeapp.AI:GetMuglifeResourcePath("blendshape_41_3.bin")
    self.config_path = likeapp.AI:GetMuglifeResourcePath("config_track.json")
  end
  
  return true
end

function liveportrait:ClearPortraitInfo()
  self.face_lmk = nil
  self.forehead_lmk = nil
  self.angle = nil
  self.lmk_visibility = nil
  self.expression = nil
  self.left_iris_center = nil
  self.right_iris_center = nil
  self.tongue = nil
  self.hair_mask = nil
  self.left_iris = nil
  self.right_iris = nil
end

function liveportrait:SetRender(ts, featurePoints, texCoords, enableProj, headCenter, foreheadCenter, lFaceCenter, lmouthCorner, rfaceCenter, rmouthCorner, triangles)
  -- Not a good place to put..
  self.render:EnableWrinkle(true, self.wrinkleCoords, self.wrinkleTex, self.foreheadMask, self.ldecreeMask, self.rdecreeMask,  self.foreheadWrinkleFactor, self.decreeWrinkleFactor)
  self.render:SetParams(ts, featurePoints, texCoords, enableProj, headCenter, foreheadCenter, lFaceCenter, lmouthCorner, rfaceCenter, rmouthCorner, triangles)
  self.render:UpdateWrinkleTexture(self.face_lmk, self.forehead_lmk)
end

function liveportrait:CheckFace()
  local pitch = self.angle:x() * defined.angleToRadian
  local roll = self.angle:y() * defined.angleToRadian
  local yaw = self.angle:z() * defined.angleToRadian
  if pitch > defined.max_pitch or pitch < -defined.max_pitch then
    LOG("Muglife: Too large pitch angle")
  end
  
  if yaw > defined.max_yaw or yaw < -defined.max_yaw then
    LOG("Muglife: Too large yaw angle")
  end
  
  return true
end

function liveportrait:Update(def)
  
  if self.inited ~= true then
    return
  end
  
  -- init all the parameters
  local weights = {}
  local bs_num = self.face_model:GetBSNumber();
  for i = 1, bs_num do
    weights[i] = 0
  end
  local left_iris_offset = {0, 0}
  local right_iris_offset = {0, 0}
  local angle = {0, 0, 0}
  local scale  = 1
  
  local frame = 1
  if self.windows_platform then
    self.timer = self.timer + def
    local process = (self.timer * 1000 - self.start_time) / self.last_time
    if process < 0 then
      process = 0
    elseif process > 1 then
      process = 1
    end
    frame = process * self.anim.last_frame
    if frame >= self.anim.last_frame then
      self.timer = 0
    end
  else
    local process = (def - self.start_time) / self.last_time
    frame = process * self.anim.last_frame
  end

  if frame < 1 then
    frame = 1
  elseif frame > self.anim.last_frame then
    frame = self.anim.last_frame
  end
  
  local params = self.anim:Play(frame)
  for k, v in pairs(params) do
    if k == self.anim.channel_map["pitch"] then 
      angle[1] = v
    elseif k == self.anim.channel_map["roll"] then
      angle[2] = v
    elseif k == self.anim.channel_map["yaw"] then
      angle[3] = v
    elseif k == self.anim.channel_map["l_iris_x"] then
      left_iris_offset[1] = v
    elseif k == self.anim.channel_map["l_iris_y"] then
      left_iris_offset[2] = v
    elseif k == self.anim.channel_map["r_iris_x"] then
      right_iris_offset[1] = v
    elseif k == self.anim.channel_map["r_iris_y"] then
      right_iris_offset[2] = v
    elseif k == self.anim.channel_map["scale"] then
      scale = v
    else
      weights[k] = v
    end
  end
  
  left_iris_offset = mathfunction.vector2(left_iris_offset[1], left_iris_offset[2])
  right_iris_offset = mathfunction.vector2(right_iris_offset[1], right_iris_offset[2])
  
  if self.face_model:Update(weights, left_iris_offset, right_iris_offset) == false then
    return
  end
  local featurePoints = self.face_model:GetFP3d()
  self.face_model:UpdateModelview(
    mathfunction.vector3(angle[1], angle[2], angle[3]), 
    mathfunction.vector3(0, 0, 0), 
    mathfunction.vector3(scale, scale, scale))
  local modelview = self.face_model:GetModelview()
  local stacked_modelview = self.face_model:GetStackedModelview()
  local upIdx = self.face_model:GetUpperTeethAttach()
  local lowIdx = self.face_model:GetLowerTeethAttach()
  local up = self.face_model:GetFacePoint(upIdx, false)
  local low = self.face_model:GetFacePoint(lowIdx, false)
  
  local headCenterIdx = self.face_model:GetHeadCenterIdx();
  local headCenter = self.face_model:GetFacePoint(headCenterIdx, false)
  local foreheadCenterIdx = self.face_model:GetForeHeadCenterIdx()
  local foreheadCenter = self.face_model:GetFacePoint(foreheadCenterIdx, false) 
  
  local lfaceCenterIdx = self.face_model:GetLeftFaceCenterIdx();  
  local lfaceCenter = self.face_model:GetFacePoint(lfaceCenterIdx, false)   
  local lmouthcornerIdx = self.face_model:GetLeftMouthCornerIdx()
  local leftMouthCorner = self.face_model:GetFacePoint(lmouthcornerIdx, false)
  
  local rfaceCenterIdx = self.face_model:GetRightFaceCenterIdx();   
  local rfaceCenter = self.face_model:GetFacePoint(rfaceCenterIdx, false) 
  local rmouthcornerIdx = self.face_model:GetRightMouthCornerIdx()
  local rightMouthCorner = self.face_model:GetFacePoint(rmouthcornerIdx, false)
  
  self.render:Update(def, featurePoints,  headCenter, foreheadCenter, lfaceCenter, leftMouthCorner, rfaceCenter, rightMouthCorner, modelview)
  self.mouth:Update(up, low, stacked_modelview)
  --LOG("update animation frame.")

end

function liveportrait:IsMLInitDone()
  if self.ml ~= nil then
    return self.ml:GetInitStatus() == true 
  else  
    return false;
  end
end

-- using for mobile
function liveportrait:UpdateMuglifeMaterial()
  self.inited = false

  self:LoadResources();
  
  if self:InitMasquerade(self.morphable_path, self.face_mesh_path, self.template_path, self.blendshape_path, self.config_path) ~= true then
    return false
  end
  
  if self:InitAnimeController(self.anim_path) ~= true then
    return false
  end
  
  if self:CheckFace() ~= true then
    return false
  end
  
  if self.face_model:Fit(
    self.face_lmk, 
    self.forehead_lmk, 
    self.angle, 
    self.lmk_visibility, 
    self.expression,
    self.right_iris_center,
    self.left_iris_center,
    self.tongue) ~= true then
    LOG("Muglife: Face fitting failed")
    return false
  end
  if self.face_model:Triangulation(
    self.hair_mask,
    self.right_iris,
    self.left_iris) ~= true then
    LOG("Muglife: Face Triangulation failed")
    return false
  end
   
   local featurePoints = self.face_model:GetFP3d()
   local texCoords = self.face_model:GetTexCoords()
   local triangles = self.face_model:GetFaceTri()  
   local enableProj = self.face_model:GetEnableProj()
   
   local headCenterIdx = self.face_model:GetHeadCenterIdx()
   local headCenter = self.face_model:GetFacePoint(headCenterIdx, false)
   local foreheadCenterIdx = self.face_model:GetForeHeadCenterIdx()
   local foreheadCenter = self.face_model:GetFacePoint(foreheadCenterIdx, false)
   
   local lfaceCenterIdx = self.face_model:GetLeftFaceCenterIdx()   
   local lfaceCenter = self.face_model:GetFacePoint(lfaceCenterIdx, false)   
   local lmouthcornerIdx = self.face_model:GetLeftMouthCornerIdx()
   local leftMouthCorner = self.face_model:GetFacePoint(lmouthcornerIdx, false)
   
   local rfaceCenterIdx = self.face_model:GetRightFaceCenterIdx()   
   local rfaceCenter = self.face_model:GetFacePoint(rfaceCenterIdx, false) 
   local rmouthcornerIdx = self.face_model:GetRightMouthCornerIdx()
   local rightMouthCorner = self.face_model:GetFacePoint(rmouthcornerIdx, false)
   
  self:ResetRenders()
  
  self.foreheadWrinkleFactor = self.face_model:GetForeheadWrinkleFactor();
  self.decreeWrinkleFactor = self.face_model:GetDecreeWrinkleFactor()
  
  self:SetRender(self.portrait, featurePoints, texCoords, enableProj, headCenter, foreheadCenter,lfaceCenter, leftMouthCorner, rfaceCenter, rightMouthCorner, triangles)  
  self:InitMouth(self.lowerteeth_path, self.upperteeth_path, self.tongue_path)
  
  self.inited = true
  return true
  
end

function liveportrait:SetAnimPath(anim_path)
  self.anim_path = anim_path
end

function liveportrait:ParseLandmarks(rawlandmarks)
  self.face_lmk = {}
  self.forehead_lmk = {}
  local angle = {}
  self.left_iris = {}
  self.right_iris = {}
  self.lmk_visibility = {}
  self.expression = {}
  for i = 1, 128 do
    self.expression[i] = 0
  end
  self.tongue = 0
  
  local landmarks = self:_mapLandmarks(rawlandmarks, false);
  
  for i = 1, defined.landmark_num do
    if i <= 106 then
      table.insert(self.face_lmk, landmarks[i * 2 - 1])
      table.insert(self.face_lmk, landmarks[i * 2])
    elseif i <= 129 then
      table.insert(self.forehead_lmk, landmarks[i * 2 - 1])
      table.insert(self.forehead_lmk, landmarks[i * 2])
    elseif i <= 149 then
      if i == 149 then
        self.right_iris_center = mathfunction.vector2(landmarks[i * 2 - 1], landmarks[i * 2])
      else
        table.insert(self.right_iris, landmarks[i * 2 - 1])
        table.insert(self.right_iris, landmarks[i * 2])
      end
    elseif i <= 169 then
      if i == 169 then
        self.left_iris_center = mathfunction.vector2(landmarks[i * 2 - 1], landmarks[i * 2])
      else
        table.insert(self.left_iris, landmarks[i * 2 - 1])
        table.insert(self.left_iris, landmarks[i * 2])
      end
    elseif i <= 172 then
      table.insert(angle, landmarks[i + 169])
    else
      self.expression[i - 172] = landmarks[i + 169]
    end
  end
  self.angle = mathfunction.vector3(angle[1], angle[2], angle[3])
  
  if self.windows_platform then
    local temp = self.right_iris_center
    self.right_iris_center = self.left_iris_center
    self.left_iris_center = temp
    
    temp = self.right_iris
    self.right_iris = self.left_iris
    self.left_iris = temp
  end
  
end

function liveportrait:SetFboSize(size)
  self.w = size:x()
  self.h = size:y()
  if (self.render ~= nil)
  then 
    self.render:SetFboSize(size); 
  end
end

function liveportrait:GetPortraitTex()
  if (self.render ~= nil)
  then 
    local tex = self.render:GetPortraitTex();
    if tex ~= nil then
      return tex; 
    else
      return self.render:_GenerateTexture(self.portrait)
    end
    
  end
  return self.portrait
end

function liveportrait:_GenerateTexture(ts)
  if self.render ~= nil then
    return self.render:_GenerateTexture(ts)
  end
end

-- Generate texture based on tex path
function liveportrait:LoadTexture(tex_path)
  if (tex_path == nil)
  then
    return;
  end
  
  local texture = apolloengine.TextureEntity();
  texture:PushMetadata(  apolloengine.TextureFileMetadata (
                         apolloengine.TextureEntity.TU_STATIC,
                         apolloengine.TextureEntity.PF_AUTO,
                         1, false,
                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                         apolloengine.TextureEntity.TF_LINEAR,
                         apolloengine.TextureEntity.TF_LINEAR,
                         tex_path));
     
  texture:SetKeepSource(true); 
  --texture:SetJobType(venuscore.IJob.JT_SYNCHRONOUS);   
  texture:CreateResource(); 
  
  return texture;
end

function liveportrait:ResetRenders()
  self.render:Reset()
  self.mouth:Clear()
end


function liveportrait:ReleaseResource()
  self.face_model = nil  
  self.portrait = nil
  self.wrinkleCoords = nil
  self.wrinkleTex = nil
  self.foreheadMask = nil
  self.ldecreeMask = nil
  self.rdecreeMask = nil
  self.bvt_inited = nil
  self.inited = nil
  self.w = nil
  self.h = nil
  self.timer = nil
  self.dir = nil
  self:ClearPortraitInfo()
  --self.ml:Clear()
  self.mouth:Clear()
  self.render:Clear()
  self.render = nil
  self.anim:Clear()
end


function liveportrait:_mapLandmarks(rawLandmarks, is_160_lmk)
  local landmarks = {}
  
  if is_160_lmk == true then
    landmarks = rawLandmarks;
  else 
    local pos = {70, 73, 76, 78, 80, 83, 86, 
      130, 128, 126, 124, 122, 
      87, 90, 95, 100, 103, 116, 111, 106 }
    
    for i = 1, 106 * 2 do
      landmarks[i] = rawLandmarks[i]
    end
  
    for i = 106 * 2 + 1, defined.face_landmark_num * 2 + defined.landmark_num - defined.face_landmark_num do
      local offset = 240 * 2 + i - 106 * 2;
      landmarks[i] = rawLandmarks[offset]
    end
    
    for i = 85, 104 do
      local offset = 107 + pos[i - 85 + 1]
      landmarks[i * 2 - 1] = rawLandmarks[offset * 2 - 1]
      landmarks[i * 2] = rawLandmarks[offset * 2]
    end
  end
  return landmarks;
end

return liveportrait
