--@author      : Zhu Fei
--@date        : 2019-06-27
--@description : lua script for BlingEffect
--@version     : 1.0
--@author      : Zhu Fei
--@date        : 2020-02-25
--@description : modify for integration into kratos
--@version     : 2.0
--@author      : Zhu Fei
--@date        : 2020-03-03
--@description : Bling 2.0 features
--@version     : 3.0

local apolloengine  = require "apolloengine"
local rendernode    = require "apolloutility.apollonode.rendernode"
local mathfunction  = require "mathfunction"
local bling         = require "blingeffect"
local patch_creator = require "blingbling.patch_img_creator"
local venuscore     = require "venuscore"
local defined = require "apolloutility.defiend"

--[[
    Render Blings as billboards on background frame.
    Usage:
    1. Create a BlingEffect object, make sure patch_creator_render_sequence is less than that of the main camera.
    2. call Update() in rendering Update()
]]

local BlingBehavior = venuscore.VenusBehavior:extend("BlingBehavior");
local patch_creator_render_sequence = -4000

function BlingBehavior:new()
    self.texture_path              = "proj:assets/resource/blingbling/resource/bling.png"
    self.second_texture_path       = "proj:assets/resource/blingbling/resource/bling.png"
    self.frame_width               = 400
    self.frame_height              = 720
    self.patch_size                = 5       -- size of patch to get intensity value
    self.max_bling_size            = 110     -- maximum size (in pixel) of the bling
    self.max_bling_num             = 120     -- maximum number of blings that can be generated on one frame
    self.intensity_threshold       = 80      -- a patch must be brighter than this value to generate bling at its center
    self.intensity_delta_threshold = 30      -- a patch must be brighter than its neighbors above this value to generate bling at patch center
    self.neighbor_count_threshold  = 0.8     -- a patch must be brighter than this fraction of its neighbors to generate bling at patch center
    self.bling_texture_num         = 2       -- do not change
    self.second_texture_ratio      = 0       -- percentage of all blings that uses secondary texture
    self.bling_size_curve_order    = 3       -- order of scale curve that scales bling size according to pixel intensity
    self.bling_size_min_scale      = 1.0     -- minimum value of additional scale on bling size
    self.bling_size_max_scale      = 1.0     -- maximum value of additional scale on bling size
    self.bling_duration            = 1       -- number of frames that a bling can live since its creation
    self.instant_bling_ratio       = 1.0     -- of all existing blings, set a ratio of youngest blings to be instant (duration == 1)
    self.manual_blend              = 1       -- do not change
    self.blend_mode                = 1.0     -- which blend mode to use
end

--第一次Update的时候会调用Start
function BlingBehavior:_OnStart()
  self.camera = self.Node:GetComponent(apolloengine.Node.CT_CAMERA);
  --self.camera:SetLayerMaskNothing();
  --self.camera:AddLayerMask("bling_layer");
  self.initialized = false
  local camera_size = self.camera:GetCameraResolution()
  self.frame_width  = camera_size.mx
  self.frame_height = camera_size.my
  self:applyConfig()
end

function BlingBehavior:_OnDestroy()
  self:_Clear(); --删除Script时需要删除动态创建的Node
end

function BlingBehavior:applyConfig()
    self:_Clear();
    
    local blingconfig  = bling.BlingEffectConfig();
    blingconfig:SetFrameWidth(self.frame_width);
    blingconfig:SetFrameHeight(self.frame_height);
    blingconfig:SetPatchSize(self.patch_size);
    blingconfig:SetMaxBlingSize(self.max_bling_size);
    blingconfig:SetMaxBlingNum(self.max_bling_num);
    blingconfig:SetIntensityThreshold(self.intensity_threshold);
    blingconfig:SetIntensityDeltaThreshold(self.intensity_delta_threshold);
    blingconfig:SetNeighborCountThreshold(self.neighbor_count_threshold);
    blingconfig:SetBlingTextureNum(self.bling_texture_num);
    blingconfig:SetSecondTextureRatio(self.second_texture_ratio);
    blingconfig:SetBlingSizeCurveOrder(self.bling_size_curve_order);
    blingconfig:SetBlingSizeMinScale(self.bling_size_min_scale);
    blingconfig:SetBlingSizeMaxScale(self.bling_size_max_scale);
    blingconfig:SetBlingDuration(self.bling_duration);
    blingconfig:SetInstantBlingRatio(self.instant_bling_ratio);
    blingconfig:SetManualBlend(self.manual_blend);

    self.patch_creator = patch_creator(blingconfig:GetFrameWidth(), blingconfig:GetFrameHeight(), blingconfig:GetPatchSize(), patch_creator_render_sequence);
    self.bling_creator = bling.BlingEffect(blingconfig);
    
    self.rendernode    = rendernode();
    local layerMask = self.camera.LayerMask;
    self.rendernode:SetLayer(layerMask);
	
	--不需要设置sequence了，有Editor标记来区分
	--self.rendernode:SetIsEditor(true);
    if _KRATOSEDITOR then
      local sequnce = self.camera:GetSequence();
      self.rendernode:SetSequence(sequnce);
    else
      local resType = self.camera:GetResourceType();
      self.rendernode:SetResourceType(resType);
      --self.rendernode:SetIsEditor(true);
    end
    --local sequnce = self.camera:GetSequence();
    --self.rendernode:SetSequence(sequnce);
    --self.camera:SetLayerMaskNothing();
    --self.camera:AddLayerMask("bling_layer");
    --TODO: set camera sequence to rendernode

    -- create shader parameter slot
    apolloengine.ShaderEntity.POINT_TEXTURE        = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "POINT_TEXTURE");
    apolloengine.ShaderEntity.SECOND_POINT_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "SECOND_POINT_TEXTURE");
    apolloengine.ShaderEntity.FRAME_TEXTURE        = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "FRAME_TEXTURE");
    apolloengine.ShaderEntity.BLEND_MODE           = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "BLEND_MODE");

    self.rendernode:SetName("bling stream node");
    -- load bling texture
    local texturePath = venuscore.IFileSystem:PathAssembly(self.texture_path)
    self.rendernode:PushTextureMetadata(apolloengine.ShaderEntity.POINT_TEXTURE, texturePath);
    local secondTexturePath = venuscore.IFileSystem:PathAssembly(self.second_texture_path)
    self.rendernode:PushTextureMetadata(apolloengine.ShaderEntity.SECOND_POINT_TEXTURE, secondTexturePath);

    -- initialize render node, render with main camera
    self.dummy_stream = apolloengine.VertexStream();
    self.dummy_stream:ReserveBuffer(blingconfig:GetMaxBlingNum() * 6);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.dummy_stream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE2,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    apolloengine.VertexBufferEntity.DT_FLOAT,
                                    2);
    self.rendernode:SetShow(false);
    self.rendernode:CreateResource(venuscore.IFileSystem:PathAssembly("docs:blingbling/material/render_bling_v2.material"),
                                   apolloengine.RenderComponent.RM_TRIANGLES,
                                   self.dummy_stream);
    self.rendernode:SetParameter(apolloengine.ShaderEntity.BLEND_MODE, mathfunction.vector1(self.blend_mode));

    self.CameraInput = apolloengine.TextureEntity();
    self.CameraInput:PushMetadata(
      apolloengine.TextureReferenceMetadata(
         apolloengine.DeviceResource.DEVICE_CAPTURE));
    self.CameraInput:CreateResource();
    self.initialized = true
    return true
end

function BlingBehavior:_OnUpdate()
    if not self.initialized then
        return
    end
    local layerMask = self.camera.LayerMask;
    --self.rendernode:SetLayer(layerMask);
    --detect resolution change
    local camera_size = self.camera:GetCameraResolution()
    if camera_size.mx ~= self.frame_width and camera_size.my ~= self.frame_height then
        self.frame_width  = camera_size.mx
        self.frame_height = camera_size.my
        self:applyConfig()
    end
    --TODO: get previous texture instead of input texture??
    local patch_image   = self.patch_creator:CreatePatchImage(self.CameraInput);
    local bling_stream  = self.bling_creator:GetBlingBillboard(patch_image);
    if bling_stream then
        self.rendernode:SetShow(true);
        -- update to render
        self.rendernode:SetParameter(apolloengine.ShaderEntity.FRAME_TEXTURE, self.CameraInput);
        self.rendernode.render:ChangeVertexBuffer(bling_stream);
        local vert_num = self.bling_creator:GetBlingNum() * 6;
        self.rendernode:SetDrawCount(vert_num);
    else
        self.rendernode:SetShow(false);
    end
end

--编辑器回调属性Get、Set
function BlingBehavior:GetTexturePath()
  return self.texture_path;
end

function BlingBehavior:SetTexturePath(value)
  self.texture_path = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetSecondTexturePath()
  return self.second_texture_path;
end

function BlingBehavior:SetSecondTexturePath(value)
  self.second_texture_path = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetFrameWidth()
  return self.frame_width;
end

function BlingBehavior:SetFrameWidth(value)
  self.frame_width = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetFrameHeight()
  return self.frame_height;
end

function BlingBehavior:SetFrameHeight(value)
  self.frame_height = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetPatchSize()
  return self.patch_size;
end

function BlingBehavior:SetPatchSize(value)
  self.patch_size = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetMaxBlingSize()
  return self.max_bling_size;
end

function BlingBehavior:SetMaxBlingSize(value)
  self.max_bling_size = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetMaxBlingNum()
  return self.max_bling_num;
end

function BlingBehavior:SetMaxBlingNum(value)
  self.max_bling_num = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetIntensityThreshold()
  return self.intensity_threshold;
end

function BlingBehavior:SetIntensityThreshold(value)
  self.intensity_threshold = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetIntensityDeltaThreshold()
  return self.intensity_delta_threshold;
end

function BlingBehavior:SetIntensityDeltaThreshold(value)
  self.intensity_delta_threshold = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetNeighborCountThreshold()
  return self.neighbor_count_threshold;
end

function BlingBehavior:SetNeighborCountThreshold(value)
  self.neighbor_count_threshold = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetBlingTextureNum()
  return self.bling_texture_num;
end

function BlingBehavior:SetBlingTextureNum(value)
  self.bling_texture_num = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetSecondTextureRatio()
  return self.second_texture_ratio;
end

function BlingBehavior:SetSecondTextureRatio(value)
  self.second_texture_ratio = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetBlingSizeCurveOrder()
  return self.bling_size_curve_order;
end

function BlingBehavior:SetBlingSizeCurveOrder(value)
  if value >= 0 then
    self.bling_size_curve_order = value;
    if self.initialized then
      self:applyConfig();
    end
  end
end

function BlingBehavior:GetBlingSizeMinScale()
  return self.bling_size_min_scale;
end

function BlingBehavior:SetBlingSizeMinScale(value)
  if value <= self.bling_size_max_scale then
    self.bling_size_min_scale = value;
    if self.initialized then
      self:applyConfig();
    end
  end
end

function BlingBehavior:GetBlingSizeMaxScale()
  return self.bling_size_max_scale;
end

function BlingBehavior:SetBlingSizeMaxScale(value)
  if value >= self.bling_size_min_scale then
    self.bling_size_max_scale = value;
    if self.initialized then
      self:applyConfig();
    end
  end
end

function BlingBehavior:GetBlingDuration()
  return self.bling_duration;
end

function BlingBehavior:SetBlingDuration(value)
  self.bling_duration = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetInstantBlingRatio()
  return self.instant_bling_ratio;
end

function BlingBehavior:SetInstantBlingRatio(value)
  self.instant_bling_ratio = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:GetBlendMode()
  return self.blend_mode;
end

function BlingBehavior:SetBlendMode(value)
  self.blend_mode = value;
  if self.initialized then
    self:applyConfig();
  end
end

function BlingBehavior:_Clear()
  if self.rendernode then
    local scene = apolloengine.SceneManager:GetOrCreateScene(defined.default_scene_name);
    --scene:DeleteNode(self.rendernode.node:GetStaticID());
    scene:DeleteNode(self.rendernode.node);
    self.rendernode = nil;
    self.patch_creator:Clear();
  end
end


BlingBehavior:MemberRegister("texture_path",
  venuscore.ScriptTypes.FilePathType(
    apolloengine.TextureEntity:RTTI(),
    BlingBehavior.GetTexturePath,
    BlingBehavior.SetTexturePath
));

BlingBehavior:MemberRegister("second_texture_path",
  venuscore.ScriptTypes.FilePathType(
    apolloengine.TextureEntity:RTTI(),
    BlingBehavior.GetSecondTexturePath,
    BlingBehavior.SetSecondTexturePath
));

BlingBehavior:MemberRegister("patch_size",
  venuscore.ScriptTypes.IntType(
    1, 1024,
    BlingBehavior.GetPatchSize,
    BlingBehavior.SetPatchSize
));

BlingBehavior:MemberRegister("max_bling_size",
  venuscore.ScriptTypes.IntType(
    1, 1024,
    BlingBehavior.GetMaxBlingSize,
    BlingBehavior.SetMaxBlingSize
));

BlingBehavior:MemberRegister("max_bling_num",
  venuscore.ScriptTypes.IntType(
    0, 1000,
    BlingBehavior.GetMaxBlingNum,
    BlingBehavior.SetMaxBlingNum
));

BlingBehavior:MemberRegister("intensity_threshold",
  venuscore.ScriptTypes.IntType(
    0, 255,
    BlingBehavior.GetIntensityThreshold,
    BlingBehavior.SetIntensityThreshold
));

BlingBehavior:MemberRegister("intensity_delta_threshold",
  venuscore.ScriptTypes.IntType(
    0, 255,
    BlingBehavior.GetIntensityDeltaThreshold,
    BlingBehavior.SetIntensityDeltaThreshold
));

BlingBehavior:MemberRegister("neighbor_count_threshold",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    BlingBehavior.GetNeighborCountThreshold,
    BlingBehavior.SetNeighborCountThreshold
));

BlingBehavior:MemberRegister("second_texture_ratio",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    BlingBehavior.GetSecondTextureRatio,
    BlingBehavior.SetSecondTextureRatio
));

BlingBehavior:MemberRegister("bling_size_curve_order",
  venuscore.ScriptTypes.FloatType(
    0, 10,
    BlingBehavior.GetBlingSizeCurveOrder,
    BlingBehavior.SetBlingSizeCurveOrder
));

BlingBehavior:MemberRegister("bling_size_min_scale",
  venuscore.ScriptTypes.FloatType(
    0, 10,
    BlingBehavior.GetBlingSizeMinScale,
    BlingBehavior.SetBlingSizeMinScale
));

BlingBehavior:MemberRegister("bling_size_max_scale",
  venuscore.ScriptTypes.FloatType(
    0, 10,
    BlingBehavior.GetBlingSizeMaxScale,
    BlingBehavior.SetBlingSizeMaxScale
));

BlingBehavior:MemberRegister("bling_duration",
  venuscore.ScriptTypes.IntType(
    1, 100,
    BlingBehavior.GetBlingDuration,
    BlingBehavior.SetBlingDuration
));

BlingBehavior:MemberRegister("instant_bling_ratio",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    BlingBehavior.GetInstantBlingRatio,
    BlingBehavior.SetInstantBlingRatio
));

BlingBehavior:MemberRegister("blend_mode",
  venuscore.ScriptTypes.ComboType(
    {
      {
        key = "Normal",
        value = 1.0
      },
      {
        key = "Add",
        value = 2.0
      },
      {
        key = "Lighten",
        value = 3.0
      },
      {
        key = "Multiply",
        value = 4.0
      },
      {
        key = "Overlay",
        value = 5.0
      },
      {
        key = "Screen",
        value = 6.0
      },
      {
        key = "Lighter",
        value = 7.0
      }
    },
    BlingBehavior.GetBlendMode,
    BlingBehavior.SetBlendMode
));

return BlingBehavior;