local Types = require "venuscore.rtti.types"
local BlueSelf = require "bluecore.blueself"
local BlueprintSelf = require "bluecore.core.blueprint_self"
local BlueDefined = require "bluecore.bluedefined"
local AE = require "apolloengine"
local venuscore = require "venuscore"
local venusjson = require "venusjson"

local BlueUtility = {};
BlueUtility.nodeTypeNameList = {"Event", "Member", "Static Func", "Comment", "No ExecFunc", "Variable", "Ref_Component", "Ref_Node"};
BlueUtility.notSupportRttiCopyList = 
{
  Types.PathType,
  Types.DirectoryFileFilterType,
  AE.TextureEntity:RTTI(),
  AE.MaterialEntity:RTTI()
}


function BlueUtility:IsDataPin(pin)
  if pin.pinType == BlueDefined.PIN_DATA_INPUT  or pin.pinType == BlueDefined.PIN_DATA_OUTPUT then
    return true;
  else
    return false;
  end
end


function BlueUtility:IsExecPin(pin)
  if pin.pinType == BlueDefined.PIN_EXEC_INPUT or pin.pinType == BlueDefined.PIN_EXEC_OUTPUT then
    return true;
  else
    return false;
  end
end

function BlueUtility:IsInputPin(pin)
  if pin.pinType == BlueDefined.PIN_EXEC_INPUT or pin.pinType == BlueDefined.PIN_DATA_INPUT then
    return true;
  else
    return false;
  end
end

function BlueUtility:IsOutputPin(pin)
  if pin.pinType == BlueDefined.PIN_EXEC_OUTPUT or pin.pinType == BlueDefined.PIN_DATA_OUTPUT then
    return true;
  else
    return false;
  end
end

function BlueUtility:IsPinSameDirection(pinA, pinB)
  local input = self:IsInputPin(pinA) and self:IsInputPin(pinB);
  local output = self:IsOutputPin(pinA) and self:IsOutputPin(pinB);
  if input or output then
    return true;
  else
    return false;
  end
end

function BlueUtility:IsPinSameFlow(pinA, pinB)
  local input = self:IsDataPin(pinA) and self:IsDataPin(pinB);
  local output = self:IsExecPin(pinA) and self:IsExecPin(pinB);
  if input or output then
    return true;
  else
    return false;
  end
end

function BlueUtility:isBlueSelf(value)
  return type(value) == 'table' and value.GetTypeName and  value:RTTI() == BlueSelf
end

function BlueUtility:isBlueprintSelf(value)
  return type(value) == 'table' and value.GetTypeName and  value:RTTI() == BlueprintSelf
end

function BlueUtility:isNone(value)
  return type(value) == 'table' and value.GetTypeName and  value:RTTI() == Types.NoneType
end

function BlueUtility:isNoneNull(value)
  return type(value) == 'table' and value.GetTypeName and  value:RTTI() == Types.NotNullType
end


-- 给定rtti 返回对应rtti的数组类型
-- 由于不支持二维数组,所以传入数组rtti返回就是传入参数rtti
function BlueUtility:GetArrayRtti(rtti)
  if rtti:isArray() then
    return rtti
  end
  if rtti == Types.AnyType then -- array of AnyType is AnyArrayType
    return Types.AnyArrayType
  end
  local rttiOfArray = Types.AnyArrayType:extend(rtti);
  -- 同样一个rtti 每次GetArrayRtti 都会返回不同的 rttiOfArray, 判断要通过isType/isTypeOrDriverType
  return rttiOfArray;
end



function BlueUtility:_GetBlueRtti()
  self._BlueRtti = require "bluecore.core.blue_rtti"
  self._GetBlueRtti = function(self) -- 替换函数
    return self._BlueRtti
  end
  return self._BlueRtti;
end


-- 基本类型(float bool string int vec1 vec2 vec3 vec4 quaternion) 返回新实例
-- 复杂类型(e.g RenderComponent) 返回 nil(None) 对于数组反射面板,如果是数组 table.insert(tbl, nil)相当于没有操作 所以这里用None代替nil
-- Lua Object类型(包含AnyType) 返回 Lua Objectd新实例
function BlueUtility:GetDefaultByRtti(rtti)
  return self:_GetBlueRtti().GetDefault(rtti)
end

-- default~=nil 生成字面值  default=nil 生成默认值
-- 基本类型(float bool string int vec1 vec2 vec3 vec4 quaternion) 返回新实例,拷贝原来数据
-- 复杂类型(e.g RenderComponent) 返回 nil ?还是返回None?
-- 任意类型(AnyType) 返回AnyType实例
-- 数组类型(AnyArrayType TextureEntity[]) 不创建TextureEntity实例数组作为字面值
-- Self类型 返回Self实例
function BlueUtility:GenerateLiteral(rtti, default)
  if _KRATOSEDITOR then LOG("[GenerateLiteral] "..tostring(rtti:GetTypeName())..","..tostring(default))  end
  return self:_GetBlueRtti().GetLiteral(rtti, default)
end


function BlueUtility:GetNodeTypeName(node)
  return self.nodeTypeNameList[node.nodeType]
end


function BlueUtility:GetDevResourceType(filename)
  local resourcelist = AE.DeviceResource:GetDeviceResources(AE.TextureEntity:RTTI())
  if resourcelist == nil then
    return nil
  end
  for key,value in pairs(resourcelist) do
    if value == filename then
      return "refs";
    end
  end
  return nil;
end

function BlueUtility:ConverToRelativePath(absolutepath,targetPath)
  local relativepath = absolutepath;
  local replacestr = targetPath or "proj:";
  local projectpath = venuscore.IFileSystem:PathAssembly(replacestr);
  local _, sEnd = string.find(absolutepath,projectpath,1,true);
  if sEnd ~= nil then
    relativepath = string.sub(absolutepath,sEnd + 1);
    relativepath = replacestr .. relativepath;
  end
  return relativepath;
end

function BlueUtility:CreateEntity(targetPath, extension,
                                  textureType,
                                  usage,
                                  pixelFormat,
                                  anisotropic,
                                  isMipMap, mipmapLevel,
                                  swarp, twarp, magfilter, minfilter)
  local entity = nil;
  if extension == "refs" then
    entity= AE.TextureEntity();
    entity:PushMetadata(AE.TextureReferenceMetadata(targetPath));
  elseif extension == "fbo" then
    entity = AE.RenderTargetEntity();
    entity:PushMetadata(AE.RenderTargetFileMetadata(targetPath));
  elseif extension == "hdr" then
    local metadatas = self:CreateCubemapEntity(targetPath)
    entity = AE.TextureEntity()
    for i = 1, #metadatas do
      entity:PushMetadata(metadatas[i])
    end
  else
    entity= AE.TextureEntity();
    local setup = false
    -- 如果不给定纹理参数,尝试从文件读取
    if textureType == nil then
      local metaPath = self:GetTextureMeta(targetPath);
      -- proj 目录可能被修改
      --local rMetaPath = self:ConverToRelativePath(metaPath);
      if metaPath == nil then
        LOG("missing meta file for " ..tostring(targetPath)..", using default")
        --metaPath =  self:GetTextureMetaCreated(targetPath); -- 避免运行时候生成文件
      else
        entity:PushMetadata(AE.TextureDescribeFileMetadata(metaPath));
        setup = true ;
      end

    end

    if not setup then
      -- false,nil -> false
      -- other,0 -> true
      entity:PushMetadata(AE.TextureFileMetadata(
              textureType ~= nil and textureType or AE.TextureEntity.TT_TEXTURE2D,
              usage ~= nil and usage or AE.TextureEntity.TU_STATIC,
              pixelFormat ~= nil and pixelFormat or AE.TextureEntity.PF_AUTO,
              anisotropic ~= nil and anisotropic or 1 ,
              isMipMap ~= nil and isMipMap or false,
              mipmapLevel ~= nil and mipmapLevel or 0,
              swarp ~= nil and swarp or AE.TextureEntity.TW_REPEAT,
              twarp ~= nil and twarp or AE.TextureEntity.TW_REPEAT,
              magfilter ~= nil and magfilter or AE.TextureEntity.TF_NEAREST,
              minfilter ~= nil and minfilter or AE.TextureEntity.TF_NEAREST,
              targetPath));
    end
  end

  assert(entity ~= nil, "[CreateEntity] entity == nil ")

  entity:SetJobType(venuscore.IJob.JT_SYNCHRONOUS);
  entity:CreateResource();

  return entity;
end

function BlueUtility:GetTextureMeta(imagePath)
  local metaPath = venuscore.IFileSystem:PathAssembly(imagePath) .. ".meta";
  return venuscore.IFileSystem:isFileExist(metaPath) and metaPath or nil
end

function BlueUtility:GetTextureMetaCreated(imagePath)
 
  local metaPath = venuscore.IFileSystem:PathAssembly(imagePath) .. ".meta";
  if not venuscore.IFileSystem:isFileExist(metaPath) then --查看是否存在png的meta文件 不存在就新建一个meta
    local defaultTextureSettings = 
    {
      Version = 1,
      TextureType = 2,
      TextureUseage = 0,
      PixelFormat = -1,
      IsMipMap = false,
      MipMapLevel = 0,
      MaxAnisotropy = 1,
      SWrap = 1,
      TWrap = 1,
      MagFilter = 1,
      MinFilter = 1,
      IsKeepSource = false,
    };
    --local textureInfo = table.deepcopy(defaultTextureSettings);
    defaultTextureSettings["TexturePath"] = self:ConverToRelativePath(imagePath);
    venusjson.SaveJsonObject(defaultTextureSettings, metaPath); --写成meta
  end
  return metaPath; 
end

function BlueUtility:ReadFileInfo(path)
  return venusjson.LaodJsonFile(path); --返回table
end

function BlueUtility:GetMetaData(path)
  path = venuscore.IFileSystem:PathAssembly(path)
  local metapath = path .. ".meta";
  return self:LoadMetaFile(metapath);
end

function BlueUtility:LoadMetaFile(metaPath)
  local metadata = nil;
  if venuscore.IFileSystem:isFileExist(metaPath) then
    metadata = self:ReadFileInfo(metaPath);
  else
    ERROR("LoadMetaFile "..tostring(metaPath).." not exists");
  end
  return metadata;
end


--FIXME(hjh) 没有hdr素材 未经过测试
function BlueUtility:CreateCubemapMeta(targetPath)
  local ae = AE
  local meta = self:GetMetaData(targetPath)
  if meta == nil or meta.fileType == nil then
    return nil
  end
  if meta.fileType ~= "CubemapSource" then
    return nil
  end
  local prefix = "out"
  local faces = {"posz", "negz", "posy", "negy", "negx", "posx"}
  local sizes = {"256x256", "128x128", "64x64", "32x32", "16x16", "8x8", "4x4"}
  local textureTypes = {
      ae.TextureEntity.TT_TEXTURECUBE_FRONT,
      ae.TextureEntity.TT_TEXTURECUBE_BACK,
      ae.TextureEntity.TT_TEXTURECUBE_TOP,
      ae.TextureEntity.TT_TEXTURECUBE_BOTTOM,
      ae.TextureEntity.TT_TEXTURECUBE_LEFT,
      ae.TextureEntity.TT_TEXTURECUBE_RIGHT,
  }
  local mipMaps = {0,1,2,3,4,5,6}
  local metadatas = {}
  if meta.convolveType == 0 then
    for i = 1, #faces do
      local facePath = string.format("%s/%s_%s.tga", meta.targetPath, prefix, faces[i])
      local metadata = ae.TextureFileMetadata(textureTypes[i], mipMaps[1], facePath)
      table.insert(metadatas, metadata)
    end
  elseif meta.convolveType == 1 then
    for j = 1, #sizes do
      for i = 1, #faces do
        local facePath = string.format("%s/%s_%s_%d_%s.tga", meta.targetPath, prefix, faces[i], mipMaps[j], sizes[j])
        local metadata = ae.TextureFileMetadata(textureTypes[i], mipMaps[j], facePath)
        table.insert(metadatas, metadata)
      end
    end
  else
    ERROR("unsupported convolveType: " .. tostring(meta.convolveType))
    return nil
  end
  return metadatas
end


function BlueUtility:PinTypeToString(pinType)
  if pinType == BlueDefined.PIN_DATA_INPUT then
    return "dat_in"
  end
  if pinType == BlueDefined.PIN_DATA_OUTPUT then
    return "dat_out"
  end
  if pinType == BlueDefined.PIN_EXEC_INPUT then
    return "exe_in"
  end
  if pinType == BlueDefined.PIN_EXEC_OUTPUT then
    return "exe_out"
  end
  return 'n/a'
end

-- 上下文菜单相关
function BlueUtility:SetClassifyNode(pinInfos, direction, info, subrttiToNode)

  if pinInfos ~= nil then
    for _, pinInfo in pairs(pinInfos) do
      local rttiName = pinInfo.rtti:GetTypeName()
      subrttiToNode[direction] = subrttiToNode[direction] or {};
      subrttiToNode[direction][rttiName] = subrttiToNode[direction][rttiName] or {};
      local isDuplicate = false;
      --防止重复
      for _,v in pairs(subrttiToNode[direction][rttiName]) do
        if v.blueNodeName == info.blueNodeName then
          isDuplicate = true;
        end
      end
      if not isDuplicate then
        --ERROR("add info "..tostring(info.blueNodeName));
        table.insert(subrttiToNode[direction][rttiName], info);
      end
    end
  end

end

-- 上下文菜单相关
-- classifyCatalog =
-- {
--     BD.BLUE_MGR_INPUT = {...}
--     BD.BLUE_MGR_OUTPUT = {...}
--     BD.BLUE_MGR_DIR = {...}
-- }
function BlueUtility:RemoveClassifyNodeInCatalog(clazz, classifyCatalog)

  local stack = {}
  local stackInfoRoot = {
    items = classifyCatalog,
  }
  table.insert(stack, stackInfoRoot)

  while #stack > 0 do
    local stackInfo = table.remove(stack)
    local items = stackInfo.items

    if items[BlueDefined.BLUE_MGR_INPUT] ~= nil  then
      for rttiName, ctxInfos in pairs(items[BlueDefined.BLUE_MGR_INPUT]) do
        for key, ctxInfo in pairs(ctxInfos) do
          if type(ctxInfo.luaPath)=='table' and ctxInfo.luaPath == clazz then
            ctxInfos[key] = nil
            break; -- 一个 NodeClass 在同一个方向 同一个rtti下之后一个
          end
        end
      end
    end

    if items[BlueDefined.BLUE_MGR_OUTPUT] ~= nil  then
      for rttiName, ctxInfos in pairs(items[BlueDefined.BLUE_MGR_OUTPUT]) do
        for key, ctxInfo in pairs(ctxInfos) do
          if type(ctxInfo.luaPath)=='table' and ctxInfo.luaPath == clazz then
            ctxInfos[key] = nil
            break;
          end
        end
      end
    end

    if items[BlueDefined.BLUE_MGR_DIR] ~= nil then -- 有目录
      for dirName2, items2 in pairs(items[BlueDefined.BLUE_MGR_DIR]) do
        local stackInfo2 = {
          items = items2,
        }
        table.insert(stack, stackInfo2)
      end
    end
  end

end


function BlueUtility.GetIns(comp)
  local t = comp.Instances;
  if t == nil then return end ;
  local _, ins = next(t)
  return ins
end


return BlueUtility;