local BlueNode = require "bluecore.bluenode"
local Types = require "venuscore.rtti.types"
local BD = require "bluecore.bluedefined"
local AE = require "apolloengine"
local BU = require "bluecore.blueutility"
local BluePinInfo = require "bluecore.bluepininfo"
local BlueRtti = require "bluecore.core.blue_rtti"
local BlueFunction = require "bluecore.bluefunction"

local GetBlueprintsOfClass = BlueNode:extend();

local INVALID_CLASS_ID = Types.BlueprintType.INVALID_CLASS_ID
local INVALID_BLUEPRINT_TYPE = Types.BlueprintType.INVALID_BLUEPRINT_TYPE

function GetBlueprintsOfClass:new(...)
  GetBlueprintsOfClass.super.new(self, ...)
  self._classObject = setmetatable({}, { __mode = 'v' })
end

function GetBlueprintsOfClass:_OnCreateWithEditor()
  self:_SetupInstanceInfoTable();
end

function GetBlueprintsOfClass:_OnDeserializePost()
  self:_SetupInstanceInfoTable();
end

function GetBlueprintsOfClass:_SetupInstanceInfoTable()

  local staticDataInputsInfo = self.infoTable[BD.PIN_DATA_INPUT]
  local staticDataOutputsInfo = self.infoTable[BD.PIN_DATA_OUTPUT]
  local staticExecInputsInfo = self.infoTable[BD.PIN_EXEC_INPUT]
  local staticExecOutputsInfo = self.infoTable[BD.PIN_EXEC_OUTPUT]

  self.infoTable = {}
  self.infoTable[BD.PIN_DATA_INPUT] = staticDataInputsInfo
  self.infoTable[BD.PIN_DATA_OUTPUT] = {}
  self.infoTable[BD.PIN_EXEC_INPUT]  = staticExecInputsInfo
  self.infoTable[BD.PIN_EXEC_OUTPUT] = staticExecOutputsInfo

  -- change output 1 info
  local info = staticDataOutputsInfo[1]
  self.infoTable[BD.PIN_DATA_OUTPUT][1] = BluePinInfo(BD.PIN_DATA_OUTPUT, info.argId, info.rtti, info.name, info.tips);

  self:_SetupOutput();
end

function GetBlueprintsOfClass:_OnUpdate(args)

end

function GetBlueprintsOfClass:GetOutputByIndex()
  return  self._classObject ; -- maybe empty table
end

-- 不支持动态场景,只支持静态场景(场景树不变)
-- 加载素材会增加耗时
function GetBlueprintsOfClass:_OnCompile()
  self.compileResult = nil
  local targetClsId = self.inputs[1].literal
  if targetClsId == INVALID_CLASS_ID then
    self.compileResult = "Blueprint is invalid"
    return false
  end

  self._classObject = setmetatable({}, { __mode = 'v' })

  local blueprint = nil

  -- BlueFunction
  if self.graph:isType(BlueFunction) then
    -- blueftnlib's function
    if self.graph:IsLibFunction() then
      self.compileResult = "library function is NOT support"
      return false
    -- blueprint's function
    else
      blueprint = self.graph:GetBlueprint();
    end
  -- Blueprint
  else
    blueprint = self.graph
  end

  local scene = blueprint.Node:GetHostScene();
  local rootNode = scene:GetRootNode();
  self:_IterateNode(rootNode, targetClsId);

  return true
end

function GetBlueprintsOfClass:_IterateNode(node, targetClsId)
  --node如果是普通节点/Prefab Top节点，则继续往下遍历
  local isPrefabSubNode = node.PrefabPath ~= node.HostPrefabPath;
  if not isPrefabSubNode then
    local blueprintComp = node:GetComponent(AE.Node.CT_BLUEPRINT);
    if blueprintComp ~= nil then
      local blueprint = BU.GetIns(blueprintComp);
      if blueprint ~= nil then
        local clsId = blueprint:GetClassId();
        if clsId == targetClsId then
          table.insert(self._classObject, blueprint);
        end
      end
    end
    local childNodes = node:GetChildrens();
    for _, node in pairs(childNodes) do
      self:_IterateNode(node, targetClsId);
    end
  end
end


function GetBlueprintsOfClass:SyncEd()
  self:_SetupOutput();
end

function GetBlueprintsOfClass:_SetupOutput()
  local output1Info = self.infoTable[BD.PIN_DATA_OUTPUT][1]
  local classId = self.inputs[1].literal
  if classId ~= INVALID_CLASS_ID then
    local blueprintType = BlueRtti.GetRttiByClsId(classId);
    local arrayType = BU:GetArrayRtti(blueprintType);
    output1Info:ChangeRtti(arrayType);
  end
end


local DynamicComboType = Types.ComboType:extend();
-- override GetComboData
function DynamicComboType:GetComboData()
  local allClassCombo = BlueRtti.GetComboDataOfClass()
  return allClassCombo;
end


GetBlueprintsOfClass:RegisterInput(1, DynamicComboType , "class", "Select Blueprint Class", INVALID_CLASS_ID);
GetBlueprintsOfClass:RegisterOutput(1, INVALID_BLUEPRINT_TYPE, "blueprints", "all blueprint instance of given class")

GetBlueprintsOfClass:RegisterExecInput(1, "exec", "exec");
GetBlueprintsOfClass:RegisterExecOutput(1, "exec", "exec");
GetBlueprintsOfClass:SetFunctionName("Get All Blueprint Of Class");

return GetBlueprintsOfClass;

