local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local venuscore = require "venuscore"

require "utility"

local bloomEffect = {}

bloomEffect.light_prefilterRO = 0;
bloomEffect.light_prefilterMA = 0;
bloomEffect.down_sampleRO = 0;
bloomEffect.down_sampleMA = 0;
bloomEffect.up_sampleRO = 0;
bloomEffect.up_sampleMA = 0;
bloomEffect.accumulationRO = 0;
bloomEffect.accumulationMA = 0;

bloomEffect.intensity = 0;
bloomEffect.threshold = 0;
bloomEffect.softthreshold = 0;
bloomEffect.Queue = 100;
-- Iterative sampling times
bloomEffect.iterations = 3;
-- sampling scale factor
-- eg: factor = 2 means that the result of each down sampling is 1/2 of the input
bloomEffect.factor = 2;

function bloomEffect:Initialize(host, size)
  self.fbosize            = mathfunction.vector2(size:x(), size:y());

  self.BLOOM_INTENSITY    = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "BLOOM_INTENSITY");
  self.BLOOM_THRESHOLD    = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "BLOOM_THRESHOLD");
  self.SOFT_THRESHOLD     = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "SOFT_THRESHOLD");
  self.TEXELSIZE          = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "TEXELSIZE");
  self.TEXTURE_SRC_A      = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "TEXTURE_SRC_A");
  self.TEXTURE_BLOOM      = apolloengine.IMaterialSystem:NewParameterSlot(
      apolloengine.ShaderEntity.UNIFORM,
      "TEXTURE_BLOOM");

  self.light_prefilterMA  = host:CreateMaterial( "comm:documents/advanced_bloom/light_prefilter.material" );
  self.light_prefilterRO  = host:CreateRenderObject();
  self.down_sampleMA      = host:CreateMaterial( "comm:documents/advanced_bloom/down_sample.material" );
  self.down_sampleRO      = host:CreateRenderObject();
  self.up_sampleMA        = host:CreateMaterial( "comm:documents/advanced_bloom/up_sample.material" );
  self.up_sampleRO        = host:CreateRenderObject();
  self.up_sample_acc_MA   = host:CreateMaterial( "comm:documents/advanced_bloom/up_sample_acc.material" );
  self.up_sample_acc_RO   = host:CreateRenderObject();
  self.accumulationMA     = host:CreateMaterial( "comm:documents/advanced_bloom/accumulation.material" );
  self.accumulationRO     = host:CreateRenderObject();

  --default intensity
  self.intensity          = mathfunction.vector1(0.2);
  self.threshold          = mathfunction.vector1(1.0);
  self.softthreshold      = mathfunction.vector1(1.0);

  self.RTa = {}
  self.RTb = {}

  --RTa 1, 2, 3, 4, 5
  local scale = 1.0;
  for i = 1, self.iterations do
    scale = scale / self.factor;
    self.RTa[i] = host:CreateRenderTarget(
    apolloengine.RenderTargetMetadata(
        apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        mathfunction.vector4(self.fbosize:x()*scale,self.fbosize:y()*scale,0,0),
        self.fbosize*scale,
        mathfunction.vector2(scale,scale)),
    apolloengine.TextureRenderMetadata(
      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
      self.fbosize*scale,
      apolloengine.TextureEntity.PF_RG11B10FLOAT));
  end

  --RTb 1, 2, 3, 4, 5
  scale = 1.0;
  for i = 1, self.iterations - 1 do
    scale = scale / self.factor;
    self.RTb[i] = host:CreateRenderTarget(
    apolloengine.RenderTargetMetadata(
        apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        mathfunction.vector4(self.fbosize:x()*scale,self.fbosize:y()*scale,0,0),
        self.fbosize*scale,
        mathfunction.vector2(scale,scale)),
    apolloengine.TextureRenderMetadata(
      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
      self.fbosize*scale,
      apolloengine.TextureEntity.PF_RG11B10FLOAT));
  end

  --final RT
  scale = 1.0;
  self.RT0 = host:CreateRenderTarget(
    apolloengine.RenderTargetMetadata(
        apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        mathfunction.vector4(self.fbosize:x()*scale,self.fbosize:y()*scale,0,0),
        self.fbosize*scale,
        mathfunction.vector2(scale,scale)),
    apolloengine.TextureRenderMetadata(
      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
      self.fbosize*scale,
      apolloengine.TextureEntity.PF_RG11B10FLOAT));
  
  host:RegisterScriptParameter(self, "intensity");
  host:RegisterScriptParameter(self, "threshold");
  host:RegisterScriptParameter(self, "softthreshold");
  return self.Queue;
end

function bloomEffect:Resizeview(size)
  self.fbosize = mathfunction.vector2(size:x(), size:y());
end


function bloomEffect:Process(context, Original, Scene, Output)

  local layerDepth = 2.0;

  --light_prefilter, select pixels by brightness
  --get filted RTa[1], size by 1/2
  context:BeginRenderPass(self.RTa[1], apolloengine.RenderTargetEntity.CF_COLOR);
  self.light_prefilterMA:SetParameter(
    apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
    Scene:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
  self.light_prefilterMA:SetParameter( self.BLOOM_THRESHOLD, self.threshold );
  self.light_prefilterMA:SetParameter( self.SOFT_THRESHOLD, self.softthreshold );
  self.light_prefilterMA:SetParameter( self.TEXELSIZE, mathfunction.vector2( 1.0 / self.fbosize:x(), 1.0 / self.fbosize:y() ) );
  context:Draw(self.light_prefilterRO, self.light_prefilterMA);
  context:EndRenderPass();

  --4 times downsample
  --get RTa[2] ~ RTa[5], size by 1/4 ~ 1/32
  local scaleInv = 1.0;
  for i = 2, self.iterations do
    scaleInv = scaleInv * self.factor;
    context:BeginRenderPass(self.RTa[i], apolloengine.RenderTargetEntity.CF_COLOR);
    self.down_sampleMA:SetParameter(
      apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
      self.RTa[i - 1]:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
    self.down_sampleMA:SetParameter( self.TEXELSIZE, mathfunction.vector2( scaleInv / self.fbosize:x(), scaleInv / self.fbosize:y() ) );
    context:Draw(self.down_sampleRO, self.down_sampleMA);
    context:EndRenderPass();
  end

  --4 times upsample and accumulation
  --get RTb[4] ~ RTb[1], size by 1/16 ~ 1/2
  scaleInv = math.pow(self.factor, self.iterations + 1);
  for i = 1, self.iterations - 1 do
    scaleInv = scaleInv / self.factor;
    if i == 1 then
      --upsample
      context:BeginRenderPass(self.RTb[self.iterations - i], apolloengine.RenderTargetEntity.CF_COLOR);
      self.up_sampleMA:SetParameter(
        apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
        self.RTa[self.iterations]:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
      self.up_sampleMA:SetParameter( self.TEXELSIZE, mathfunction.vector2( scaleInv / self.fbosize:x(), scaleInv / self.fbosize:y() ) );
      context:Draw(self.up_sampleRO, self.up_sampleMA);
      context:EndRenderPass();
    else
      --upsample and accumulation
      context:BeginRenderPass(self.RTb[self.iterations - i], apolloengine.RenderTargetEntity.CF_COLOR);
      self.up_sample_acc_MA:SetParameter(
        apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
        self.RTb[self.iterations + 1 - i]:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
      self.up_sample_acc_MA:SetParameter(
        self.TEXTURE_SRC_A,
        self.RTa[self.iterations + 1 - i]:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
      self.up_sample_acc_MA:SetParameter( self.TEXELSIZE, mathfunction.vector2( scaleInv / self.fbosize:x(), scaleInv / self.fbosize:y() ) );
      context:Draw(self.up_sample_acc_RO, self.up_sample_acc_MA);
      context:EndRenderPass();
    end
  end

  --final accumulation to get RT0, size by 1
  context:BeginRenderPass(self.RT0, apolloengine.RenderTargetEntity.CF_COLOR);
  self.up_sample_acc_MA:SetParameter(
    apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
    self.RTb[1]:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
  self.up_sample_acc_MA:SetParameter(
    self.TEXTURE_SRC_A,
    self.RTa[1]:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
  self.up_sample_acc_MA:SetParameter( self.TEXELSIZE, mathfunction.vector2( self.factor / self.fbosize:x(), self.factor / self.fbosize:y() ) );
  context:Draw(self.up_sample_acc_RO, self.up_sample_acc_MA);
  context:EndRenderPass();

  --final combine
  context:BeginRenderPass(Output, apolloengine.RenderTargetEntity.CF_COLOR);
  self.accumulationMA:SetParameter(
    apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
    Scene:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
  self.accumulationMA:SetParameter(
    self.TEXTURE_SRC_A,
    self.RT0:GetAttachment( apolloengine.RenderTargetEntity.TA_COLOR_0 ) );
  self.accumulationMA:SetParameter( self.BLOOM_INTENSITY, self.intensity );
  context:Draw(self.accumulationRO, self.accumulationMA);
  context:EndRenderPass();

end

return bloomEffect;