--@author      : Jin Shiyu
--@date        : 2020-08-25
--@description : lua script to blur the image by one filter
--@version     : 1.0

local Object       = require "classic"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local apollonode   = require "apolloutility.apollonode"
local venuscore    = require "venuscore"
local defined      = require "apolloutility.defiend"
--[[
    deal width the current video frame by one filter.

    Usage:
    1. Initialize a FaceBeautyQuadRender.
    2. Call GetResultImage() for multiple video frames to get the result image.
    3. Destroy FaceBeautyQuadRender when it's no longer needed.
]]
local FaceBeautyQuadRender = Object:extend();

function FaceBeautyQuadRender:new(render_name, render_sequence, mask_layer, material_path, width, height)

    self.node = apollonode.QuadNode();
    self.node:SetName(render_name);

    self.camera       = self.node:CreateComponent(apolloengine.Node.CT_CAMERA);
    self.frame_width  = width;
    self.frame_height = height;
    self.initialized  = false;

    local clear_color   = mathfunction.Color(0.0, 0.0, 0.0, 0.0);

    -- setup fbo
    local fbo_size      = mathfunction.vector2(self.frame_width, self.frame_height);
    self.render_target  = apolloengine.RenderTargetEntity();
    --self.render_target:PushMetadata(apolloengine.RenderTargetFileMetadata(fbo_path));
    self.render_target:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
                                                                      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
                                                                      mathfunction.vector4(0, 0, self.frame_width, self.frame_height),
                                                                      fbo_size));
    self.image_gpu = self.render_target:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
    self.image_gpu:PushMetadata(apolloengine.TextureBufferMetadata(fbo_size,
                                                                   apolloengine.TextureEntity.TT_TEXTURE2D,
                                                                   apolloengine.TextureEntity.TU_STATIC,
                                                                   apolloengine.TextureEntity.PF_R8G8B8A8,
                                                                   1, false,
                                                                   apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                                   apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                                   apolloengine.TextureEntity.TF_LINEAR,
                                                                   apolloengine.TextureEntity.TF_LINEAR));
    self.render_target:CreateResource();

    -- setup camera
    self.camera:SetSequenceCulling(true);
    self.camera:SetLayerMaskNothing();
    self.camera:AddLayerMask(mask_layer);
    self.camera:SetSequence(render_sequence);
    self.camera:SetClearColor(clear_color);
    self.camera:FixedResolution(fbo_size);
    self.camera:AttachRenderTarget(self.render_target);
    self.camera:Deactivate();

    -- setup node
    local material = venuscore.IFileSystem:PathAssembly(material_path);
    self.node:SetLayer(mask_layer);
    self.node:SetShow(false);
    self.node:CreateResource(material, true);
    self.node:SetSequence(render_sequence);

    self.initialized   = true;
    LOG("[FaceBeautyQuadRender] init success "..mask_layer)
end

function FaceBeautyQuadRender:SetShow(is_show)
    if not self.initialized then
        LOG("[FaceBeautyQuadRender] SetShow fail");
        return;
    end

    self.node:SetShow(is_show);
    if not is_show then
        self.camera:Deactivate();
    else
        self.camera:Activate();
    end
end

function FaceBeautyQuadRender:SetSequence(sequence)
    if not self.initialized then
        LOG("[FaceBeautyQuadRender] preUpdate fail");
        return;
    end
    self.camera:SetSequence(sequence);
    self.node:SetSequence(sequence);
    --self.node:SetShow(false);
    --self.camera:Deactivate();
end

function FaceBeautyQuadRender:Update(frame_width, frame_height)
    if not self.initialized then
        LOG("[FaceBeautyQuadRender] update fail");
        return false;
    end

    -- set show for node
    self:SetShow(true);

    if frame_width == self.frame_width and frame_height == self.frame_height then
        return false;
    end

    -- change fbo
    local fbo_size = mathfunction.vector2(frame_width, frame_height);
    self.camera:FixedResolution(fbo_size);

    self.frame_width  = frame_width;
    self.frame_height = frame_height;
    return true;
end

function FaceBeautyQuadRender:SetParameter(key, value)
    if key == nil or value == nil then
        LOG("[FaceBeautyQuadRender] SetParameter nil "..key);
        return;
    end

    if self.initialized then
        self.node:SetParameter(key, value);
    end
end

function FaceBeautyQuadRender:GetResultImage()
    return self.image_gpu;
end

function FaceBeautyQuadRender:Clear()
    if self.node ~= nil then
        self.node:Destroy();
        self.Camera = nil;
    end
end

return FaceBeautyQuadRender;
