require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"

local EyeAngDis = {}

function EyeAngDis:_SwitchAng(on)
	-- venus editor on/off facial effect's switch
	self.ang_on = on;
	if self.ang_on then
		self.coef_ang = self.coef_ang_temp;
	else
		self.coef_ang = 0;
	end
end

function EyeAngDis:_SetAngTempCoef(coef)
	-- venus editor temporary params value, used when switched on
    -- coef: 1 -> max effect, 0 -> min effect 
	self.coef_ang_temp = coef;
	if self.ang_on then
		self.coef_ang = self.coef_ang_temp;
	end  
end

function EyeAngDis:_SwitchDis(on)
	-- venus editor on/off facial effect's switch
	self.dis_on = on;
	if self.dis_on then
		self.coef_dis = self.coef_dis_temp;
	else
		self.coef_dis = 0;
	end
end

function EyeAngDis:_SetDisTempCoef(coef)
	-- venus editor temporary params value, used when switched on
    -- coef: 1 -> max effect, 0 -> min effect 
	self.coef_dis_temp = coef;
	if self.dis_on then
		self.coef_dis = self.coef_dis_temp;
	end  
end

function EyeAngDis:_InitParams(defined)
  beautydefined = defined
  self.coef_dis = self.coef_dis or 0
  self.coef_ang = self.coef_ang or 0
  self.coef_ang_temp = 1
  self.coef_dis_temp = 1
  apolloengine.ShaderEntity.UNIFORM_EYEAD_LEFT_FROM = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEAD_LEFT_FROM");
  apolloengine.ShaderEntity.UNIFORM_EYEAD_LEFT_TO = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEAD_LEFT_TO");
  apolloengine.ShaderEntity.UNIFORM_EYEAD_RIGHT_FROM = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEAD_RIGHT_FROM"); 
  apolloengine.ShaderEntity.UNIFORM_EYEAD_RIGHT_TO = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEAD_RIGHT_TO");  
  apolloengine.ShaderEntity.UNIFORM_EYEAD_TRIANGLE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEAD_TRIANGLE");
  apolloengine.ShaderEntity.UNIFORM_EYEADCOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_EYEADCOEF");
end

function EyeAngDis:SetParams(config, shaderParams)
  shaderParams.eyead_left_from = mathfunction.vector2array();
  shaderParams.eyead_left_to = mathfunction.vector2array();
  shaderParams.eyead_right_from = mathfunction.vector2array();
  shaderParams.eyead_right_to = mathfunction.vector2array(); 
end

function EyeAngDis:_InitQuad(quadnode, shaderParams)
  local eyeTriangles = mathfunction.vector3array();
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_LEFT_FROM, shaderParams.eyead_left_from);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_LEFT_TO, shaderParams.eyead_left_to);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_RIGHT_FROM, shaderParams.eyead_right_from);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_RIGHT_TO, shaderParams.eyead_right_to);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_TRIANGLE, eyeTriangles);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEADCOEF, mathfunction.vector1(0)); 
end
function EyeAngDis:_get_eye_center(quadnode, keypoints, config, selIndex, eye_center)

	eye_center[1]=0;
	eye_center[2]=0;
    for i = 1, #selIndex, 1 do
		local p0 = keypoints[selIndex[i] + 1]; 
		eye_center[1]=eye_center[1]+keypoints[selIndex[i] + 1][1];
		eye_center[2]=eye_center[2]+keypoints[selIndex[i] + 1][2]; 
	end
	eye_center[1]=eye_center[1]/#selIndex;
	eye_center[2]=eye_center[2]/#selIndex;
end

function EyeAngDis:rotatePoint(pcenter,pdir,ang,p)
	local dir= {p[1]-pcenter[1],p[2]-pcenter[2]};
	local ca=math.cos(ang);
	local sa=math.sin(ang); 
	p[1]=pcenter[1]+dir[1]*ca-dir[2]*sa;
	p[2]=pcenter[2]+dir[1]*sa+dir[2]*ca;
end

function EyeAngDis:_get_eye_from_to(quadnode, keypoints, config, selIndex, pcenter,pdir,eyead_from,eyead_to,paras)
  local r1=paras[1];
	local r2 = paras[2];
	local d=paras[3];
	local ang=paras[4];	
	
	if r2<r1 then	
		LOG("warning: eye width r2 too small; r2="..r2.."<r1="..r1)
		r2=r1;
	end
	
	local eye_c0 = keypoints[selIndex[1] + 1];
	local eye_c1 = keypoints[selIndex[5] + 1];
	local eye = {eye_c1[1]-eye_c0[1], eye_c1[2]-eye_c0[2]};
	local eye_width=math.sqrt(eye[1]*eye[1]+eye[2]*eye[2]);
	--LOG("eye width:"..eye_width)
	d=d*eye_width;
	
	for i = 1, 4, 1 do
		local p0 = keypoints[selIndex[i] + 1];
		local p1 = keypoints[selIndex[i+4] + 1];
		local dir = {p1[1]-p0[1], p1[2]-p0[2]};
		local dir_len=math.sqrt(dir[1]*dir[1]+dir[2]*dir[2]);
		if dir_len > 0.0001 then
			dir[1]=dir[1]/dir_len;
			dir[2]=dir[2]/dir_len;
		else
			dir[1]=1;
			dir[2]=0;
		end
		
		local p00={p0[1]-dir[1]*eye_width*r1, p0[2]-dir[2]*eye_width*r1};
		local p01={p0[1]-dir[1]*eye_width*r2, p0[2]-dir[2]*eye_width*r2};
		local p10={p1[1]+dir[1]*eye_width*r1, p1[2]+dir[2]*eye_width*r1};
		local p11={p1[1]+dir[1]*eye_width*r2, p0[2]+dir[2]*eye_width*r2};
		eyead_from:PushBack(beautyutils._ArrayToVec2(p00));
		eyead_from:PushBack(beautyutils._ArrayToVec2(p01));
		eyead_from:PushBack(beautyutils._ArrayToVec2(p10));
		eyead_from:PushBack(beautyutils._ArrayToVec2(p11));
		
		--LOG(p00[1]..","..p00[2])
		self:rotatePoint(pcenter,pdir,ang,p00);
		--LOG(p00[1]..","..p00[2])
		self:rotatePoint(pcenter,pdir,ang,p10);
		
		local p00t={p00[1]+d*pdir[1],p00[2]+d*pdir[2]};
		local p01t={p01[1],p01[2]};
		local p10t={p10[1]+d*pdir[1],p10[2]+d*pdir[2]};
		local p11t={p11[1],p11[2]} ;
		eyead_to:PushBack(beautyutils._ArrayToVec2(p00t));
		eyead_to:PushBack(beautyutils._ArrayToVec2(p01t));
		eyead_to:PushBack(beautyutils._ArrayToVec2(p10t));
		eyead_to:PushBack(beautyutils._ArrayToVec2(p11t));
		
		--LOG(p00[1]..","..p00[2]..","..p00t[1]..","..p00t[2]..",")
		--LOG(p01[1]..","..p01[2]..","..p01t[1]..","..p01t[2]..",")
		--LOG(p10[1]..","..p10[2]..","..p10t[1]..","..p10t[2]..",")
		--LOG(p11[1]..","..p11[2]..","..p11t[1]..","..p11t[2]..",")
		 
	end
	 
end

function EyeAngDis:_UpdateTriPoint(quadnode, keypoints, config, rate)
  
  local dis_c = self.coef_dis*rate
  local ang_c = self.coef_ang*rate
	
	local eyeTriangles = mathfunction.vector3array();
	for i = 1, #beautydefined.EyeTriangles / 3, 1 do
		local cur_tri = mathfunction.vector3(beautydefined.EyeTriangles[i * 3 - 2], beautydefined.EyeTriangles[i * 3 - 1], beautydefined.EyeTriangles[i * 3]);
		eyeTriangles:PushBack(cur_tri); 
	end
	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_TRIANGLE, eyeTriangles); 
	
	local center_left = {0,0};
	local center_right = {0,0};
	self:_get_eye_center(quadnode, keypoints, config, beautydefined.LeftEyeKeyPoints,center_left);
	self:_get_eye_center(quadnode, keypoints, config, beautydefined.RightEyeKeyPoints,center_right);
	--LOG("eye center points:"..center_left[1]..","..center_left[2]..","..center_right[1]..","..center_right[2])
	self.center_left = center_left
  self.center_right = center_right
  
	self.eyead_left = {mathfunction.vector2array(),mathfunction.vector2array()};
	self.eyead_right = {mathfunction.vector2array(),mathfunction.vector2array()};
	local leftDir={center_right[1]-center_left[1],center_right[2]-center_left[2]};
	local dirL=math.sqrt(leftDir[1]*leftDir[1]+leftDir[2]*leftDir[2]);
	leftDir[1]=leftDir[1]/dirL;
	leftDir[2]=leftDir[2]/dirL;
 	
	local r1, r2, d, ang;
	
	if(self.coef_dis > 0) then
		r1=config.EyeAngleDistance[1];
		r2 = config.EyeAngleDistance[2];
		d=config.EyeAngleDistance[3]*dis_c;
		ang=3.1415926/180*config.EyeAngleDistance[4]*ang_c;
	else
		r1 = config.EyeAngleDistance_minus[1];
		r2 = config.EyeAngleDistance_minus[2];
		d = config.EyeAngleDistance_minus[3] * (-dis_c);
		ang = 3.1415926/180*config.EyeAngleDistance_minus[4] * ang_c;
	end

	local leftPara={r1,r2,d,ang};
	self:_get_eye_from_to(quadnode, keypoints, config,beautydefined.LeftEyeKeyPoints,center_left,leftDir,self.eyead_left[1],self.eyead_left[2],leftPara);
	local rightDir={-leftDir[1],-leftDir[2]};
 
	local rightPara={r1,r2,d,-ang};
	self:_get_eye_from_to(quadnode, keypoints, config,beautydefined.RightEyeKeyPoints,center_right,rightDir,self.eyead_right[1],self.eyead_right[2],rightPara);

	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_LEFT_FROM, self.eyead_left[1]);
	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_LEFT_TO, self.eyead_left[2]);
	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_RIGHT_FROM, self.eyead_right[1]);
	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEAD_RIGHT_TO, self.eyead_right[2]);
  if dis_c > 0 or ang_c > 0 then
    quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_EYEADCOEF, mathfunction.vector1(1.0));
  end
end

function EyeAngDis:_DrawQuad(quadnode, keypoints, config, rate)
  self:_UpdateTriPoint(quadnode, keypoints, config, rate)
end

function EyeAngDis:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, config, rate)
  if self.coef_dis == 0 and self.coef_ang == 0 then
    return;
  end
  local dis_c=self.coef_dis*rate;
  local ang_c=self.coef_ang*rate;
  
  local eye_width= beautyutils.GetDistance(keypoints[beautydefined.LeftEyeKeyPoints[1] + 1], keypoints[beautydefined.LeftEyeKeyPoints[5] + 1])
  local d, ang
  if(self.coef_dis > 0) then
	d=config.EyeAngleDistance[3]*dis_c;
	ang=3.1415926/180*config.EyeAngleDistance[4]*ang_c;
  else
	d=config.EyeAngleDistance_minus[3]*(-dis_c);
	ang=3.1415926/180*config.EyeAngleDistance_minus[4]*ang_c;
  end
  
  local pdir = {self.center_right[1]-self.center_left[1],self.center_right[2]-self.center_left[2]}
  local dirL=math.sqrt(pdir[1]*pdir[1]+pdir[2]*pdir[2]);
	pdir[1]=pdir[1]/dirL;
	pdir[2]=pdir[2]/dirL;
  local pcenter = self.center_left
  
  local leftkeypoints = {106, 78}
  if #keypoints >= 240 then
    for i =22, 43 do
     table.insert(leftkeypoints, i + 106 + 1)  
    end
  end
  for i = 1, #beautydefined.LeftEyeKeyPoints, 1 do
    table.insert(leftkeypoints, beautydefined.LeftEyeKeyPoints[i] + 1)
  end
  
  for i = 1, #leftkeypoints, 1 do
    local index = leftkeypoints[i];
    local newpoint = {keypoints[index][1], keypoints[index][2]}
    self:rotatePoint(pcenter,pdir,ang,newpoint);
    newpoint={newpoint[1]+d*pdir[1]*eye_width, newpoint[2]+d*pdir[2]*eye_width};
    local tmp = {keypoints[index][1] - newpoint[1], keypoints[index][2] - newpoint[2]}
    keypointsDisplacement[index][1] = keypointsDisplacement[index][1] + tmp[1]
    keypointsDisplacement[index][2] = keypointsDisplacement[index][2] + tmp[2]
  end
  
  pcenter = self.center_right
  pdir = {-pdir[1], -pdir[2]}
  ang = -ang
  eye_width= beautyutils.GetDistance(keypoints[beautydefined.RightEyeKeyPoints[1] + 1], keypoints[beautydefined.RightEyeKeyPoints[5] + 1])
  
  local rightkeypoints = {105, 75}
  if #keypoints >= 240 then
    for i =0, 21 do
     table.insert(rightkeypoints, i + 106 + 1)  
    end
  end
  for i = 1, #beautydefined.RightEyeKeyPoints, 1 do
    table.insert(rightkeypoints, beautydefined.RightEyeKeyPoints[i] + 1)
  end
  
  for i = 1, #rightkeypoints, 1 do
    local index = rightkeypoints[i];
    local newpoint = {keypoints[index][1], keypoints[index][2]}
    self:rotatePoint(pcenter,pdir,ang,newpoint);
    newpoint={newpoint[1]+d*pdir[1]*eye_width, newpoint[2]+d*pdir[2]*eye_width};
    local tmp = {keypoints[index][1] - newpoint[1], keypoints[index][2] - newpoint[2]}
    keypointsDisplacement[index][1] = keypointsDisplacement[index][1] + tmp[1]
    keypointsDisplacement[index][2] = keypointsDisplacement[index][2] + tmp[2]
  end
end

function EyeAngDis:UpdateParaValue(quadnode, config, key_index, attribute, value)
  if(attribute == "Ext1") then
    config.EyeAngleDistance[1] = value; 
  elseif(attribute == "Ext2") then
    config.EyeAngleDistance[2] = value; 
  elseif(attribute == "Dis") then
    config.EyeAngleDistance[3] = value;  
  elseif(attribute == "Ang") then
    config.EyeAngleDistance[4] = value;
  end
  LOG("eye: para= " .. config.EyeAngleDistance[1].." ,"..config.EyeAngleDistance[2].." ,"..config.EyeAngleDistance[3].." ,"..config.EyeAngleDistance[4])
  --self:_UpdateTriPoint(quadnode, keypoints, config)
end

function EyeAngDis:UpdateCoeff(quadnode, value_ang,value_dis, config, keypoints)
  if(value_ang == 1) then
    self.coef_ang = 1.0;
  else
    self.coef_ang = 0;
  end
  if(value_dis == 1) then
    self.coef_dis = 1.0;
  else
    self.coef_dis = 0;
  end
  --self:_UpdateTriPoint(quadnode, keypoints, config)
  LOG("eye: ang coef= " .. self.coef_ang.." dis coef:"..self.coef_dis)
end

function EyeAngDis:SetCoef(coef_ang,coef_dis)
  if coef_ang == nil then
    coef_ang = 0
  end
  if coef_dis == nil then
    coef_dis = 0
  end

  self.coef_ang = coef_ang;
  self.coef_dis = coef_dis;
  --LOG("eye: ang coef= " .. self.coef_ang.." dis coef:"..self.coef_dis)
end

function EyeAngDis:SetCoefAng(coef_ang)
	if coef_ang == nil then
	  coef_ang = 0
	end
	self.coef_ang = coef_ang;
end

function EyeAngDis:SetCoefDis(coef_dis)
	if coef_dis == nil then
	  coef_dis = 0
	end
	self.coef_dis = coef_dis;
end

function EyeAngDis:GetParams(config)
  return config.EyeAngleDistance
end 

function EyeAngDis:Release()
end

return EyeAngDis