#pragma once

float3 rgb2lab(float3 color)
{ 
    const float epsilon = 0.008856;  //actual CIE standard
    const float kappa = 903.3;       //actual CIE standard
    
    const float Xr = 0.950456;   //reference white
    const float Yr = 1.0;        //reference white
    const float Zr = 1.088754;   //reference white
    
    float r = color.r;
    float g = color.g;
    float b = color.b;

    r = ((r > 0.04045) ? pow((r + 0.055) / 1.055, 2.4) : (r / 12.92));
    g = ((g > 0.04045) ? pow((g + 0.055) / 1.055, 2.4) : (g / 12.92));
    b = ((b > 0.04045) ? pow((b + 0.055) / 1.055, 2.4) : (b / 12.92));

    float x = r * 0.4124564 + g * 0.3575761 + b * 0.1804375;
    float y = r * 0.2126729 + g * 0.7151522 + b * 0.0721750;
    float z = r * 0.0193339 + g * 0.1191920 + b * 0.9503041;
    
    //------------------------
    // XYZ to LAB conversion
    //------------------------
    float xr = x / Xr;
    float yr = y / Yr;
    float zr = z / Zr;
    
    xr = (xr > epsilon) ? pow(xr, 1.0 / 3.0) : (kappa * xr + 16.0) / 116.0;
    yr = (yr > epsilon) ? pow(yr, 1.0 / 3.0) : (kappa * yr + 16.0) / 116.0;
    zr = (zr > epsilon) ? pow(zr, 1.0 / 3.0) : (kappa * zr + 16.0) / 116.0;
    
    float ll = 116.0 * yr - 16.0;
    float aa = 500.0 *(xr - yr);
    float bb = 200.0 *(yr - zr);
    return float3(ll, aa, bb);
}

float3 lab2rgb(float3 lab) 
{
    const float epsilon = 0.008856;  //actual CIE standard
    const float kappa = 903.3;       //actual CIE standard
    const float Xr = 0.950456;   //reference white
    const float Yr = 1.0;        //reference white
    const float Zr = 1.088754;   //reference white

    float y = (lab.r + 16.0) / 116.0;
    float x = lab.g / 500.0 + y;
    float z = y - lab.b / 200.0;
    
    float x3 = pow(x, 3);
    float y3 = pow(y, 3);
    float z3 = pow(z, 3);
    
    float invKappa = kappa / 116.0;
    
    x = ((x3 > epsilon) ? x3 : ((x - 16.0 / 116.0) / invKappa)) * Xr;
    y = ((y3 > epsilon) ? y3 : ((y - 16.0 / 116.0) / invKappa)) * Yr;
    z = ((z3 > epsilon) ? z3 : ((z - 16.0 / 116.0) / invKappa)) * Zr;
    
    float r = x * 3.2404542 + y * -1.5371385 + z * -0.4985314;
    float g = x * -0.9692660 + y * 1.8760108 + z * 0.0415560;
    float b = x * 0.0556434 + y * -0.2040259 + z * 1.0572252;
    
    r = ((r > 0.0031308) ? (1.055*pow(r, 1 / 2.4) - 0.055) : (12.92*r));
    g = ((g > 0.0031308) ? (1.055*pow(g, 1 / 2.4) - 0.055) : (12.92*g));
    b = ((b > 0.0031308) ? (1.055*pow(b, 1 / 2.4) - 0.055) : (12.92*b));
    
    return float3(r, g, b);
}

float3 normalizelab(float3 lab)
{
    float x = lab.r / 100.0;
    float y = (lab.g + 110) / 220.0;
    float z = (lab.b + 110) / 220.0;
    return float3(x, y, z);
}

float3 denormalizelab(float3 color)
{
    float x = color.r * 100.0;
    float y = color.g * 220.0 - 110.0;
    float z = color.b * 220.0 - 110.0;
    return float3(x, y, z);
}

