local Object        = require "classic"
local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local mathfunction  = require "mathfunction"
local defined       = require "facemorph.defined"
local venuscore     = require "venuscore"
local likeapp       = require "likeapp"

local MorphRenderer = Object:extend();

function MorphRenderer:new(morphtime, showtime)  
  self.points = {}; 
  self.texs  = {};
  self.texCoords = {};
  self.curTexid = {};
  
  self.imagelist = {};
  self.landmarklist = {};
  self.indiceslist = {};
  self.interval = {};  
  
  self.defaultMorphInterval = morphtime;
  self.defaultShowInterval = showtime;
  
  self:_GenerateRenderContexts();
end

function MorphRenderer:Clear()  
  self.points = {};
  self.texs = {};
  self.texCoords = {};
  self.curTexid = {};
  
  self.imagelist = {};
  self.landmarklist = {};
  self.indiceslist = {};
  self.interval = {};  
end

function MorphRenderer:ReleaseResource()
  self:Clear()
end

-- InitMorph
-- Note: if use mockup data to do internal test, please use SetMorphTextureInfos to set data
--       comment self:Clear(); then call this func
function MorphRenderer:InitMorph(indiceslist, showtimelist, transtimelist)   
  self:Clear();
  
  if ( indiceslist == nil or showtimelist == nil or transtimelist == nil)
  then
    return false;
  end    
  
  self.indiceslist = indiceslist;

  -- start
  self.interval[1] = 0;    
  for i = 1, #self.indiceslist 
  do
    local showtime = showtimelist[i] or self.defaultShowInterval;
    local transtime = transtimelist[i] or self.defaultMorphInterval;
    self.interval[2*i] = self.interval[2*i-1] + showtime;
    -- do not add the last transition time to the interval list
    if (i ~= self.indiceslist)
    then
      self.interval[2*i + 1] = self.interval[2*i] + transtime;    
    end
  end
  return true;
end

-- Create FBO context based on input fbosize
function MorphRenderer:SetFboSize(fbosize)
  self.morphCamera = apollonode.CameraNode();
  self.morphCamera:Activate();
  self.morphCamera:SetSequence(defined.SEQUENCE);
  self.morphCamera:SetClearColor(mathfunction.Color(0.0, 0.0, 0.0, 0.0));
  self.rt, self.morphTex = self:_CreateRenderTarget(fbosize);
  self.morphCamera:AttachRenderTarget(self.rt);
end

-- Update
function MorphRenderer:Update(def)
  local leftI, rightI = self:_Search(def);  
  
  if (leftI == rightI)
  then
    self:_DoExhibition(leftI);
  else
    local _start = self.interval[2 * leftI];
    local _end = self.interval[2 * rightI - 1];
    local alpha = (def - _start) / (_end - _start);
    self:_DoMorphing(leftI, rightI, alpha);
    return;
  end
  
  self:SetShow(true);
end

-- Setshow 
function MorphRenderer:SetShow(show)
  if self.renderNode ~= nil then
    self.renderNode:SetShow(show);
  end
end

function MorphRenderer:GetMorphTex()
  return self.morphTex;
end

-- Generate texture based on tex path
function MorphRenderer:LoadTexture(tex_path)
  if (tex_path == nil)
  then
    return;
  end
  
  local texture = apolloengine.TextureEntity();
  texture:PushMetadata(  apolloengine.TextureFileMetadata (
                         apolloengine.TextureEntity.TU_STATIC,
                         apolloengine.TextureEntity.PF_AUTO,
                         1, false,
                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                         apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                         apolloengine.TextureEntity.TF_LINEAR,
                         apolloengine.TextureEntity.TF_LINEAR,
                         tex_path));
     
  texture:SetKeepSource(true); 
  --texture:SetJobType(venuscore.IJob.JT_SYNCHRONOUS);   
  texture:CreateResource(); 
  
  return texture;
end

-- Generate Render Contexts
-- 1. Init vertex/indices stream
-- 2. Load material
-- 3. Create RenerNode
function MorphRenderer:_GenerateRenderContexts()  
  self.texSlot = {};
  self.vtxStream, self.idxStream     =  self:_InitStreams();  
  
  self.texSlot[defined.SOURCE]     =  apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE0");                          
  self.texSlot[defined.TARGET]     =   apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE1");                                
  self.alphaSlot      =   apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ALPHA");
  
  self.renderNode     =  apollonode.RenderNode(); 
  local materialPath  = "docs:facemorph/material/facemorph.material";
  local renderMode    =  apolloengine.RenderComponent.RM_TRIANGLES ;

  self.renderNode:CreateResource(materialPath, renderMode, self.vtxStream, self.idxStream);    
  self.renderNode:SetSequence(defined.SEQUENCE);
  self.renderNode:SetShow(false);
end


-- find the morph images in def timeslot
-- return [left, right]
function MorphRenderer:_Search(def)
  local left, right = 0, #self.interval;
  
  for i = 1, #self.interval 
  do
    local interval = self.interval[i];
    if def >= interval
    then
      left = i;
    else
      right = i;
      break;
    end
  end
  -- Get the index in indiceslist
  left  = math.floor((left + 1 ) / 2);
  right = math.floor((right + 1) / 2);  
  
  return left, right;
end

-- Init buffer, generates morph vertices and textures
-- The vertices data will be normalized to [-1, 1]
-- The landmarks data will be normalized to [0, 1]
function MorphRenderer:_InitBuffer(index, ts, landmarks)
  self.points[index] = self:_GenerateVertices(ts, landmarks);
  self.texCoords[index] = self:_GenerateTextureCoords(ts, self.points[index]);   
  self:_NormalizeVertices( self.points[index], ts);
end

-- Init shader parameter
function MorphRenderer:_InitShaderParameter(alpha)
  if (self.imagelist == nil or #self.imagelist == 0)
  then
    return;
  end
  
  self.texs[defined.SOURCE] = self:_GenerateTexture(self.imagelist[1]);
  if (#self.imagelist == 1)
  then
    self.texs[defined.TARGET] = self.texs[defined.SOURCE];
  else
    self.texs[defined.TARGET] = self:_GenerateTexture(self.imagelist[2]);
  end
  
  self:_UpdateBlendfactor(alpha);
  for i = defined.SOURCE, defined.TARGET 
  do
    if (self.texs[i] ~= nil) 
    then
      local texSlot, texAttr = self:_GetTextureSlotAttr(i);
      self.renderNode:SetParameter(texSlot, self.texs[i]);  
    end
  end  
end

-- Get texture slot and attribute based on index
function MorphRenderer:_GetTextureSlotAttr(index)
  if index < defined.SOURCE or index > defined.TARGET
  then
    return nil;
  end
  
  local slot = self.texSlot[index];  
  
  local attr = nil;
  if (index == 1)
  then
    attr = apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0;
  elseif(index == 2)
  then
    attr = apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1;
  end
  return slot, attr;
end

-- generate texture based on texture stream
function MorphRenderer:_GenerateTexture(ts)
  if (ts == nil)
  then
    return nil;
  end
  
  local texture = apolloengine.TextureEntity();
  texture:PushMetadata(  apolloengine.TextureBufferMetadata(
                          apolloengine.TextureEntity.TU_WRITE,
                          1, false,
                          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                          apolloengine.TextureEntity.TF_LINEAR,
                          apolloengine.TextureEntity.TF_LINEAR,
                          ts));
                      
  texture:CreateResource();  
  return texture;
end

-- Create RT context
function MorphRenderer:_CreateRenderTarget(size)      
  --local clear_color = mathfunction.Color(0.0, 0.0, 0.0, 0.0); -- move to camera
  
  local rt = apolloengine.RenderTargetEntity();
  rt:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
                                                    apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
                                                    mathfunction.vector4(0, 0, size:x(), size:y()),
                                                    size)
                  );

  local tex = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
  tex:PushMetadata(apolloengine.TextureBufferMetadata(size,
                                                      apolloengine.TextureEntity.TT_TEXTURE2D,
                                                      apolloengine.TextureEntity.TU_READ,
                                                      apolloengine.TextureEntity.PF_R8G8B8A8,
                                                      1, false,
                                                      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                      apolloengine.TextureEntity.TF_LINEAR,
                                                      apolloengine.TextureEntity.TF_LINEAR));
                  
  tex:SetKeepSource(true)
  rt:CreateResource();

  return rt, tex;
end

-- Initialize the vertex stream and indicies stream
function MorphRenderer:_InitStreams()  
  local vtxstream     = apolloengine.VertexStream();
  
  -- set vertex format
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2);
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2);
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2);     

  for i = 1, defined.NUM_DELAUNAY_POINTS do 
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_POSITION, mathfunction.vector2(0, 0));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, mathfunction.vector2(0, 0));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1, mathfunction.vector2(0, 0));
  end
  
  local idxStream = apolloengine.IndicesStream();
  local indicesNum = #defined.DELAUNAY_INDICE_WITH_MOUTH;
  
  idxStream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
  idxStream:ReserveBuffer(indicesNum);

  for i = 1, indicesNum do
    idxStream:PushIndicesData(defined.DELAUNAY_INDICE_WITH_MOUTH[i]);
  end

  return vtxstream, idxStream;
end


-- normalize vertices to [-1, 1]
-- @ vertices:        input/output vertices
-- @ ts:              input texturestream
function MorphRenderer:_NormalizeVertices(vertices, ts)
  if (vertices == nil or ts == nil)
  then
    return;
  end

  local size = ts:GetSize();
  for i = 1, defined.NUM_DELAUNAY_POINTS do    
    local pnt = vertices:Get(i);
    --convert to position coordinate(-1, 1)
    local x = pnt:x() / size:x() * 2 - 1;
    local y = (1- pnt:y() / size:y()) * 2 - 1;
    local point = mathfunction.vector2(x, -y);
    vertices:Set(i, point);
  end
end

-- Update Blendfactor
function MorphRenderer:_UpdateBlendfactor(alpha)
  self.renderNode:SetParameter(self.alphaSlot, mathfunction.vector1(alpha));
end

-- update morph vertices based on alpha 
function MorphRenderer:_UpdateVertices(pointsTbl, alpha)
  if (pointsTbl == nil or #pointsTbl ~= defined.TARGET) then
    return;
  end  
  
  local morphvertices = mathfunction.vector2array();
  if (alpha == 0)
  then
    morphvertices = pointsTbl[defined.SOURCE];
  elseif(alpha == 1)
  then
    morphvertices = pointsTbl[defined.TARGET];
  else
    local points = {};
    for i = 1, defined.NUM_DELAUNAY_POINTS do
      for j = 1, #pointsTbl do
        local pnt = pointsTbl[j]:Get(i);
        table.insert(points, j, pnt);
      end
      morphvertices:PushBack(points[defined.SOURCE] * (1 - alpha) + points[defined.TARGET] * alpha);
    end
  end
  self:_UpdateVertexStream(apolloengine.ShaderEntity.ATTRIBUTE_POSITION, morphvertices);  
end

-- Update texture 
-- @ index: which needs to be updated, valid input is SOURCE or TARGET
-- @ texcoords: texcoords data to be updated
-- @ tex:       texture to be updated
function MorphRenderer:_UpdateTexture(index, texcoords, tex)  
  if (texcoords == nil or tex == nil)
  then
    return;
  end
  
  local texSlot, texAttr = self:_GetTextureSlotAttr(index);
  if (self.texs[index] ~= nil)
  then
    self:_UpdateVertexStream(texAttr, texcoords); 
    --self.texs[index]:SubstituteTextureBuffer(ts);
    self.renderNode:SetParameter(texSlot, tex);  
  end
end

-- Update vertex stream
-- @ attribute: indicates which needs to be updated
-- @ newpoints: new data
function MorphRenderer:_UpdateVertexStream(attribute, newpoints)
  local newpoint =  mathfunction.vector2(0, 0)
  local offset = self.vtxStream:GetAttributeIndex(attribute);
  for i = 1, newpoints:Size() do
    local point = newpoints:Get(i);
    newpoint:Set(point:x(), point:y());
    self.vtxStream:ChangeVertexDataWithAttributeFast(
                                  offset,
                                  i,
                                  newpoint);
  end
   
  self.vtxStream:SetReflushInterval(1, newpoints:Size());
  self.renderNode.render:ChangeVertexBuffer(self.vtxStream);
end

-- Generate initiate landmarks/border points
-- @ ts:              input TextureStream
-- @ lmk:             input mathfunction.vector2array: face+foreheads;
function MorphRenderer:_GenerateVertices(ts, lmk)
  if lmk == nil or ts == nil then
    return nil;
  end
  
  if #lmk ~= (defined.NUM_FACE_LAMDMARK + defined.NUM_FOREHEAD_LANDMARK) * 2 
  then
    return nil;
  end
  
  local size = ts:GetSize();
  
  local points = mathfunction.vector2array();
  for i = 1, #lmk, 2 do
    points:PushBack(mathfunction.vector2(lmk[i], lmk[i+1]));
  end

  local col, row = 0, 0;
  local left, bot, width, height = 0, 0, size:x(), size:y();
  
  local NUM_DIVISION = 10;
  
  for cx = 1, NUM_DIVISION + 1 do
    col = math.min(left + width * (cx - 1) / NUM_DIVISION, left + width);
    for cy = 1, NUM_DIVISION + 1 do
      row =  math.min(bot + height * (cy - 1) / NUM_DIVISION, bot + height);
      if not (
        col ~= left and 
        col ~= (left + width) and
        row ~= bot and 
        row ~= (bot + height) ) then
        points:PushBack(mathfunction.vector2(col, row));
      end 
    end
  end
  
  return points;
end

-- Generate Texture Coordinates
-- @ ts     :         input instance of TextureStream
-- @ points  :        input mathfunction.vector2array
function MorphRenderer:_GenerateTextureCoords(ts, points)
  if (points == nil or ts == nil) then
    return nil;
  end
  
  local size = ts:GetSize();
  local texCoords = mathfunction.vector2array();

  for j = 1, defined.NUM_DELAUNAY_POINTS do
    local point = points:Get(j); 
    -- normalize to [0, 1]
    texCoords:PushBack(point/size);     
  end
  return texCoords;
end

-- Morphing source image to target image in interval seconds
function MorphRenderer:_DoMorphing(index1, index2, alpha)  
  if (index1 < 1 or index1 > #self.indiceslist or 
      index2 < 1 or index2 > #self.indiceslist)
  then
    return false;
  end
  
  local indices = {self.indiceslist[index1], self.indiceslist[index2]};
  
  for i = defined.SOURCE, defined.TARGET do 
    local index = indices[i];
    if (index < 1)
    then
      return false;
    end  
    
    if (self.curTexid[i] == nil or self.curTexid[i] ~= index)
    then 
      if self.imagelist[index] == nil
      then
        self.imagelist[index] = likeapp.AI:GetMorphTexture(index);      
      end
      
      if self.landmarklist[index] == nil
      then
        self.landmarklist[index] = likeapp.AI:GetMorphLandmarks(index);
      end
      
      self.texs[i] = self:_GenerateTexture(self.imagelist[index]);
      self:_InitBuffer(i, self.imagelist[index], self.landmarklist[index]);
      self:_UpdateTexture(i, self.texCoords[i], self.texs[i]);      
      self.curTexid[i] =  indices[i];
    end
  end

  self:_UpdateBlendfactor(alpha);
  self:_UpdateVertices(self.points, alpha);
  return true;
end

-- Show an image
function MorphRenderer:_DoExhibition(index)
  return self:_DoMorphing(index, index, 0);
end

-- SetMorphTextureInfos, this is an internally lua used API  
function MorphRenderer:SetMorphTextureInfos(tslist, landmarklist)
  self.imagelist = tslist;
  self.landmarklist = landmarklist;
end

return MorphRenderer;
