local venuscore = require "venuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local SDFSubtitleCacheInfo = require "behavior.sdf_subtitle_cache_info"
local TC = require "bluecore.base.transaction_center"
local BD = require "bluecore.bluedefined"
local venusjson = require "venusjson"
local likeapp = require "likeapp"

local SDFSubtitleNodeBehavior = venuscore.VenusBehavior:extend("SDFSubtitleNodeBehavior");

local EMOJI_MAX_DURATION = 2 * 1000  --2s
local REQUEST_TYPE = BD.RequestType.EMOJI
local TOP_CAMERA_SEQUENCE = 10000;

G_EmojiRequestId = 110;

function SDFSubtitleNodeBehavior:new()
    self.node = nil;
    self.subNodeList = {};
    self.mainCam = {};
    self.sdfCom = nil;
    self.lastpts = 0

    self.sdfSubtitleCacheInfo = nil;
    self.textStr = "";
    self.resourcePath = nil;
    self.screenPos = mathfunction.vector2(0.5, 0.5);
    self.rotationZ = 0;
    self.scale = 1;
    self.ttfPath = "";
    self.rgba = nil;
    self.useGradient = false;
    self.startTs = 0;
    self.duration = 2^32-1
    self.cameraSequence = 0;

    self.emojiRequestTimeOffset = 0;
    self.REFRESH_FRAME_ID = 12345;

    self.emojiRequestId = 0;

    self.REFRESH_FRAME_FUNCTION = {
        _IsTimeOut = function()
            return true
        end,
        _TimeOut = function()
        end,
        _Ready = function()
        end
    }
end

function SDFSubtitleNodeBehavior:initBehaviorData(node, subNodeList, resourcePath)
    self.node = node;
    self.subNodeList = subNodeList;
    self.resourcePath = resourcePath;
    self.sdfCom = self:_GetSDFComponent(subNodeList)
    --初始化位置和旋转，方便统一调整素材大小
    self:InitializePositionAndScale();
    --默认初始化行间距和emoji的默认渲染顺序
    if self.sdfCom then
      self.sdfCom.LineSpace = 10;
      local emojiRender = self.sdfCom.EmojiRender;
      if emojiRender then
        emojiRender:SetRenderOrder(10);
      end
    end
    return self.sdfCom ~= nil;
end

function SDFSubtitleNodeBehavior:InitializePositionAndScale()
  local trans = self:_GetTransformComponent();
  local pos = trans:GetLocalPosition();
  trans:SetLocalPosition(mathfunction.vector3(pos:x(), pos:y(), -13));
  trans:SetLocalScale(mathfunction.vector3(1,1,1));
end

function SDFSubtitleNodeBehavior:StoreAttributes()
    LOG("SDFSubtitleNodeBehavior:StoreAttributes")
    if self.rgba then
      LOG("Store Color " .. self.rgba:x() .. "_" .. self.rgba:y() .. "_" .. self.rgba:z() .. "_" .. self.rgba:w())
    else
      LOG("RGBA is nil..!!!!!!!!!!!")
    end
    self.sdfSubtitleCacheInfo = SDFSubtitleCacheInfo(self.textStr, self.screenPos, self.rotationZ,
            self.scale, self.ttfPath, self.rgba, self.useGradient, self.startTs, self.duration, self.resourcePath)
end

function SDFSubtitleNodeBehavior:GetStoreAttributes()
  local info = nil;
  if self.sdfSubtitleCacheInfo then
    info = {};
    info["fontPath"] = self.sdfSubtitleCacheInfo:GetTTFPath();
    local screenPos = self.sdfSubtitleCacheInfo:GetPosition();
    info["pos"] = mathfunction.vector2(screenPos:x(), screenPos:y());
    info["rotation"] = self.sdfSubtitleCacheInfo:GetRotationZ()
    info["scale"] = self.sdfSubtitleCacheInfo:GetScale()
    local color, useGradient = self.sdfSubtitleCacheInfo:GetColor()
    if color then
      info["color"] = mathfunction.vector4(color:x(), color:y(), color:z(), color:w());
    else
      info["color"] = nil;
    end
    info["useGradient"] = useGradient
    local startTs, duration = self.sdfSubtitleCacheInfo:GetTimeRange()
    info["timeRange"] = mathfunction.vector2(startTs, duration)
    info["text"] = self.sdfSubtitleCacheInfo:GetTextStr()
    info["resourcePath"] = self.sdfSubtitleCacheInfo:GetResourcePath()
  end
  return info;
end

function SDFSubtitleNodeBehavior:RestoreAttributes()
    LOG("SDFSubtitleNodeBehavior:RestoreAttributes")
    if self.sdfSubtitleCacheInfo then
        self:UpdateSubtitleText(self.sdfSubtitleCacheInfo:GetTextStr())
        self:SetSubtitleFontPosition(self.sdfSubtitleCacheInfo:GetPosition())
        self:SetSubtitleFontRotation(self.sdfSubtitleCacheInfo:GetRotationZ())
        self:SetSubtitleFontScale(self.sdfSubtitleCacheInfo:GetScale())
        self:UpdateSubtitleFontPath(self.sdfSubtitleCacheInfo:GetTTFPath())
        self:UpdateFontColor(self.sdfSubtitleCacheInfo:GetColor())
        self:UpdateSubtitleTimeRange(self.sdfSubtitleCacheInfo:GetTimeRange())
        self.sdfSubtitleCacheInfo = nil
    end
end

function SDFSubtitleNodeBehavior:GetSubtitleInfo()
    local info = {}
    info["size"] = self:GetSubtitleFontSize()
    local screenPos = self:GetSubtitleFontPosition();
    info["transform"] = mathfunction.vector4(screenPos:x(), screenPos:y(), self.scale, self.rotationZ);
    info["color"] = self.rgba
    if self.rgba then
      LOG("GetSubtitleInfo Get Color " .. self.rgba:x() .. "_" .. self.rgba:y() .. "_" .. self.rgba:z() .. "_" .. self.rgba:w())
    end
    info["useGradient"] = self.useGradient
    info["timeRange"] = mathfunction.vector2(self.startTs, self.duration)
    info["text"] = self.textStr
    info["fixedRect"] = self.sdfCom.FixedRect
    return info
end

function SDFSubtitleNodeBehavior:UpdateSubtitleText(textStr)
    LOG("SDFSubtitleNodeBehavior:UpdateSubtitleText")
  if textStr ~= self.textStr then
    self.textStr = textStr
    self.sdfCom.Text = textStr
    self:_RequestEmojiImage()
  end
end

--斜体
function SDFSubtitleNodeBehavior:UpdateSubtitleItalic(flag)
  LOG("SDFSubtitleNodeBehavior:UpdateSubtitleItalic");
  self.sdfCom.Italic = flag;
  self.sdfCom:RenderText();
end

--粗体
function SDFSubtitleNodeBehavior:UpdateSubtitleBold(flag)
    LOG("SDFSubtitleNodeBehavior:UpdateSubtitleBold");
    self.sdfCom.Bold = flag;
    self.sdfCom:RenderText();
end

--字间距
function SDFSubtitleNodeBehavior:UpdateSubtitleWordSpace(value)
      LOG("SDFSubtitleNodeBehavior:UpdateSubtitleWordSpace");
      self.sdfCom.WordSpace = value;
      self.sdfCom:RenderText();
end

--行间距
function SDFSubtitleNodeBehavior:UpdateSubtitleLineSpace(value)
      LOG("SDFSubtitleNodeBehavior:UpdateSubtitleLineSpace");
      self.sdfCom.LineSpace = value;
      self.sdfCom:RenderText();
end

function SDFSubtitleNodeBehavior:UpdateSubtitleFontPath(ttfPath)
    LOG("SDFSubtitleNodeBehavior:UpdateSubtitleFontPath " .. ttfPath)
    if ttfPath == "" or ttfPath == nil then
      LOG("SDFSubtitleNodeBehavior:UpdateSubtitleFontPath is nil !");
      return;
    end
    if ttfPath ~= self.ttfPath then
        self.ttfPath = ttfPath
        self.sdfCom.FontPath = apolloengine.PathMetadata(ttfPath)
        self.sdfCom:RenderText()
    end
end

function SDFSubtitleNodeBehavior:UpdateFontColor(rgba, useGradient)
  LOG("SDFSubtitleNodeBehavior:UpdateFontColor")
  if rgba ~= self.rgba or useGradient ~= self.useGradient then
      if useGradient then
          self.sdfCom:SetGradientColor();
      else
        if rgba then
          LOG("SDFSubtitleNodeBehavior:UpdateFontColor Set Color " .. rgba:x() .. "_" .. rgba:y() .. "_" .. rgba:z() .. "_" .. rgba:w());
          self.sdfCom:SetColor(rgba);
        end
      end
      self.useGradient = useGradient;
      self.rgba = rgba;
  end
end

function SDFSubtitleNodeBehavior:SetSubtitleFontPosition(screenPos)
    LOG("SDFSubtitleNodeBehavior:SetSubtitleFontPosition ")
    if screenPos ~= self.screenPos then
        self.screenPos = screenPos
        local screenPosNormalize = screenPos * 2 - mathfunction.vector2(1,1);
        local trans = self:_GetTransformComponent()
        local m = trans:GetWorldTransform();
        local camera = self.mainCam[#self.mainCam];
        if camera then
            local vp = camera:GetViewProj();
            local originScreenPos = mathfunction.vector4(0, 0, 0, 1) * m * vp;
            local z = originScreenPos:z()/originScreenPos:w();
            local screenPosVec4 = mathfunction.vector4(screenPosNormalize:x(), screenPosNormalize:y(), z, 1);
            local position = screenPosVec4 * camera:GetUnViewProj();
            position = position/position:w();
            local worldPos = mathfunction.vector3(position:x(), position:y(), position:z());
            trans:SetLocalPosition(worldPos);
        else
            LOG("ERROR: camera is nil")
        end
    end
end

function SDFSubtitleNodeBehavior:GetSubtitleFontPosition()
    LOG("SDFSubtitleNodeBehavior:GetSubtitleFontPosition ")
    local trans = self:_GetTransformComponent()
    local m = trans:GetWorldTransform();
    local camera = self.mainCam[#self.mainCam];
    if camera then
        local vp = camera:GetViewProj();
        local screenPos = mathfunction.vector4(0, 0, 0, 1) * m * vp;
        local screenPosVec2 = (mathfunction.vector2(screenPos:x() / screenPos:w(), screenPos:y() / screenPos:w()) + mathfunction.vector2(1, 1)) * 0.5;
        return screenPosVec2;
    end
    return mathfunction.vector2(0.5,0.5);
end

function SDFSubtitleNodeBehavior:SetSubtitleFontRotation(rotationZ)
    LOG("SDFSubtitleNodeBehavior:SetSubtitleFontRotation")
    if rotationZ ~= self.rotationZ then
        self.rotationZ = rotationZ
        local trans = self:_GetTransformComponent()
        local rotation = mathfunction.Quaternion();
        rotation:RotateXYZ(0, 0, math.pi * 2 * rotationZ); --rotationZ范围是0~1
        trans:SetLocalRotation(rotation);
    end
end

function SDFSubtitleNodeBehavior:SetSubtitleFontScale(scale, isFinalize)
    LOG("SDFSubtitleNodeBehavior:SetSubtitleFontScale")
    if scale ~= self.scale then
        self.scale = scale
        local trans = self:_GetTransformComponent()
        trans:SetLocalScale(mathfunction.vector3(scale, scale, 1.0))
        if isFinalize then

        end
    end
end

function SDFSubtitleNodeBehavior:GetSubtitleFontSize()
    LOG("SDFSubtitleNodeBehavior:GetSubtitleFontSize")
    local trans = self:_GetTransformComponent()
    local m = trans:GetWorldTransform();
    local pos = trans:GetLocalPosition();
    local scale = trans:GetLocalScale();
    local camera = self.mainCam[#self.mainCam];
    if camera then
        local vp = camera:GetViewProj();
        local rect = self:_GetRect();
        local rightTop = mathfunction.vector4(0.5 * 0.01 * rect:x() * scale:x(), 0.5 * 0.01 * rect:y() * scale:y(), pos:z(), 1) * vp;
        local leftBottom = mathfunction.vector4(-0.5 * 0.01 * rect:x() * scale:x(), -0.5 * 0.01 * rect:y() * scale:y(), pos:z(), 1) * vp;
        local rightTopVec2 = (mathfunction.vector2(rightTop:x() / rightTop:w(), rightTop:y() / rightTop:w()) + mathfunction.vector2(1, 1)) * 0.5;
        local leftBottomVec2 = (mathfunction.vector2(leftBottom:x() / leftBottom:w(), leftBottom:y() / leftBottom:w()) + mathfunction.vector2(1, 1)) * 0.5;
        local size =  mathfunction.vector2(math.abs(rightTopVec2:x() - leftBottomVec2:x()), math.abs(rightTopVec2:y() - leftBottomVec2:y()));
        return size;
    end
    return mathfunction.vector2(0,0);
end

function SDFSubtitleNodeBehavior:SetSubtitleFontAlpha(alpha)
    local val = mathfunction.vector1(alpha)
    local root = self.sdfCom:GetHostNode();
    local renders = {};
    self:_GetAllRenders(root, renders);
    for k,v in pairs(renders) do
        v:SetParameter("_Alpha", val);
    end
end

function SDFSubtitleNodeBehavior:UpdateSubtitleTimeRange(startTs, duration)
    LOG("SDFSubtitleNodeBehavior:UpdateSubtitleTimeRange")
    if startTs ~= self.startTs or duration ~= self.durationMs then
        self.startTs = startTs
        self.duration = duration
    end
end

function SDFSubtitleNodeBehavior:ConfigRenderTarget(rt)
    if #self.mainCam > 0 then
        for i=1, #self.mainCam do
            local cameracom = self.mainCam[i];
            cameracom:AttachRenderTarget(rt);
        end
    else
        for i=1, #self.subNodeList do
            local cameracom = self.subNodeList[i]:GetComponent(apolloengine.Node.CT_CAMERA);
            if cameracom and cameracom:GetAttachedRenderTarget() == nil then
                cameracom:AttachRenderTarget(rt);
                table.insert(self.mainCam, cameracom);
            end
        end
    end
    local mainCam = self.mainCam[#self.mainCam];
    if mainCam then
      --加载完素材主动更新相机vp矩阵
      LOG("SDFSubtitleNodeBehavior:Recalculate Camera")
      local resolution = apolloengine.Framework:GetResolution();
      mainCam:ResizeView(resolution:x(), resolution:y());
      mainCam:Recalculate();
      self.cameraSequence = mainCam:GetSequence();
      self.sdfCom:SetMainCamera(mainCam)
    end
end

function SDFSubtitleNodeBehavior:UpdateFrameInfo(frameIndex, pts)
    if pts >= self.startTs and pts <= self.startTs + self.duration then
        self.node.Active = true
        self:_UpdateWallTs(pts)
        LOG("====> active")
    else
        self.node.Active = false
        LOG("xxxx> deactive")
    end
end

function SDFSubtitleNodeBehavior:_GetSDFComponent(subNodeList)
    for i = 1, #subNodeList do
        local com = subNodeList[i]:GetComponent(apolloengine.Node.CT_SDF_COM);
        if com ~= nil then
            LOG("[_GetSDFComponent] got it");
            return com;
        end
    end
    return nil;
end

function SDFSubtitleNodeBehavior:_GetTransformComponent()
    if self.sdfCom then
        local node = self.sdfCom:GetHostNode()
        LOG("host node = " .. node.Name)
        return node:GetComponent(apolloengine.Node.CT_TRANSFORM);
    end
    return nil;
end

function SDFSubtitleNodeBehavior:_UpdateNode(node,deltaTs)
    if node:isActiveHierarchy() == true then
        if node.Update then
            node:Update(deltaTs)
        end
        local subNodes = node:GetChildrens();
        for k,v in pairs(subNodes) do
            self:_UpdateNode(v, deltaTs);
        end
    end
end

function SDFSubtitleNodeBehavior:_UpdateWallTs(pts)
    local deltaTs = 0.0;
    if _PLATFORM_WINDOWS then --windows测试代码
        deltaTs = pts;
    else
        deltaTs = (pts - self.lastpts) * 0.001;
        self.lastpts = pts;
    end
    -- LOG("MixNodeBehavior _UpdateWallTs" .. tostring(pts) .. "self.lastpts" .. tostring(self.lastpts) .. "def" .. tostring(deltaTs));
    self:_UpdateNode(self.node, deltaTs);
end

function SDFSubtitleNodeBehavior:_GetAllRenders(node, renders)
  local childs = node:GetChildrens();
  for _,v in pairs(childs) do
    local render = v:GetComponent(apolloengine.Node.CT_RENDER);
    if render then
      table.insert(renders, render);
    end
    self:_GetAllRenders(v, renders);
  end
end

function SDFSubtitleNodeBehavior:_GetAllScriptComponent(node, scriptComs)
  local childs = node:GetChildrens();
  for _,v in pairs(childs) do
    local scriptCom = v:GetComponent(apolloengine.Node.CT_SCRIPT);
    if scriptCom then
      table.insert(scriptComs, scriptCom);
    end
    self:_GetAllScriptComponent(v, scriptComs);
  end
end

function SDFSubtitleNodeBehavior:_GetRect()
  local rect = mathfunction.vector2(0,0);
  local root = self.sdfCom:GetHostNode();
  local scriptComs = {};
  self:_GetAllScriptComponent(root, scriptComs);
  LOG("get all scriptComs" .. #scriptComs)
  for i = 1, #scriptComs do
    if scriptComs[i] ~= nil then
      local paraSet = scriptComs[i].Instances
      for scrKey,scrValue in pairs(paraSet) do
        if scrValue.GetMaxRect then
          local currentRect = scrValue:GetMaxRect();
          if rect:LengthPow() < currentRect:LengthPow() then
            rect = currentRect;
          end
        end
      end
    end
  end
  if rect:LengthPow() < self.sdfCom:GetUnfixedRect():LengthPow() then
    rect = self.sdfCom:GetUnfixedRect();
  end
  return rect;
end

function SDFSubtitleNodeBehavior:_RequestEmojiImage()
    LOG("=====> _RequestEmojiImage")
    local unicodes = self.sdfCom:GetEmojiUnicodes()
    if #unicodes <= 0 then
        LOG("NOT EMOJI EXISTED");
        self.sdfCom:RenderText()
        return
    end

    if TC:HasTransaction(self.emojiRequestId) then
        LOG("WARNING: has pending work!!!!")
        self.sdfCom:RenderText()
        return
    end

    self.emojiRequestId = self:_GetEmojiRequestId()

    local isFirstItem = true;
    local jsonRequest = "{\"emoji\":[";
    for i=1, #unicodes do
        if isFirstItem then
            isFirstItem = false
        else
            jsonRequest = jsonRequest .. ","
        end
        jsonRequest = jsonRequest .. tostring(unicodes[i])
    end
    jsonRequest = jsonRequest .. "]}"

    TC:Request(self, self.emojiRequestId, REQUEST_TYPE, EMOJI_MAX_DURATION, jsonRequest)
    LOG("Request id = " .. tostring(self.emojiRequestId) .. ", " .. jsonRequest)
    self.emojiRequestTimeOffset = 0
end

function SDFSubtitleNodeBehavior:_GetEmojiRequestId()
    G_EmojiRequestId = G_EmojiRequestId + 1
    return G_EmojiRequestId
end

function SDFSubtitleNodeBehavior:_IsTimeOut(deltaMs)
    if self.emojiRequestTimeOffset == 0 then    --暂停渲染后这时再继续渲染deltaMs会特别大，需要先过滤掉
        self.emojiRequestTimeOffset = 30
        return false
    end
    self.emojiRequestTimeOffset = self.emojiRequestTimeOffset + deltaMs
    return self.emojiRequestTimeOffset >= EMOJI_MAX_DURATION
end

function SDFSubtitleNodeBehavior:_TimeOut()
    LOG("====> SDFSubtitleNodeBehavior::_TimeOut")
    self.sdfCom:RenderText()
end

function SDFSubtitleNodeBehavior:_Ready(jsonResult)
    LOG("====> SDFSubtitleNodeBehavior::_Ready")
    local jsonTbl = venusjson.LoadJsonString(jsonResult)

    if jsonTbl["emoji"] == nil then
        LOG("ERROR: invalid json: " .. jsonResult)
        self.sdfCom:RenderText();
        return
    end

    local unicodes = jsonTbl["emoji"]
    for i = 1, #unicodes do
        LOG("handleId = " .. tostring(unicodes[i]))
    end
    if #unicodes > 0 then
        local ret = likeapp.TextureStreamRepo:GetTextureStreams(self.emojiRequestId, unicodes)
        local hasEmoji = false
        for k,v in pairs(ret) do
            if v then
                hasEmoji = true
                break
            end
        end
        if hasEmoji then
            LOG("to call SetEmojiBufferPool")
            self.sdfCom:SetEmojiBufferPool(ret)
        end
    else
        LOG("ERROR: not emoji callback")
    end

    self.sdfCom:RenderText();
    self:_RequestRefreshFrame()
end

function SDFSubtitleNodeBehavior:_RequestRefreshFrame()
    self.REFRESH_FRAME_ID = self.REFRESH_FRAME_ID + 1
    TC:Request(self.REFRESH_FRAME_FUNCTION, self.REFRESH_FRAME_ID, BD.RequestType.REFRESH_FRAME, 1, "") --请求刷新画面，不然可能没有显示出来
end

function SDFSubtitleNodeBehavior:SetTopRenderOrder()
  LOG("SDFSubtitleNodeBehavior:SetTopRenderOrder")
  local camera = self.mainCam[#self.mainCam];
  if camera then
    camera:SetSequence(TOP_CAMERA_SEQUENCE);
  end
end

function SDFSubtitleNodeBehavior:SetOriginRenderOrder()
  LOG("SDFSubtitleNodeBehavior:SetTopRenderOrder")
  local camera = self.mainCam[#self.mainCam];
  if camera then
    LOG("Set Camera Origin Sequence " .. tostring(self.cameraSequence));
    camera:SetSequence(self.cameraSequence);
  end
end

return SDFSubtitleNodeBehavior;