local Object        = require "classic"
local venuscore     = require "venuscore"
local venusjson     = require "venusjson"
local libmasquerade = require "libmasque"
local apolloengine  = require "apollocore"
local utils         = require "behavior.cartoon_behavior.utils"

local FaceTransfer = venuscore.VenusBehavior:extend("FaceTransferBehavior")

function FaceTransfer:new()
  FaceTransfer.super.new(self)
  self.Config = ""
  self.Inited = false
  self.SaveModel = 1
  self.SavePath = ""
  self.FaceTransferedDelegates = {}
  self.FaceTransferStartDelegates = {}
  self.faceTransfer = libmasquerade.FaceTransfer()
end

function FaceTransfer:Clear()
  self.faceTransfer = nil
end

function FaceTransfer:SetConfig(value)
  self.Config = value
  self.Inited = false
  self:_OnConfigChanged()
end

function FaceTransfer:GetConfig()
  return self.Config
end

function FaceTransfer:_OnAwake()
  self.morphComp = self.Node:GetComponent(apolloengine.Node.CT_MORPH)
  self.renderComp = self.Node:GetComponent(apolloengine.Node.CT_RENDER)
  self:_OnConfigChanged()
end

function FaceTransfer:_OnConfigChanged()

  if self.morphComp == nil then
    return
  end
  
  if self.Config == nil or self.Config == "" then
    self.morphTarget = "identity"
  else  
    local configStr = venusjson.LaodJsonFile(self.Config);
    if configStr.morpherTarget ~= nil then
      self.morphTarget = configStr.morpherTarget
    else  
      local identityGroup = configStr.identityMorperGroup or "identity"
      self.expressionGroup = configStr.expressionMorperGroup or "expression"
      
      if self.faceTransfer:SetMorpherTarget(self.morphComp, identityGroup, self.expressionGroup) and
        self.faceTransfer:AddConstrainConfig(configStr) then
          self.dualTarget = true
          self.Inited = true
      end
    end
  end
  
  if self.morphTarget ~= nil then
    self.dualTarget = false
    self.morphGroupIdx = self.morphComp:GetGroupIndexByName(self.morphTarget)
    self.Inited = (self.morphGroupIdx ~= -1) 
  end
end

function FaceTransfer:GetFaceData()
  if self.morphComp == nil then
    return nil
  end
  
  local group = self:GetTargetGroup()
  local grpIndex = self.morphComp:GetGroupIndexByName(group)
  local vtxStream = self.renderComp:GetVertexStream()
  local vertexData = vtxStream:GetVertexData()
  
  local channelVertexData = {}
  for i = 0, self.morphComp:GetNumberOfChannel(grpIndex) - 1 do
    local channelIndex = self.morphComp:GetChannelVertices(grpIndex, i, 0)
    local channelName = self.morphComp:GetChannelName(grpIndex, i)
    channelVertexData[channelName] = channelIndex
  end
  return vertexData, channelVertexData
end

function FaceTransfer:GetTargetGroup()
  local group = nil
  if self.dualTarget then 
    group = self.expressionGroup
  else
    group = self.morphTarget
  end
  return group
end

function FaceTransfer:_OnTransferStart()
  local vertexData, channelVertexData = self:GetFaceData()
  if vertexData == nil then
    return 
  end 
  
  for obj, func in pairs(self.FaceTransferStartDelegates) do
    func(obj, vertexData,channelVertexData)
  end
end

function FaceTransfer:_OnTransferFinished()
  local vertexData, channelVertexData = self:GetFaceData()
  if vertexData == nil then
    return 
  end
  
  if self.SaveModel == 2 and self.SavePath ~= nil then
    local idxStream = self.renderComp:GetIndexStream()
    local indicesData = idxStream:GetIndicesData()
  
    self:_SaveModel(self.SavePath, vertexData, indicesData)
  end

  for obj, func in pairs(self.FaceTransferedDelegates) do
    func(obj, vertexData,channelVertexData)
  end
end

function FaceTransfer:RegisterFaceTransferedEvent(sender, func)
  self.FaceTransferedDelegates[sender] = func;
end

function FaceTransfer:RegisterFaceTransferStartEvent(sender, func)
  self.FaceTransferStartDelegates[sender] = func;
end

function FaceTransfer:TransferWeight(weights_dict)
  if self.Inited == false then
    return
  end
  
  self:_OnTransferStart()

  if self.dualTarget then
    if self.faceTransfer:Transfer(weights_dict, true) == false then
      return
    end
    
    self:_OnTransferFinished()
  else
    local lowercase_weightsdict = {}
    for n, v in pairs(weights_dict) do
      lowercase_weightsdict[n:lower()] = v
    end
    
    for i = 1, self.morphComp:GetNumberOfChannel(self.morphGroupIdx) do      
      local cname = self.morphComp:GetChannelName(self.morphGroupIdx, i - 1)
      if lowercase_weightsdict[cname] then
        self.morphComp:SetChannelPercents(self.morphGroupIdx, i - 1, lowercase_weightsdict[cname])
      end
    end
    self.transfer = true
  end
end

function FaceTransfer:_OnUpdate(delta)
  if self.Inited == false then
    return
  end
  
  if not self.dualTarget then
    if self.transfer then
      self:_OnTransferFinished()
      self.transfer = false
    end
  end
end

function FaceTransfer:_SaveModel(path, vertices, tris)
  local fp = io.open(path, "w")
  
  for i = 1, #vertices do
    fp:write("v "..vertices[i][1].." " ..vertices[i][2].." "..vertices[i][3].."\n")
  end
  
  for i = 1, #tris/3 do
    fp:write("f "..(tris[3*i-2]+1).." " ..(tris[3*i-1]+1).." "..(tris[3*i]+1).."\n")
  end
  
  fp:close()
end

FaceTransfer:MemberRegister("Config",  
  venuscore.ScriptTypes.FilePathType(
    {"json"},
    FaceTransfer.GetConfig,
    FaceTransfer.SetConfig));

FaceTransfer:MemberRegister("SaveModel",  
 venuscore.ScriptTypes.ComboType(
    {
      {
        key = "FALSE",
        value = 1
      },
      {
        key = "TRUE",
        value = 2
      }
    }
))

FaceTransfer:MemberRegister("SavePath")

return FaceTransfer