local venuscore = require "libvenuscore"
local Types = require "venuscore.rtti.types"
local BlueNode = require "bluecore.bluenode"
local BlueSelf = require "bluecore.blueself"
local EventDispatcher = require "bluecore.eventdispatcher"
local BD = require "bluecore.bluedefined"
local CV = require "computervisionfunction"
local BU = require "bluecore.blueutility"
local RecognitionIris = BlueNode:extend();

---- 游戏运行模式
function RecognitionIris:new(...)
  RecognitionIris.super.new(self, ...)
  self._formatOutputArray = {}
end

function RecognitionIris:_OnUpdate(args)
  self._formatOutputArray = args
  return self._formatOutputArray;
end

function RecognitionIris:_OnNextBranch()
  if self._formatOutputArray ~= nil and next(self._formatOutputArray) then
    return 1
  else
    return 0
  end
end

function RecognitionIris:RegisterTriggerFunc(func, bluePrintSelf)
  self:_GetInput()

  local comp = self.component or self.inputArgs[1]

  EventDispatcher:RegisterCallback(
          comp:GetContentPath(),
          CV.RecognitionComponent.cvIrisDetection ,
          func,
          bluePrintSelf)

end


function RecognitionIris:_OnCompile()
  self.compileResult = nil
  if RecognitionIris.super._OnCompile(self) then

    local comp = self.component

    -- Step.0 判断反射面板有override
    if comp ~= nil then
      if venuscore.isNil(comp) then
        self.compileResult = BD.EHINT.COMP_LOST
        return false
      end
    end

    if comp == nil then
      -- Step.1 确保Component引脚有连接
      if #self.inputs[1].links ~= 0 then
        local bluePinLink = self.inputs[1].links[1] -- 输入引脚只允许一个连接
        local otherNode = self.graph:GetNode(bluePinLink.nodeUid);

        -- Step.2 确保上一个节点是 引用组件类型 的节点
        if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp == nil then
          self.compileResult = BD.EHINT.NOT_LINK_TO_COMP
          return false
        end

        if venuscore.isNil(otherNode.comp) then
          self.compileResult = BD.EHINT.COMP_LOST
          return false
        end
        comp = otherNode.comp
      else
        self.compileResult = BD.EHINT.PIN_1_EMPTY;
        return false
      end
    end

    -- Step.3 RecognitionComponent没有激活当前的检测类型
    local hasType = self:_HasType(comp);
    if not hasType then
      self.compileResult = BD.EHINT.NOT_ENABLE;
    end

    return true ;
  end
  return false;
end

function RecognitionIris:_HasType(comp)
  return bit.band(comp.Type, CV.RecognitionComponent.cvIrisDetection) ~= 0
end

function RecognitionIris:_OnDeserialize()
end


-- 蓝图节点反射面板相关
function RecognitionIris:_SetComp(comp)
  self.component = comp ;
end

function RecognitionIris:_GetComp()
  return self.component
end

function RecognitionIris:ReflectFilterFtnOnPrefab(obj, member, value)
  return member == "component"
end

function RecognitionIris:LiteralMismatchOnCompile(pin, pinInfo)
  if pin.argId ==  1 and BU:isBlueSelf(pin.literal) and  not venuscore.isNil(self.component) then
    return true
  end
  return false
end

RecognitionIris:MemberRegister("component",
        Types.ReferenceType(CV.RecognitionComponent:RTTI(),
                RecognitionIris._GetComp, RecognitionIris._SetComp));


---- 编辑器模式
if _KRATOSEDITOR then

function RecognitionIris:_OnLink(selfPin, otherNode, otherPin)
  if BU:IsInputPin(selfPin) and selfPin.argId == 1 then
    if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp then
      if venuscore.isNil(otherNode.comp) then
        ERROR(BD.EHINT.COMP_LOST)
      else
        local hasType = self:_HasType(otherNode.comp);
        if not hasType then
          ERROR(BD.EHINT.NOT_ENABLE)
        end
      end
    else
      ERROR(BD.EHINT.NOT_LINK_TO_COMP)
    end
  end
end

function RecognitionIris:_OnUnLink(unlinkPin)
  if unlinkPin.argId == 1 then -- 断开是第一个引脚
     ERROR(BD.EHINT.PIN_1_EMPTY)
  end
end

function RecognitionIris:_OnCreateWithEditor()
  RecognitionIris.super._OnCreateWithEditor(self);
end

function RecognitionIris:_OnUpdateByEditor()
  RecognitionIris.super._OnUpdateByEditor(self);
  self:_OnCompile();
end

end

RecognitionIris:RegisterInput(1, CV.RecognitionComponent:RTTI(), "component", "监听组件", BlueSelf());

RecognitionIris:RegisterOutput(1, BU:GetArrayRtti(Types.IrisType), "result", "检测结果");

RecognitionIris:RegisterExecOutput(1, "OnTrigger", "执行");

RecognitionIris:SetFunctionName("Iris");

RecognitionIris:SetFunctionType(BD.EVENT_FUNCTION_NODE);

RecognitionIris:SetEventType(BD.EventType.RegisterCallback);

return RecognitionIris;

