local libvenuscore = require "libvenuscore"
local Types = require "venuscore.rtti.types"

local BD = require "bluecore.bluedefined"
local BS = require "bluecore.bluesingle"
local BU = require "bluecore.blueutility"

local BlueContext = libvenuscore.Object:extend("BlueContext");

-- 当前蓝图类上下文 支持创建的节点类:上下文菜单和分类上下文菜单
function BlueContext:new(graph)
  self.graph = graph
  self.BluePrintSpecClass = {};       -- 动态生成的节点类(属于这个蓝图类) 反序列化Bluenode时候补全
  self.myContextNodeAll = {}          -- 当前蓝图类动态创建的节点类(所有) 反序列化BluePrintSpecClass时候补全
  self.myContextNodeClassify = {}     -- 当前蓝图类动态创建的节点类(分类) 反序列化BluePrintSpecClass时候补全

  if _KRATOSEDITOR then -- 非编辑器模式 是不用加载菜单上下文的

    local mgr = BS.GetBlueManager();
    local baseAllNode = mgr:GetAllNode()
    self.myContextNodeAll = table.deepcopy(baseAllNode);

    local baseClassifyNode = mgr:GetClassifyNode();
    self.myContextNodeClassify = table.deepcopy(baseClassifyNode);

  end
end

-- 返回当前蓝图类,支持的所有节点类
function BlueContext:GetAllNode()
  return self.myContextNodeAll
end

-- 返回当前蓝图类,支持的所有节点类
function BlueContext:GetClassifyNode()
  return self.myContextNodeClassify
end

-- 获取模板特化的节点类
function BlueContext:GetNodeClass(template_path, cls_spec_data)

  if self.BluePrintSpecClass[template_path] == nil then
    if _KRATOSEDITOR then WARNING("[GetNodeClass] cls_path not valid(des is ok)") end
    return nil
  end

  local signature = ""
  for k, v in pairs(cls_spec_data) do
    signature = signature..tostring(k)..":"..tostring(v)..","
  end

  if _KRATOSEDITOR then
     LOG("[GetNodeClass] template_path "..tostring(template_path).." cls_spec_data "..tostring(signature))
  end

  local id = libvenuscore.Utility:HashCode(signature) -- 应该不会重复?

  return self.BluePrintSpecClass[template_path][id]
end

-- 增加模板特化的节点类(需要给定spec_data)
function BlueContext:AddNodeClass(templatePath, clsSpecData, cls)

  if self.BluePrintSpecClass[templatePath] == nil then
    self.BluePrintSpecClass[templatePath] = {}
  end

  local signature = ""
  for k, v in pairs(clsSpecData) do
    signature = signature..tostring(k)..":"..tostring(v)..","
  end

  if _KRATOSEDITOR then
    LOG("[AddNodeClass] template_path "..tostring(templatePath).." cls_spec_data "..tostring(signature))
  end

  local id = libvenuscore.Utility:HashCode(signature)

  assert(  self.BluePrintSpecClass[templatePath][id] == nil,"[AddNodeClass] duplicate node class")

  self.BluePrintSpecClass[templatePath][id] = cls

end

-- 移除模板特化的节点类(需要给定spec_data)
function BlueContext:RemoveNodeClass(templatePath, clsSpecData)

  assert(self.BluePrintSpecClass[templatePath], "RemoveNodeClass template not found" )

  local signature = ""
  for k, v in pairs(clsSpecData) do
    signature = signature..tostring(k)..":"..tostring(v)..","
  end

  local id = libvenuscore.Utility:HashCode(signature)

  assert(self.BluePrintSpecClass[templatePath][id], "RemoveNodeClass class not found")

  local clazz =   self.BluePrintSpecClass[templatePath][id]

  self.BluePrintSpecClass[templatePath][id] = nil

  return clazz
end

-- 增加变量节点类 并加到菜单上下文
function BlueContext:_SetupVariableNodeClass(varinfo)

  assert(varinfo, "_SetupVariableNodeClass varinfo")
  local pathOfTemplateSet = "bluecore.variable.setvariable"
  local pathOfTemplateGet = "bluecore.variable.getvariable"

  local rtti = varinfo:GetRtti();

  local specData = varinfo:GetSpecData();

  local templateSet = require(pathOfTemplateSet)
  local clazzSet = templateSet:extend(self.graph, specData, rtti);

  local templateGet = require(pathOfTemplateGet)
  local clazzGet = templateGet:extend(self.graph, specData, rtti);

  -- 记录新建的NodeClass
  self:AddNodeClass(pathOfTemplateSet, specData, clazzSet);
  self:AddNodeClass(pathOfTemplateGet, specData, clazzGet);

  -- NodeClass加入到上下文菜单
  self:_AddClazzIntoContext(clazzSet)
  self:_AddClazzIntoContext(clazzGet)

  return clazzSet, clazzGet
end

-- 增加变量节点类 但不加到菜单上下文
-- 目前用在反序列化 把两个步骤分开
-- 如果反序列化过程中 有节点是模板节点类生成的,就已经会生成模板节点类(不完整,还差rtti)
function BlueContext:_RegisterVariableNodeClass(varinfo)

  local specData = {} specData[BD.RefVarName] = varinfo:GetName()
  local pathOfTemplateSet = "bluecore.variable.setvariable"
  local pathOfTemplateGet = "bluecore.variable.getvariable"

  local clazzSet =  self:GetNodeClass(pathOfTemplateSet, specData)
  if clazzSet == nil then
    local template = require(pathOfTemplateSet)
    clazzSet = template:extend(self.graph, specData)
    self:AddNodeClass(pathOfTemplateSet, specData, clazzSet)
  end

  local clazzGet =  self:GetNodeClass(pathOfTemplateGet, specData)
  if clazzGet == nil then
    local template = require(pathOfTemplateGet)
    clazzGet = template:extend(self.graph, specData)
    self:AddNodeClass(pathOfTemplateGet, specData, clazzGet)
  end

  return clazzSet, clazzGet
end

-- 删除变量节点类 并从菜单上下文移除
function BlueContext:_RemoveVariableNodeClass(varinfo)

  local pathOfTemplateSet = "bluecore.variable.setvariable"
  local pathOfTemplateGet = "bluecore.variable.getvariable"
  local specData = varinfo:GetSpecData();
  local clazzSet = self:GetNodeClass(pathOfTemplateSet, specData);
  local clazzGet = self:GetNodeClass(pathOfTemplateGet, specData);

  self:_RemoveClazzFromContext(clazzSet);
  self:_RemoveClazzFromContext(clazzGet);

  -- 从蓝图移除变量节点类
  self:RemoveNodeClass(pathOfTemplateSet, specData);
  self:RemoveNodeClass(pathOfTemplateGet, specData);

  -- 返回用于删除节点
  return clazzSet, clazzGet;
end

function BlueContext:onPropertyChange(publisher, property, oldValue, newValue)

  if property == BD.VarProp.Name then
    -- 三个地方需要修改 BluePrintSpecClass myContextNodeAll中的ctxInfo(myContextNodeClassify引用同一个)

    local pathOfTemplateSet = "bluecore.variable.setvariable"
    local pathOfTemplateGet = "bluecore.variable.getvariable"

    local varInfo = publisher
    local specData = varInfo:GetSpecData();
    local clazzSet = self:GetNodeClass(pathOfTemplateSet, specData);
    local clazzGet = self:GetNodeClass(pathOfTemplateGet, specData);

    local newDisplayName = newValue

    local chk1 = false ; local chk2 = false
    for _, ctxNodeInfo in pairs(self.myContextNodeAll[BD.Catalog.VAR] ) do
      if ctxNodeInfo.luaPath  == clazzSet then
        WARNING("Context Variable Node Class Change:"..tostring(ctxNodeInfo.blueNodeName).."->"..tostring(newDisplayName))
        ctxNodeInfo.blueNodeName = "Set "..newDisplayName; chk1 = true ;
      elseif ctxNodeInfo.luaPath == clazzGet then
        WARNING("Context Variable Node Class Change:"..tostring(ctxNodeInfo.blueNodeName).."->"..tostring(newDisplayName))
        ctxNodeInfo.blueNodeName = "Get "..newDisplayName; chk2 = true ;
      end
    end
    assert(chk1 and  chk2, "context clazz not complete change")
    self.graph:SetModified();

  elseif property == BD.VarProp.BaseType or property == BD.VarProp.IsArray  then
    -- rtti修改 需要更新 myContextNodeClassify
    local varInfo = publisher

    local pathOfTemplateSet = "bluecore.variable.setvariable"
    local pathOfTemplateGet = "bluecore.variable.getvariable"
    local specData = varInfo:GetSpecData();
    local clazzSet = self:GetNodeClass(pathOfTemplateSet, specData);
    local clazzGet = self:GetNodeClass(pathOfTemplateGet, specData);

    local ctxInfoGet
    local ctxInfoSet
    for _, ctxNodeInfo in pairs(self.myContextNodeAll[BD.Catalog.VAR]) do
      if ctxNodeInfo.luaPath == clazzSet then
        ctxInfoGet = ctxNodeInfo
      elseif ctxNodeInfo.luaPath == clazzGet then
        ctxInfoSet = ctxNodeInfo
      end
      if ctxInfoGet and ctxInfoSet then break end
    end
    assert(ctxInfoGet and ctxInfoSet, "[_OnPropertyChange] info not found")

    self:_RemoveClazzFromContext(clazzSet);
    self:_RemoveClazzFromContext(clazzGet);

    self:_AddClazzIntoContext(clazzSet, ctxInfoGet);
    self:_AddClazzIntoContext(clazzGet, ctxInfoSet);

    self.graph:SetModified(); -- 通知蓝图已修改
  end
end

function BlueContext:_OnDeserializePost()
  if self.BluePrintSpecClass ~= nil then
    for _, paths in pairs(self.BluePrintSpecClass) do
      for __, clazz in pairs(paths) do
        -- 模板节点类的反序列化后处理 (目前变量节点类通过这步更新RTTI信息)
        if clazz._OnClassDeserializePost then
          clazz:_OnClassDeserializePost()
        end
        -- 模板节点类加入到上下文菜单(类信息完整 可以加入上下文)
        if _KRATOSEDITOR then --非编辑器不用恢复上下文
          self:_AddClazzIntoContext(clazz);
        end
      end
    end
  end
end

function BlueContext:_RemoveClazzFromContext(clazz)

  local catalogName = clazz:GetFunctionType() == BD.VARIABLE_NODE and BD.Catalog.VAR or BD.Catalog.OTHER

  local catalog = self.myContextNodeAll[catalogName]
  for key , ctxInfo in pairs(catalog) do
    if ctxInfo.luaPath == clazz then
      catalog[key] = nil
    end
  end

  catalog = self.myContextNodeClassify[BD.BLUE_MGR_DIR][catalogName]

  BU:RemoveClassifyNodeInCatalog(clazz, catalog);

end

-- 如果给定info 就不会认为是一个新的info 不会增加到myContextNodeAll
function BlueContext:_AddClazzIntoContext(clazz, clazzinfo)

  local ctxInfo = clazzinfo or
          {
            luaPath = clazz,
            blueNodeName = clazz:GetName(),
            blueNodeType = clazz:GetFunctionType()
          }

  -- 特化类/动态生成的节点类 放在单独目录方便删除
  local catalogName = clazz:GetFunctionType() == BD.VARIABLE_NODE and BD.Catalog.VAR or BD.Catalog.OTHER

  self.myContextNodeAll[catalogName] = self.myContextNodeAll[catalogName] or {}

  local catalog1 = self.myContextNodeAll[catalogName]

  for _, ctxInfoTemp in pairs(catalog1) do
      assert(ctxInfoTemp ~= ctxInfo,"_AddClazzIntoContext twice ");
  end
  table.insert(catalog1, ctxInfo)


  self.myContextNodeClassify[BD.BLUE_MGR_DIR] = self.myContextNodeClassify[BD.BLUE_MGR_DIR] or {}
  self.myContextNodeClassify[BD.BLUE_MGR_DIR][catalogName] = self.myContextNodeClassify[BD.BLUE_MGR_DIR][catalogName] or {}

  local catalog = self.myContextNodeClassify[BD.BLUE_MGR_DIR][catalogName]

  local inputInfos = clazz:GetInputsType()
  BU:SetClassifyNode(inputInfos, BD.BLUE_MGR_INPUT, ctxInfo, catalog);

  local outputInfos = clazz:GetOutputsType()
  BU:SetClassifyNode(outputInfos, BD.BLUE_MGR_OUTPUT, ctxInfo, catalog);

  local execInputInfos = clazz:GetExecInput()
  BU:SetClassifyNode(execInputInfos, BD.BLUE_MGR_INPUT, ctxInfo, catalog);

  local execOutputInfos = clazz:GetExecOutput()
  BU:SetClassifyNode(execOutputInfos, BD.BLUE_MGR_OUTPUT, ctxInfo, catalog);

end

return BlueContext