--@author      : Jin Shiyu
--@date        : 2020-11-02
--@description : lua script for face beauty
--@version     : 1.0

local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local renderqueue   = require "apolloutility.renderqueue"
local mathfunction  = require "mathfunction"
local venuscore     = require "venuscore"
local defined       = require "apolloutility.defiend"
local quad_render   = require "facebeautyutils.face_beauty_quad_render"
local face_defined  = require "facebeautyutils.defined"
local cv            = require "computervisionfunction"
local autotoucher   = require "autotoucherfunction"

--[[
    Face beauty on one frame, including brighten eye, nasolabial removal, eyebag removal and whiten teeth
    NOTE:
    It can be used in testshell main and editor behavior which should offer one main camera.
]]

local FaceBeauty = {};

local BRIGHT_EYE   = 1;
local NASOLABIAL   = 2;
local EYEBAG       = 3;
local WHITEN_TEETH = 4;

function FaceBeauty:Create(layer_sequence, scene)
    self.mask_texture_path         = "docs:facebeauty/resource/standard_face_v4.png"
    self.teeth_lookup_texture_path = "docs:facebeauty/resource/whiten_teeth_lut.png"
    self.eye_mask_texture_path     = "docs:facebeauty/resource/eye_face_mask.png"

    --[[
        1: current strength of brighten eye (0-1)
        2: current strength of nasolabial removal (0-1)
        3: current strength of eyebag removal (0-1)
        4: current strength of whiten teeth (0-1)
    ]]
    self.beauty_strengths = { 0.0, 0.0, 0.0, 0.0};

    self.scaled_factor    = 0.5;            -- scaled factor of downsample
    self.offset_factor    = 2.8958829;      -- factor of texture sampling step

    self.max_face_num     = 4;

    self.virtual_node     = nil;

    self.is_show          = true;
    self.initialized      = false;
    self.need_update      = false;
    self.preUpdateResult  = 0;
    self.updateResult     = 0;
    self:Initialize(layer_sequence, scene);
end

function FaceBeauty:UpdateTextureSize(frame_width, frame_height)

  local scaled_width  = frame_width * self.scaled_factor;
  local scaled_height = frame_height * self.scaled_factor;

  local scaler_updated     = self.image_scaler:Update(scaled_width, scaled_height);
  local vertical_updated   = self.vertical_box_blur:Update(scaled_width, scaled_height);
  local horizontal_updated = self.horizontal_box_blur:Update(scaled_width, scaled_height);

  -- update
  if scaler_updated and vertical_updated and horizontal_updated then
      self.vertical_box_blur:SetParameter("UNIFORM_INPUT_TEXTURE", self.image_scaler:GetResultImage());
      self.vertical_box_blur:SetParameter("UNIFORM_TEXEL_WIDTH_OFFSET", mathfunction.vector1(0));
      self.vertical_box_blur:SetParameter("UNIFORM_TEXEL_HEIGHT_OFFSET", mathfunction.vector1(self.offset_factor / scaled_height));

      self.horizontal_box_blur:SetParameter("UNIFORM_INPUT_TEXTURE", self.vertical_box_blur:GetResultImage());
      self.horizontal_box_blur:SetParameter("UNIFORM_TEXEL_WIDTH_OFFSET", mathfunction.vector1(self.offset_factor / scaled_width));
      self.horizontal_box_blur:SetParameter("UNIFORM_TEXEL_HEIGHT_OFFSET", mathfunction.vector1(0));
  end

end

function FaceBeauty:LoadTexture(texture_path)
    local texture = apolloengine.TextureEntity();
    texture:PushMetadata(apolloengine.TextureFileMetadata(
                                   apolloengine.TextureEntity.TU_STATIC,
                                   apolloengine.TextureEntity.PF_AUTO,
                                   1, false,
                                   apolloengine.TextureEntity.TW_REPEAT,
                                   apolloengine.TextureEntity.TW_REPEAT,
                                   apolloengine.TextureEntity.TF_LINEAR,
                                   apolloengine.TextureEntity.TF_LINEAR,
                                   venuscore.IFileSystem:PathAssembly(texture_path)));
    texture:SetJobType(venuscore.IJob.JT_SYNCHRONOUS);
    texture:CreateResource();
    return texture;
end

function FaceBeauty:Initialize(layer_sequence, scene)

    -- load texture
    self.mask_texture         = self:LoadTexture(self.mask_texture_path);
    self.teeth_lookup_texture = self:LoadTexture(self.teeth_lookup_texture_path);
    self.eye_mask_texture     = self:LoadTexture(self.eye_mask_texture_path);

    -- setup quad render
    local initial_width  = 360;
    local initial_height = 640;
    local max_sequence   = layer_sequence - self.max_face_num;

    self.image_scaler        = quad_render("face_beauty_identity", max_sequence - 3, "identity_layer", "docs:facebeauty/material/identity.material", initial_width, initial_height);
    self.vertical_box_blur   = quad_render("face_beauty_vertical", max_sequence - 2, "vertical_blur_layer", "docs:facebeauty/material/box_blur.material", initial_width, initial_height);
    self.horizontal_box_blur = quad_render("face_beauty_horizontal", max_sequence - 1, "horizontal_blur_layer", "docs:facebeauty/material/box_blur.material", initial_width, initial_height);

    self.vertical_box_blur:SetParameter("UNIFORM_INPUT_TEXTURE", self.image_scaler:GetResultImage());
    self.vertical_box_blur:SetParameter("UNIFORM_TEXEL_WIDTH_OFFSET", mathfunction.vector1(0));
    self.vertical_box_blur:SetParameter("UNIFORM_TEXEL_HEIGHT_OFFSET", mathfunction.vector1(self.offset_factor / initial_height));

    self.horizontal_box_blur:SetParameter("UNIFORM_INPUT_TEXTURE", self.vertical_box_blur:GetResultImage());
    self.horizontal_box_blur:SetParameter("UNIFORM_TEXEL_WIDTH_OFFSET", mathfunction.vector1(self.offset_factor / initial_width));
    self.horizontal_box_blur:SetParameter("UNIFORM_TEXEL_HEIGHT_OFFSET", mathfunction.vector1(0));

    self.CaptureInput = apolloengine.TextureEntity();
    self.CaptureInput:PushMetadata(apolloengine.TextureReferenceMetadata(apolloengine.DeviceResource.DEVICE_CAPTURE));
    self.CaptureInput:CreateResource();

    self.virtual_node       = apollonode.VirtualNode();
    local classify          = self.virtual_node:CreateComponent(apolloengine.Node.CT_CV_CLASSIFY);
    classify.Enable         = true;
    classify.Mode           = cv.IVisionComponent.VIDEO;
    classify.FaceExpression = true;
    classify:SetTexture(self.CaptureInput);
    self.classify = classify;

    local recognition   = self.virtual_node:CreateComponent(apolloengine.Node.CT_CV_RECOGNITION);
    recognition.Enable  = true;
    recognition.Mode    = cv.IVisionComponent.VIDEO;
    recognition.Type    = cv.RecognitionComponent.cvFace;
    recognition.ShareCV = true;
    recognition:SetTexture(self.CaptureInput);
    self.recognition    = recognition;

    self.autotoucher = autotoucher.AutoToucherFaceBeauty();
    local res = self.autotoucher:InitFaceBeautyRenderNode(scene,
                                                          self.mask_texture,
                                                          self.teeth_lookup_texture,
                                                          self.eye_mask_texture,
                                                          layer_sequence);

    if res > 0 then
        ERROR("[FaceBeauty] InitFaceBeautyRenderNode return value: "..res);
        self.initialized = false;
    end

    self.initialized = true;
end

function FaceBeauty:SetShow(is_show)
    self.is_show = is_show;
    self.autotoucher:SetFaceBeautyShow(is_show);
end


--[[
    This function should be called before Update(), it will
    1. disable all render first
    2. enable render and return true if has face or valid strength
]]
function FaceBeauty:PreUpdate(def)

    if not self.initialized then
        ERROR("[FaceBeauty] not initialized");
        return false;
    end

    -- reset state and disable all render first
    self.need_update = false;
    self.image_scaler:SetShow(false);
    self.vertical_box_blur:SetShow(false);
    self.horizontal_box_blur:SetShow(false);

    -- check if enabled
    if self.is_show == false then
        -- LOG("[FaceBeauty] don't show");
        return false;
    end

    if _KRATOSEDITOR then

    else
        self.virtual_node.node:Update(def);
    end

    local res = self.autotoucher:PreUpdateFaceBeautyRenderNode(self.recognition, self.classify);

    self.need_update = (res == 0);

    if res ~= 0 then
        if self.preUpdateResult ~= res then
            ERROR("[FaceBeauty] PreUpdateFaceBeautyRenderNode return value: "..res);
        end
    end
    self.preUpdateResult = res;

    return self.need_update;
end


function FaceBeauty:Update(sequence, texture)

    if not self.need_update then
        LOG("[FaceBeauty] no need to update");
        return;
    end

    local max_sequence = sequence - self.max_face_num;
    self.image_scaler:SetSequence(max_sequence - 3);
    self.vertical_box_blur:SetSequence(max_sequence - 2);
    self.horizontal_box_blur:SetSequence(max_sequence - 1);

    -- update quad render
    local texture_size = texture:GetSize();
    self:UpdateTextureSize(texture_size:x(), texture_size:y());
    self.image_scaler:SetParameter("UNIFORM_INPUT_TEXTURE", texture);

    -- update face render
    local res = self.autotoucher:UpdateFaceBeautyRenderNode(texture,
                                                            self.image_scaler:GetResultImage(),
                                                            self.horizontal_box_blur:GetResultImage(),
                                                            sequence);

    if res ~= 0 then
        if self.updateResult ~= res then
            ERROR("[FaceBeauty] UpdateFaceBeautyRenderNode return value: "..res);
        end
    end

    self.updateResult = res;

end

function FaceBeauty:SetBeautyStrength(beauty_type, strength)

    self.autotoucher:SetFaceBeautyStrengthWithType(math.ceil(strength * 100.0), beauty_type - 1);
    return true;

end

function FaceBeauty:GetBeautyStrength(beauty_type)
    if beauty_type <= 0 or beauty_type > 4 then
        return 0.0;
    end
    return self.beauty_strengths[beauty_type];
end

function FaceBeauty:GetMaskTexturePath()
  return self.mask_texture_path;
end

function FaceBeauty:SetMaskTexturePath(value)
  self.mask_texture_path = value;
  self.mask_texture      = self:LoadTexture(self.mask_texture_path);
end

function FaceBeauty:GetTeethLookupTexturePath()
  return self.teeth_lookup_texture_path;
end

function FaceBeauty:SetTeethLookupTexturePath(value)
  self.teeth_lookup_texture_path = value;
  self.teeth_lookup_texture      = self:LoadTexture(self.teeth_lookup_texture_path);
end

function FaceBeauty:GetEyeMaskTexturePath()
  return self.eye_mask_texture_path;
end

function FaceBeauty:SetEyeMaskTexturePath(value)
  self.eye_mask_texture_path = value;
  self.eye_mask_texture_texture = self:LoadTexture(self.eye_mask_texture_path);
end

function FaceBeauty:Destroy()

    if self.initialized == true then
        if self.image_scaler ~= nil then
            self.image_scaler:Clear()
            self.image_scaler = nil;
        end
        if self.vertical_box_blur ~= nil then
            self.vertical_box_blur:Clear()
            self.vertical_box_blur = nil;
        end
        if self.horizontal_box_blur ~= nil then
            self.horizontal_box_blur:Clear()
            self.horizontal_box_blur = nil;
        end

        if self.virtual_node ~= nil then
            self.virtual_node:Destroy();
            self.virtual_node = nil;
        end

        if self.autotoucher ~= nil then
            self.autotoucher:ReleaseFaceBeautyRenderNode();
            self.autotoucher = nil;
        end

        self.preUpdateResult = 0;
        self.updateResult    = 0;
        self.initialized     = false;
    end

end

return FaceBeauty;