local Types = require "venuscore.rtti.types"

local BlueDynamicNode = require "bluecore.base.blue_dynamic_node"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"

local BlueAnyTypeNode = BlueDynamicNode:extend("BlueAnyTypeNode");
BlueAnyTypeNode.DEFAULT_GROUP = "default"

BlueAnyTypeNode:MemberRegister("pinGroup");

-- 用途:
-- 1.引脚数量不变 但是rtti/tips/name等会在编辑时修改 需要序列化保存引脚信息的节点类
-- 2.存在引脚rtti是any类型 任何引脚的连接 将会触发其他引脚的特化
--   pinGroup 只有在同一个组下的引脚 才会相互特化
--   pinGroup 作为特化和泛化的标记,只有在group中的引脚才会在link和unlink分别执行特化和泛化
function BlueAnyTypeNode:new(...)
  BlueAnyTypeNode.super.new(self,...)
  self.pinGroup = {}
  self:CreateGroup(BlueAnyTypeNode.DEFAULT_GROUP);
end

function BlueAnyTypeNode:CreateGroup(name)
  self.pinGroup[name] = {}
end

--unlink ExitGroup DestroyGroup(本函数未被验证)
function BlueAnyTypeNode:DestroyGroup(name)
  self.pinGroup[name] = nil -- anything to do? to be defined
end

-- register infoTable inputsInfo/outputsInfo
-- inputs
-- JoinGroup
-- add to graph.bluePrintAllList
-- link
function BlueAnyTypeNode:JoinGroup(name, pin)

  local grp = self.pinGroup[name];
  local gfo = { pinUid = pin.uid } -- 考虑序列化

  assert(grp, "group "..tostring(name).." not exists")

  table.insert(grp, gfo)

  local pinInfo = self:GetPinInfoByPin(pin)
  pinInfo.myGroup = grp -- 反序列化需补充上 不放在pin,因为pin会全部反序列化

  if pinInfo.rtti ~= Types.AnyArrayType and pinInfo.rtti ~= Types.AnyType then
    local rttiOfArray = pinInfo.rtti:isArray() and pinInfo.rtti or BU:GetArrayRtti(pinInfo.rtti)
    local rtti = pinInfo.rtti:isArray() and pinInfo.rtti:GetElementType() or pinInfo.rtti
    self:_WalkThroughGroup(grp, rtti, rttiOfArray);
  end
end

-- unlinks
-- deregister infoTable inputsInfo/outputsInfo
-- ExitGroup
-- inputs重建
-- remove from graph.bluePrintAllList
function BlueAnyTypeNode:ExitGroup(name, pinUid)
  assert(self.pinGroup[name], "group "..tostring(name).." not exists")
  local index = 1
  local exit  = false
  for _, gfo in pairs(self.pinGroup[name]) do
    if gfo.pinUid == pinUid then
      table.remove(self.pinGroup[name], index)
      exit = true
      break;
    else
      index = index + 1
    end
  end
  assert(exit, "ExitGroup "..tostring(pinUid).." not exit in group "..tostring(name))
end


-- 每次链接断开都会调用 多个链接 任何一个连接断开都回调
-- 如果不在Group中不做处理,如果需要特化的话,可以加入到default默认组 或者新增组
function BlueAnyTypeNode:_OnUnLink(unlinkPin)
  local pinInfo = self:GetPinInfoByPin(unlinkPin)
  local grp = pinInfo.myGroup ; --self:GetMyGroup(pinInfo);
  if grp ~= nil then
    if #unlinkPin.links == 0 then -- 当前引脚没有连接才会泛化
      local isAny = true ;
      for __, gfo in pairs(grp) do
        local pin = self:_GetPinByGfo(gfo);
        if unlinkPin ~= pin and #pin.links ~=0 then
          isAny = false ;
          break;
        end
      end
      if isAny then -- 当前引脚需要泛化 说明整个group都要泛化
        for __, gfo in pairs(grp) do
          local grpPinInfo = self:_GetPinInfoByGfo(gfo)
          grpPinInfo.rtti = grpPinInfo.rtti:isArray() and Types.AnyArrayType or Types.AnyType
          self:_GenerateLiteral(grpPinInfo);
        end
      end
    end
  end
end

-- 当连接pin之后 回调两个pin所在的Node  link_pin当前节点的pin信息
function BlueAnyTypeNode:_OnLink(selfPin, otherNode, otherPin)

  local pinInfo = self:GetPinInfoByPin(selfPin);
  if pinInfo.rtti == Types.AnyType or pinInfo.rtti == Types.AnyArrayType then
    local otherPinInfo = otherNode:GetPinInfoByPin(otherPin);
    if otherPinInfo.rtti ~= Types.AnyType and otherPinInfo.rtti ~= Types.AnyArrayType then
      if pinInfo.rtti:isArray() then
        pinInfo.rtti = otherPinInfo.rtti:isArray() and otherPinInfo.rtti or BU:GetArrayRtti(otherPinInfo.rtti)
      else
        pinInfo.rtti = otherPinInfo.rtti:isArray() and otherPinInfo.rtti:GetElementType() or otherPinInfo.rtti
      end

      self:_GenerateLiteral(pinInfo);

      if pinInfo.myGroup ~= nil then -- 特化其他pin 同group的pin 要么都泛化 要么都特化
        local rttiOfArray = otherPinInfo.rtti:isArray() and otherPinInfo.rtti or BU:GetArrayRtti(otherPinInfo.rtti)
        local rtti = otherPinInfo.rtti:isArray() and otherPinInfo.rtti:GetElementType() or otherPinInfo.rtti
        self:_WalkThroughGroup(pinInfo.myGroup, rtti, rttiOfArray);
      end
    end
  end
end

-- 对pin进行特化(如pin需特化,其所在组也要特化;为防止死循环,需要判断rtti是否any)
function BlueAnyTypeNode:_WalkThroughPin(selfPin, rtti, rttiOfArray)
  local pinInfo = self:GetPinInfoByPin(selfPin);
  if pinInfo.rtti == Types.AnyType or pinInfo.rtti == Types.AnyArrayType then
    pinInfo.rtti =  pinInfo.rtti:isArray() and rttiOfArray or rtti
    self:_GenerateLiteral(pinInfo);
    if pinInfo.myGroup ~= nil then
      self:_WalkThroughGroup(pinInfo.myGroup, rtti, rttiOfArray);
    end
  end
end

-- 对group进行特化
function BlueAnyTypeNode:_WalkThroughGroup(group, rtti, rttiOfArray)
  for __, gfo in pairs(group) do
    local pinInfo = self:_GetPinInfoByGfo(gfo)
    if pinInfo.rtti == Types.AnyType or pinInfo.rtti == Types.AnyArrayType then
      pinInfo.rtti = pinInfo.rtti:isArray() and rttiOfArray or rtti
      local pin = self:_GenerateLiteral(pinInfo);
      for _, link in pairs(pin.links) do -- 处理pin的连接 pin.links
        local otherNode = self.graph:GetNode(link.nodeUid);
        local otherPin = self.graph:GetPinById(link.pinUid);
        if otherPin == nil then
          ERROR("nil ")
        end
        if otherNode._WalkThroughPin then
          otherNode:_WalkThroughPin(otherPin, rtti, rttiOfArray);
        end
      end
    end
  end
end


-- 注意不能再unregister之后调用
function BlueAnyTypeNode:_GetPinInfoByGfo(gfo)
  local pinInfo = self:GetPinInfoByPinId(gfo.pinUid);
  assert(pinInfo, "_GetPinInfoByGfo not found "..tostring(gfo.pinUid))
  return pinInfo;
end

function BlueAnyTypeNode:_GetPinByGfo(gfo)
  local pin = self:GetPinByPinId(gfo.pinUid)
  assert(pin, "_GetPinByGfo not found "..tostring(gfo.pinUid))
  return pin
end

-- 对应pin生成默认值和字面值
function BlueAnyTypeNode:_GenerateLiteral(pinInfo)
  pinInfo.default = BU:GetDefaultByRtti(pinInfo.rtti);
  local pin =  self:GetPinByInfo(pinInfo);
  pin.literal = BU:GenerateLiteral(pinInfo.rtti,  pinInfo.default);
  return pin
end

-- 派生类实现
function BlueAnyTypeNode:_OnDeserialize()
  BlueAnyTypeNode.super._OnDeserialize(self);
  -- 反序列化必须先调用基类的 _OnDeserialize 再执行派生类的反序列化操作
  -- 到这里已经完成pinGroup的反序列化
  for name, grp in pairs(self.pinGroup) do
    for _, gfo in pairs(grp) do
      local pinInfo = self:_GetPinInfoByGfo(gfo);
      pinInfo.myGroup = grp ;
    end
  end

end

function BlueAnyTypeNode:_OnCreateWithEditor()
  BlueAnyTypeNode.super._OnCreateWithEditor(self);
  -- 把 MemberRegister注册时标记需要加入Group的pin,键入对应的组
  for _, pinInfo in pairs(self.infoTable[BD.PIN_DATA_INPUT]) do
    if pinInfo.groupName ~= nil then
      local pin = self:GetPinByInfo(pinInfo);
      self:JoinGroup(pinInfo.groupName, pin);
    end
  end
  for _, pinInfo in pairs(self.infoTable[BD.PIN_DATA_OUTPUT]) do
    if pinInfo.groupName ~= nil then
      local pin = self:GetPinByInfo(pinInfo);
      self:JoinGroup(pinInfo.groupName, pin);
    end
  end
end

-- 派生类实现
function BlueAnyTypeNode:_OnDeserializePost()
  BlueAnyTypeNode.super._OnDeserializePost(self);
end

return BlueAnyTypeNode