package sg.bigo.libvideo.cam.runtime;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.concurrent.LinkedBlockingQueue;

import sg.bigo.libvideo.cam.abs.Log;

public class VcRuntime {
    private final String TAG = "VcRuntime";
    private static final VcRuntime sInstance = new VcRuntime();

    private VcRuntime() {
    }

    public static VcRuntime instance() {
        return sInstance;
    }

    public enum RuntimeType {
        Single, Multiple
    }
    public boolean enableFrameStory = false;
    private long lastFrameTime = 0l;
    private int maxCost = 0;
    private int minCost = 0;
    private int avgCost = 0;
    private int index = 0;
    private final int MAX_STORIES = 10;
    private LinkedBlockingQueue<VcFrameStory> mRecycledStories = new LinkedBlockingQueue(2);
    private LinkedHashMap<Long, VcFrameStory> mFrameStories = new LinkedHashMap<>();
    private HashMap<String, VcHandler> mRuntimeMap = new HashMap<>();

    public VcHandler runtime() {
        return runtime(null);
    }

    /**
     * 获取相机线程
     *
     * @param cameraIndex 如果为单线程模式可传入"Single",否则传入"相机id"
     * @return
     */
    public VcHandler runtime(String cameraIndex) {
        synchronized (mRuntimeMap) {
            if (cameraIndex == null) {
                cameraIndex = RuntimeType.Single.name();
            }
            VcHandler runtime = mRuntimeMap.get(cameraIndex);
            if (runtime == null) {
                runtime = new VcHandler();
                mRuntimeMap.put(cameraIndex, runtime);
            }
            return runtime;
        }
    }

    private final VcFpsContext mFpsController = new VcFpsContext();

    public void decreaseFps() {
        mFpsController.decreaseFps();
    }

    public void increaseFps() {
        mFpsController.increaseFps();
    }

    public boolean needDropFrame() {
        return mFpsController.needDropFrame();
    }

    public void clearMonitor() {
        mFpsController.clear();
        mFrameStories.clear();
    }

    private VcFrameStory getEmptyStory() {
        return mRecycledStories.poll();
    }

    private boolean recycleStory(VcFrameStory story) {
        if (story == null) {
            return false;
        }
        story.clear();
        return mRecycledStories.offer(story);
    }

    private void calcFps(long current) {
        if (lastFrameTime > 0) {
            int cost = (int) (current - lastFrameTime);
            if (cost > maxCost) {
                maxCost = cost;
            } else if (cost < minCost) {
                minCost = cost;
            }
            if (avgCost > 0) {
                avgCost = (cost + avgCost) / 2;
            } else {
                avgCost = cost;
            }
            if (index % 30 == 0 && index > 0) {
                int avgfps = avgCost <= 0 ? -1 : (int) (1000.0f / avgCost);
                int maxFps = minCost <= 0 ? -1 : (int) (1000.0f / minCost);
                int minFps = maxCost <= 0 ? -1 : (int) (1000.0f / maxCost);
                Log.e(TAG, "avgFps:" + avgfps + ";maxFps:" + maxFps + ";minFps:" + minFps + ";deltaFps:" + (mFpsController.getCurrentFps()));
                if (avgfps < 20) {
                    Log.e(TAG, "story:  "+readLastFrameStory());
                }
                avgCost = 0;
                maxCost = 0;
                minCost = Integer.MAX_VALUE;
                index = 0;
            }
        }
        lastFrameTime = current;
        index++;
    }

    public void onFrameMileStone(long frameIndex, VcProvenance runtime, String desc) {
        if(!enableFrameStory){
            return;
        }
        synchronized (mFrameStories) {
            VcFrameStory story = mFrameStories.get(frameIndex);
            boolean isFrameStart = false;
            if (story == null) {
                isFrameStart = true;
                if (mFrameStories.size() > MAX_STORIES) {
                    Iterator it = mFrameStories.values().iterator();
                    if (it.hasNext()) {
                        VcFrameStory oldStory = (VcFrameStory) it.next();
                        recycleStory(oldStory);
                        it.remove();
                    }
                }
                story = getEmptyStory();
                if (story == null) {
                    story = new VcFrameStory();
                }
            }
            VcFrameStory.MileStone mileStone = story.mileStone(frameIndex, runtime, desc);
            if (isFrameStart) {
                calcFps(mileStone.time);
            }
            mFrameStories.put(frameIndex, story);
        }
    }

    public void onFrameEvent(long frameIndex, String event) {
        if(!enableFrameStory){
            return;
        }
        synchronized (mFrameStories) {
            VcFrameStory story = mFrameStories.get(frameIndex);
            if (story != null) {
                story.event(event);
            }
        }
    }

    public String readLastFrameStory() {
        if(!enableFrameStory){
            return "";
        }
        synchronized (mFrameStories) {
            String storyText = null;
            if (!mFrameStories.isEmpty()) {
                Iterator it = mFrameStories.values().iterator();
                while (it.hasNext()) {
                    VcFrameStory story = (VcFrameStory) it.next();
                    storyText = story.readStory();
                    break;
                }
            }
            return storyText;
        }
    }

    public String readFrameStories() {
        if(!enableFrameStory){
            return "";
        }
        synchronized (mFrameStories) {
            StringBuilder stringBuilder = new StringBuilder();
            if (!mFrameStories.isEmpty()) {
                Iterator it = mFrameStories.values().iterator();
                while (it.hasNext()) {
                    VcFrameStory story = (VcFrameStory) it.next();
                    stringBuilder.append("fsp:");
                    stringBuilder.append(mFpsController.getCurrentFps());
                    stringBuilder.append(";");
                    stringBuilder.append(story.readStory());
                }
            }
            return stringBuilder.toString();
        }
    }

    public static int readNv21ToYuv420Times(int param) {
        return 0x0f & param;
    }

    public static int readNativeCopyTimes(int param) {
        return (0xf0 & param) >> 4;
    }

}
