package sg.bigo.libvideo.cam.runtime;

import android.os.Bundle;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.NonNull;

import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import sg.bigo.libvideo.cam.abs.Log;
import sg.bigo.libvideo.cam.abs.VcCamera;
import sg.bigo.libvideo.cam.abs.VcCharacterListener;

public class VcHandler {

    private final String TAG = "VcHandler";
    private CameraHandler mCameraHandler;
    private ReentrantReadWriteLock mLock = new ReentrantReadWriteLock();
    private VcCamera mCamera;
    private VcCamera mPendingCamera;
    private VcCharacterListener mParamListener;

    public CameraHandler handler() {
        return mCameraHandler;
    }

    public void open(VcCamera camera, VcCharacterListener listener) {
        Log.e(TAG, "open");
        mLock.writeLock().lock();
        try {
            if (camera == null) {
                Log.e(TAG, "start camera with null VcCamera");
                return;
            }
            if (mCameraHandler == null) {
                Log.e(TAG, "start camera thread:" + camera.getCameraIndex());
                HandlerThread thread = new HandlerThread(TAG);
                thread.start();
                mCameraHandler = new CameraHandler(this, thread.getLooper());
            }
            mParamListener = listener;
            mPendingCamera = camera;
            mCameraHandler.removeMessages(MSG_OPEN);
            mCameraHandler.removeMessages(MSG_RELEASE);
            mCameraHandler.sendEmptyMessage(MSG_OPEN);
        } finally {
            mLock.writeLock().unlock();
        }
    }

    public void applyParam(final int requestCode, final int key, final int[] paramsInt) {
        mLock.readLock().lock();
        try {
            if (mCameraHandler != null) {
                Message message = mCameraHandler.obtainMessage(MSG_APPLY_PARAM);
                Bundle data = new Bundle();
                data.putInt(KEY_REQUEST_CODE, requestCode);
                data.putInt(KEY_PARAM, key);
                data.putIntArray(KEY_VALUES, paramsInt);
                message.setData(data);
                message.sendToTarget();
            } else {
                Log.e(TAG, "close when mCameraHandle null");
            }
        } finally {
            mLock.readLock().unlock();
        }
    }

    public void queryParam(final int requestCode, final int key) {
        mLock.readLock().lock();
        try {
            if (mCameraHandler != null) {
                Message message = mCameraHandler.obtainMessage(MSG_QUERY_PARAM);
                Bundle data = new Bundle();
                data.putInt(KEY_REQUEST_CODE, requestCode);
                data.putInt(KEY_PARAM, key);
                message.setData(data);
                message.sendToTarget();
            } else {
                Log.e(TAG, "close when mCameraHandle null");
            }
        } finally {
            mLock.readLock().unlock();
        }
    }

    public void close() {
        Log.e(TAG, "close");
        mLock.readLock().lock();
        try {
            if (mCameraHandler != null) {
                mCameraHandler.removeMessages(MSG_CLOSE);
                mCameraHandler.removeMessages(MSG_RELEASE);
                mCameraHandler.sendEmptyMessage(MSG_CLOSE);
                mCameraHandler.sendEmptyMessageDelayed(MSG_RELEASE, 10_000);//相机关闭十秒后无操作则取消相机线程
            } else {
                Log.e(TAG, "close when mCameraHandle null");
            }
        } finally {
            mLock.readLock().unlock();
        }
    }

    static class CameraHandler extends android.os.Handler {
        final SoftReference<VcHandler> reference;

        public CameraHandler(VcHandler camera, Looper looper) {
            super(looper);
            reference = new SoftReference<>(camera);
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            super.handleMessage(msg);
            VcHandler camera = reference.get();
            if (camera != null) {
                camera.handleMessage(msg);
            }
        }
    }

    private final String KEY_REQUEST_CODE = "requestCode";
    private final String KEY_PARAM = "param";
    private final String KEY_VALUES = "values";

    private final int MSG_OPEN = 0x01;
    private final int MSG_CLOSE = 0X02;
    private final int MSG_RELEASE = 0x03;
    private final int MSG_APPLY_PARAM = 0x04;
    private final int MSG_QUERY_PARAM = 0x05;

    private void handleMessage(Message message) {
        switch (message.what) {
            case MSG_APPLY_PARAM: {
                Bundle data = message.getData();
                if (data == null || mCamera == null) {
                    return;
                }
                int requestCode = data.getInt(KEY_REQUEST_CODE);
                int param = data.getInt(KEY_PARAM);
                int[] values = data.getIntArray(KEY_VALUES);
                int result = mCamera.applyParam(param, values);
                mLock.readLock().lock();
                try {
                    if (mParamListener != null) {
                        mParamListener.onApplyResult(requestCode, param, result);
                    }
                } finally {
                    mLock.readLock().unlock();
                }
            }
            break;
            case MSG_QUERY_PARAM: {
                Bundle data = message.getData();
                if (data == null || mCamera == null) {
                    return;
                }
                int requestCode = data.getInt(KEY_REQUEST_CODE);
                int param = data.getInt(KEY_PARAM);
                List<Integer> result = new ArrayList<>();
                int res = mCamera.queryParam(param, result);
                if (res < 0) {
                    Log.e(TAG, "query param failed, key:" + param + ";requestCode:" + requestCode);
                }
                mLock.readLock().lock();
                try {
                    if (mParamListener == null) {
                        return;
                    }
                    int[] resultArray = null;
                    int size = result.size();
                    if (size > 0) {
                        resultArray = new int[result.size()];
                        for (int i = 0; i < size; i++) {
                            resultArray[i] = result.get(i);
                        }
                    }
                    mParamListener.onQueryResult(requestCode, param, resultArray);
                } finally {
                    mLock.readLock().unlock();
                }
            }
            break;
            case MSG_OPEN:
                if (mCamera != null) {
                    Log.e(TAG, "close old " + mCamera.getCameraIndex() + ", before open new");
                    mCamera.close();
                }
                synchronized (mLock) {
                    if (mPendingCamera != null) {
                        mCamera = mPendingCamera;
                        mPendingCamera = null;
                    }
                }
                if (mCamera != null) {
                    Log.e(TAG, "open " + mCamera.getCameraIndex());
                    mCamera.open();
                } else {
                    Log.e(TAG, "openCamera with null camera");
                }
                break;
            case MSG_CLOSE:
                _close();
                break;
            case MSG_RELEASE:
                _close();
                synchronized (mLock) {
                    Log.e(TAG, "quit Looper ");
                    Looper looper = mCameraHandler.getLooper();
                    if (looper != null) {
                        looper.quit();
                        mCameraHandler = null;
                    }
                }
                break;
        }
    }

    private void _close() {
        if (mCamera != null) {
            Log.e(TAG, "_close 00 " + mCamera.getCameraIndex());
            mCamera.close();
            mCamera = null;
        }
    }
}
