package sg.bigo.libvideo.cam.report;

import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;
import android.util.Pair;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import sg.bigo.libvideo.cam.abs.Log;
import sg.bigo.libvideo.cam.abs.VcCameraDeviceManager;
import sg.bigo.libvideo.cam.abs.VcProperties;

/**
 * @author wilbert
 * @Date 2021/1/11 15:05
 * @email jiangwang.wilbert@bigo.sg
 **/
public class HEMediaReporter {
    private static final String TAG = "MediaReporter";

    public static final String KEY = "key";

    private static final String PREFERENCE_NAME = "media_report";

    private static boolean mIsVideoSystem = false;

    private static long mStartTick = -1;

    public static void WriteBooleanPreferences(Context context, String key, boolean value) {
        if (context == null || TextUtils.isEmpty(key)) {
            return;
        }
        SharedPreferences preferences = context.getSharedPreferences(PREFERENCE_NAME, Context.MODE_PRIVATE);
        preferences.edit().putBoolean(key, value).apply();
    }

    public static boolean ReadBooleanPreferences(Context context, String key, Boolean defValue) {
        if (context == null || TextUtils.isEmpty(key)) {
            return false;
        }
        SharedPreferences preferences = context.getSharedPreferences(PREFERENCE_NAME, Context.MODE_PRIVATE);
        return preferences.getBoolean(key, defValue);
    }

    public interface Key {
        String KEY_METER = "meter";
        String KEY_FOCUS = "focus";
        String KEY_DRASTIC = "drastic";
        String KEY_METER_FOCUS_AREAS = "mf_areas";
        String KEY_CAMERA_PREVIEW_SIZE_ERROR = "err_psize";
        String KEY_CAMERA_FOCUS_ERROR = "err_focus";
        String KEY_CAMERA_OPERATIONS = "camera_operations";
        String KEY_CAMERA_OPEN_FAIL_INFO = "camera_open_fail_info";
        String KEY_CAMERA_ATTRIBUTES = "camera_attributes";
        String KEY_CAMERA_EXCEPTIONS = "camera_exceptions";
        String KEY_TIME_PROFILER_EXCEPTIONS = "time_profiler_exceptions";
        String KEY_OPTIMAL_VIEW_SIZE_SELECTOR = "optimal_view_size_selector";
        String KEY_MODIFIED_VIEW_SIZE_SELECTOR = "modified_view_size_selector";
        String KEY_CAMERA_JANK = "camJank";
        String KEY_NO_HUMAN_FACE_POINTS = "no_human_face_points";
        String KEY_ERROR_ARRAY_COPY = "arrcp";
        String KEY_BVT_ASYNC_DETECT_STRATEGY = "bvt_async_detect_strategy";
        String KEY_PIXEL_READER_INIT = "PRInit";
        String KEY_CAMERA_OPEN_FAILURE_ON_PK_BACKGROUND = "open_failure_pk_back";
        String KEY_PREPROCESS_RESOLUTION_CHANGED = "preprocess_resolution_changed";
        String KEY_CAMERA_FRAME_SKIP = "camera_frame_skip";
    }

    public interface Field {
        /**
         * 旧的测光模式
         * {@link sg.bigo.libvideo.cam.metering.Metering.ExposureStatus}
         */
        String OLD_METERING_TYPE = "meter_old";

        /**
         * 新的测光模式
         * {@link sg.bigo.libvideo.cam.metering.Metering.ExposureStatus}
         */
        String NEW_METERING_TYPE = "meter_new";

        /**
         * 是否锁定手动测光模式
         * {@link sg.bigo.libvideo.cam.metering.ManualType}
         */
        String MANUAL = "meter_manual";

        /**
         * 比如说从人脸测光切换到手动测光，在切换发生之前有过多次人脸测光，meter_times代表了这个次数
         */
        String INNER_TIMES = "meter_times";

        /**
         * 本次测光最终设置到camera的行为，0：由于camera不支持所以未做任何测光操作，1：成功设置测光参数
         */
        String CAMERA_METER_RESULT = "meter_result";

        /**
         * 像素变化检测时长
         */
        String DRASTIC_DURATION = "dra_dur";

        /**
         * 从手动测光开始到检测完成的时长
         */
        String MANUAL_DURATION = "mal_dur";

        /**
         * 像素变化相似度
         */
        String DRASTIC_SIMULATE = "simulate";

        /**
         * Camera最大支持的测光数
         */
        String MAX_METERING_AREAS = "m_areas";

        /**
         * Camera最大支持的对焦数
         */
        String MAX_FOCUS_AREAS = "f_areas";

        /**
         * 当前cameraId
         */
        String CAMERA_ID = "cameraId";

        /**
         * camera的预览宽高 width_height
         */
        String PREVIEWSIZE = "preSize";

        /**
         * config的Camera预览宽高 width_height
         */
        String CONFIGSIZE = "csize";

        /**
         * 手动对焦点击区域点 x_y
         */
        String FOCUSPOINT = "focusP";

        /**
         * 手动对焦控件宽高 width_height
         */
        String FOCUSVIEWSIZE = "focusVS";

        /**
         * 手动对焦预览宽高 width_height
         */
        String FOCUSCAPTURESIZE = "focusCS";

        /**
         * 相机打开状态：0-失败, 1-成功
         */
        String CAMERA_OPEN_STATUS = "camera_open_status";

        /**
         * 相机打开失败时使用的 API
         */
        String CAMERA_OPEN_FAIL_API = "camera_open_fail_api";

        /**
         * 相机打开成功时使用的 API
         */
        String CAMERA_OPEN_SUCCESS_API = "camera_open_success_api";

        /**
         * 是否第一次打开相机成功
         */
        String CAMERA_OPEN_IS_FIRST_TRY = "camera_open_is_first_try";

        /**
         * 成功打开相机需要的重试次: 0 ~ n  // 此行为likee 注释
         * 相机打开失败，重试次数打点: 1    // 此行为bigo 注释
         */
        String CAMERA_OPEN_RETRY_TIMES = "camera_open_retry_times";

        /**
         * 重试打开相机成功花费次数: 0 ~ n
         */
        String CAMERA_OPEN_RETRY_SUCCESS_TIMES = "camera_open_retry_success_times";

        /**
         * 相机从打开耗时: ms
         */
        String CAMERA_OPEN_DELAY = "camera_open_delay";

        /**
         * 相机从打开成功到首帧回调耗时: ms
         */
        String OPENED_TO_FIRST_FRAME_DELAY = "opened_to_first_frame_delay";

        /**
         * 相机从打开成功到开始预览的耗时: ms
         */
        String OPEN_TO_PREVIEW_DELAY = "open_to_preview_delay";

        /**
         * 开启预览的结果：0-失败, 1-成功
         */
        String START_PREVIEW_STATUS = "start_preview_status";

        /**
         * 相机从开始预览到收到第一帧回调的耗时: ms
         */
        String PREVIEW_TO_FIRST_FRAME_DELAY = "preview_to_first_frame_delay";

        /**
         * camera1执行startFaceDetect()方法的耗时: ms
         */
        String CAMERA1_START_FACE_DETECT_DELAY = "camera1_start_face_detect_delay";

        /**
         * 相机关闭状态: 0-失败, 1-成功
         */
        String CAMERA_CLOSE_STATUS = "camera_close_status";

        /**
         * 采集分辨率
         */
        String CAMERA_RAW_WIDTH = "camera_raw_width";
        String CAMERA_RAW_HEIGHT = "camera_raw_height";

        /**
         * 采集帧率区间
         */
        String PREVIEW_FPS_DEFAULT = "preview_fps_default";

        /**
         * 所用API: 1或2
         */
        String CAMERA_API = "camera_api";

        /**
         * 设备支持的相机个数
         */
        String CAMERA_NUMS = "camera_nums";

        /**
         * 当前相机是否为前置
         */
        String IS_FRONT_FACING = "is_front_facing";

        /**
         * 相机支持的人脸检测最大个数
         */
        String MAX_DETECTED_FACES = "max_detected_faces";

        /**
         * 是否支持闪光灯
         */
        String IS_FLASH_SUPPORTED = "is_flash_supported";

        /**
         * 是否支持防抖
         */
        String IS_VIDEO_STABILIZATION_SUPPORTED = "is_video_stabilization_supported";

        /**
         * 是否支持zoom
         */
        String IS_ZOOM_SUPPORTED = "is_zoom_supported";

        /**
         * 编码推荐分辨率, 实际选择分辨率和最接近的16：9分辨率
         */
        String PREFER_ENCODE_WH = "prefer_encode_wh";
        String SELECT_WH = "select_wh";
        String NORM_WH = "norm_wh";

        /**
         * ICamera.OptimalViewSizeSelector选择策略的生效路径
         */
        String STRATEGY_PATH = "strategy_path";

        /**
         * 读图buffer未更新
         */
        String camJanks = "camJanks";

        /**
         * 对焦类型，0为请求对焦，1为重置
         */
        String FOCUS_TYPE = "focus_type";

        /**
         * 对焦结果
         */
        String FOCUS_RESULT = "focus_result";

        /**
         * 对焦总次数
         */
        String FOCUS_TIMES = "total_focus_time";

        /**
         * 对焦时间戳
         */
        String FOCUS_TIMESTAMP = "focus_ts";

        /**
         * 点击 AE & AF 次数
         */
        String TOUCH_TIMES = "touch_times";

        /**
         * 性能统计上报错误具体信息
         */
        String TIME_PROFILER_ERROR_INFO = "tp_err_info";

        /**
         * 通用时长、间隔 ms
         */
        String DURATION = "duration";

        /**
         * 打开相机的调用堆栈
         */
        String CAMERA_OPEN_STACK = "camera_open_stack";

        /**
         * 打开相机的 api Level
         */
        String CAMERA_OPEN_API_LEVEL = "camera_open_api_level";

        /**
         * 相机打开后出帧为 0
         */
        String CAMERA_OUTPUT_NULL_FRAME = "camera_output_null_frame";

        /**
         * 通过重启相机修复了相机打开后出帧为 0 的问题
         */
        String CAMERA_OUTPUT_NULL_FRAME_FIX = "camera_output_null_frame_fix";

        /**
         * 打开相机首次出帧成功上报
         */
        String CAMERA_OUTPUT_FRAME_SUCCESS_FIRST_TIME = "camera_output_frame_success_first_time";

        /**
         * 相机因无权限打开失败
         */
        String CAMERA_OPEN_FAILED_NO_PERMISSION = "camera_open_failed_no_permission";

        /**
         * 相机错误相关信息
         */
        String CAMERA_EXCEPTION_MSG = "camera_exception_msg";

        String CAMERA_ERROR_TYPE = "error_type";

        String CAMERA_ERROR_SUB_CODE = "error_sub_code";

        String CAMERA_EXCEPTION_STATUS = "camera_exception_status";


    }

    public static void report(List<Pair<String, String>> params) {
        VcCameraDeviceManager.report(params);
    }


    /**
     * @param focusType   请求对焦，还是恢复托管,0 为请求对焦，1 为恢复系统托管对焦
     * @param focusResult 请求对焦结果
     * @param focusTimes  对焦总次数
     */
    public static void reportFocusEvent(int focusType, boolean focusResult, int focusTimes, int touchTimes) {
        String timeStamp = System.currentTimeMillis() + "";
        Log.e(TAG, "reportFocusEvent focusType:" + focusType + ";focusResult:" + focusResult + ";times:"
                + focusTimes + ";touchTimes:" + touchTimes + ";timeStamp:" + timeStamp);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_FOCUS);
        Pair<String, String> ft = new Pair<>(Field.FOCUS_TYPE, focusType + "");
        Pair<String, String> fr = new Pair<>(Field.FOCUS_RESULT, focusResult + "");
        Pair<String, String> tt = new Pair<>(Field.TOUCH_TIMES, touchTimes + "");
        Pair<String, String> tft = new Pair<>(Field.FOCUS_TIMES, focusTimes + "");
        Pair<String, String> ts = new Pair<>(Field.FOCUS_TIMESTAMP, timeStamp);
        params.add(key);
        params.add(ft);
        params.add(fr);
        params.add(tt);
        params.add(tft);
        params.add(ts);
        report(params);
    }

    /**
     * @param oldMeter         切换前的meter， {@link sg.bigo.libvideo.cam.metering.Metering.ExposureStatus}
     * @param newMeter         切换后的meter， {@link sg.bigo.libvideo.cam.metering.Metering.ExposureStatus}
     * @param manualType       本地使用的是否智能测光 {@link sg.bigo.libvideo.cam.metering.ManualType}
     * @param innerSwitchTimes 在从old切换到new之前，在old中执行过的测光次数
     */
    public static void reportSwitchMeter(int oldMeter, int newMeter, int manualType, int innerSwitchTimes, int meterResult) {
        Log.e(TAG, "reportSwitchMeter oldMeter:" + oldMeter + ";newMeter:" + newMeter + ";manualType:"
                + manualType + ";innerSwitchTimes:" + innerSwitchTimes + ";meterResult:" + meterResult);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_METER);
        Pair<String, String> oldM = new Pair<>(Field.OLD_METERING_TYPE, oldMeter + "");
        Pair<String, String> newM = new Pair<>(Field.NEW_METERING_TYPE, newMeter + "");
        Pair<String, String> manual = new Pair<>(Field.MANUAL, manualType + "");
        Pair<String, String> times = new Pair<>(Field.INNER_TIMES, innerSwitchTimes + "");
        Pair<String, String> result = new Pair<>(Field.CAMERA_METER_RESULT, meterResult + "");
        params.add(key);
        params.add(oldM);
        params.add(newM);
        params.add(manual);
        params.add(times);
        params.add(result);
        report(params);
    }

    public static void reportDrasticChanged(long drasticDuration, long manualDuration, float simulate) {
        Log.e(TAG, "reportDrasticChanged drasticDuration:" + drasticDuration + ";manualDuration:" + manualDuration + ";simulate:" + simulate);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_DRASTIC);
        Pair<String, String> dra_dur = new Pair<>(Field.DRASTIC_DURATION, drasticDuration + "");
        Pair<String, String> mal_dur = new Pair<>(Field.MANUAL_DURATION, manualDuration + "");
        Pair<String, String> sm = new Pair<>(Field.DRASTIC_SIMULATE, simulate + "");
        params.add(key);
        params.add(dra_dur);
        params.add(mal_dur);
        params.add(sm);
        report(params);
    }

    public static void reportMaxMeterFocusNum(Context context, int cameraId, int maxMeteringAreas, int maxFocusAreas) {
        if (ReadBooleanPreferences(context, Key.KEY_METER_FOCUS_AREAS, false)) {
            return;
        }
        Log.e(TAG, "reportMaxMeterFocusNum:" + maxMeteringAreas + ";" + maxFocusAreas);
        WriteBooleanPreferences(context, Key.KEY_METER_FOCUS_AREAS, true);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_METER_FOCUS_AREAS);
        Pair<String, String> id = new Pair<>(Field.CAMERA_ID, cameraId + "");
        Pair<String, String> m_areas = new Pair<>(Field.MAX_METERING_AREAS, maxMeteringAreas + "");
        Pair<String, String> f_areas = new Pair<>(Field.MAX_FOCUS_AREAS, maxFocusAreas + "");
        params.add(key);
        params.add(id);
        params.add(m_areas);
        params.add(f_areas);
        report(params);
    }

    public static void reportCameraPreviewSizeError(int width, int height, int pWidth, int pHeight) {
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_PREVIEW_SIZE_ERROR);
        Pair<String, String> ps = new Pair<>(Field.PREVIEWSIZE, width + "_" + height);
        Pair<String, String> cs = new Pair<>(Field.CONFIGSIZE, pWidth + "_" + pHeight);
        params.add(key);
        params.add(ps);
        params.add(cs);
        report(params);
    }

    public static void reportFocusError(float touchX, float touchY, int viewWidth, int viewHeight, int pWidth, int pHeight) {
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_FOCUS_ERROR);
        Pair<String, String> tp = new Pair<>(Field.FOCUSPOINT, touchX + "_" + touchY);
        Pair<String, String> vs = new Pair<>(Field.FOCUSVIEWSIZE, viewWidth + "_" + viewHeight);
        Pair<String, String> ps = new Pair<>(Field.CONFIGSIZE, pWidth + "_" + pHeight);
        params.add(key);
        params.add(tp);
        params.add(vs);
        params.add(ps);
        report(params);
    }

    public static void reportCameraOperations(String action, String value, int apiLevel) {
        Log.e(TAG, "reportCameraOperations\t" + action + ": " + value + " apiLevel: " + apiLevel);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_OPERATIONS);
        Pair<String, String> item = new Pair<>(action, value);
        Pair<String, String> api = new Pair<>(Field.CAMERA_OPEN_API_LEVEL, apiLevel + "");
        params.add(key);
        params.add(item);
        params.add(api);
        report(params);
    }

    public static String getExceptionString(String msg, Throwable exception,
                                                               String cameraIndex, int facingFront,
                                                               int captureWidth, int captureHeight,
                                                               int cameraStatus) {
        List<Pair<String ,String>> params = new ArrayList<>();
        if (exception != null) {
            params.add(new Pair<>(VcProperties.OperateStatus.exception.toString(), exception.getClass().getSimpleName()));
            params.add(new Pair<>(VcProperties.OperateStatus.stackTrace.toString(), android.util.Log.getStackTraceString(exception)));
        }
        if (msg != null) {
            params.add(new Pair<>(VcProperties.OperateStatus.errMsg.toString(), msg));
        }
        params.add(new Pair<>(VcProperties.OperateStatus.cameraId.toString(), cameraIndex));
        params.add(new Pair<>(VcProperties.OperateStatus.isFront.toString(), ""+ facingFront));
        params.add(new Pair<>(VcProperties.OperateStatus.api.toString(), "" + VcProperties.API.CAMERA1));
        params.add(new Pair<>(VcProperties.OperateStatus.capSize.toString(), captureWidth + "*" + captureHeight));
        params.add(new Pair<>(VcProperties.OperateStatus.status.toString(), "" + cameraStatus));
        return VcCameraDeviceManager.packReportPairs(params);
    }

    public static void reportCameraOpenFailedInfo(String openStack, Exception e, int apiLevel) {
        String name = e.getClass().getSimpleName();
        Log.e(TAG, " reportCameraOpenFailedInfo:\t Exceptions: " + name + "\t operatePath: " + openStack);
        String stackTrace = android.util.Log.getStackTraceString(e);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_OPEN_FAIL_INFO);
        Pair<String, String> opStack = new Pair<>(Field.CAMERA_OPEN_STACK, openStack);
        Pair<String, String> excep = new Pair<>(name, stackTrace);
        Pair<String, String> api = new Pair<>(Field.CAMERA_OPEN_API_LEVEL, apiLevel + "");
        params.add(key);
        params.add(opStack);
        params.add(excep);
        params.add(api);
        report(params);
    }

    public static synchronized void setStartPreviewTick(long tick) {
        mStartTick = tick;
    }

    public static synchronized void reportPreviewToFirstFrameDelay(long endTick, int apiLevel) {
        if (mStartTick != -1 && endTick > mStartTick) {
            long delay = endTick - mStartTick;
            reportCameraOperations(Field.PREVIEW_TO_FIRST_FRAME_DELAY, delay + "", apiLevel);
        }
    }

    public static void reportCameraAttributes(String label, String value) {
        Log.e(TAG, "reportCameraAttributes\t" + label + ": " + value);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_ATTRIBUTES);
        Pair<String, String> item = new Pair<>(label, value);
        params.add(key);
        params.add(item);
        report(params);
    }

    public static void reportCameraAttributes(Map<String, String> pairs) {
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_ATTRIBUTES);
        params.add(key);
        Iterator<Map.Entry<String, String>> iterator = pairs.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, String> entry = iterator.next();
            params.add(new Pair<>(entry.getKey(), entry.getValue()));
            Log.e(TAG, "reportCameraAttributes\t" + entry.getKey() + ": " + entry.getValue());
        }
        report(params);
    }

    public static void reportCameraExceptions(Exception e, int apiLevel) {
        String name = e.getClass().getSimpleName();
        String stackTrace = android.util.Log.getStackTraceString(e);
        Log.e(TAG, "reportCameraExceptions: " + name);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_EXCEPTIONS);
        Pair<String, String> item = new Pair<>(name, stackTrace);
        Pair<String, String> api = new Pair<>(Field.CAMERA_OPEN_API_LEVEL, apiLevel + "");
        params.add(key);
        params.add(item);
        params.add(api);
        report(params);
    }

    public static void reportCameraExceptionWithMsg(Exception e, String msg, int apiLevel) {
        String name = e.getClass().getSimpleName();
        String stackTrace = android.util.Log.getStackTraceString(e);
        Log.e(TAG, "reportCameraExceptions: " + name);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_EXCEPTIONS);
        Pair<String, String> item = new Pair<>(name, stackTrace);
        Pair<String, String> msgs = new Pair<>(Field.CAMERA_EXCEPTION_MSG, msg);
        Pair<String, String> api = new Pair<>(Field.CAMERA_OPEN_API_LEVEL, apiLevel + "");
        params.add(key);
        params.add(item);
        params.add(msgs);
        params.add(api);
        report(params);
    }

    public static void reportTimeProfilerExceptions(Exception e, String errorInfo) {
        String name = e.getClass().getSimpleName();
        String stackTrace = android.util.Log.getStackTraceString(e);
        Log.e(TAG, "reportTimeProfilerExceptions: " + name + " : " + errorInfo);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_TIME_PROFILER_EXCEPTIONS);
        Pair<String, String> item = new Pair<>(name, stackTrace);
        Pair<String, String> errInfo = new Pair<>(Field.TIME_PROFILER_ERROR_INFO, errorInfo);
        params.add(key);
        params.add(item);
        params.add(errInfo);
        report(params);
    }

    public static void reportCameraOpenFailureOnPkBackground(Exception e, long duration) {
        String name = e.getClass().getSimpleName();
        String stackTrace = android.util.Log.getStackTraceString(e);
        Log.e(TAG, "reportCameraOpenFailureOnPkBackground: " + stackTrace + ", happened after " + duration + " ms");
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_CAMERA_OPEN_FAILURE_ON_PK_BACKGROUND);
        Pair<String, String> item = new Pair<>(name, stackTrace);
        Pair<String, String> time = new Pair<>(Field.DURATION, duration + "");
        params.add(key);
        params.add(item);
        params.add(time);
        report(params);
    }

    public static void reportOptimalViewSizeSelectorStrategy(int preferEncodeWidth, int preferEncodeHeight, int selectWidth, int selectHeight, String path) {
        Log.e(TAG, "reportOptimalViewSizeSelectorStrategy: prefer " + preferEncodeWidth + "x" + preferEncodeHeight + ", select " + selectWidth + "x" + selectHeight + ", path " + path);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_OPTIMAL_VIEW_SIZE_SELECTOR);
        Pair<String, String> prefer = new Pair<>(Field.PREFER_ENCODE_WH, preferEncodeWidth + "x" + preferEncodeHeight);
        Pair<String, String> select = new Pair<>(Field.SELECT_WH, selectWidth + "x" + selectHeight);
        Pair<String, String> spath = new Pair<>(Field.STRATEGY_PATH, path);
        params.add(key);
        params.add(prefer);
        params.add(select);
        params.add(spath);
        report(params);
    }

    // called when path = p3
    public static void reportModifiedViewSizeSelectorStrategy(int preferEncodeWidth, int preferEncodeHeight, int selectWidth, int selectHeight, int normWidth, int normHeight) {
        Log.e(TAG, "reportModifiedViewSizeSelectorStrategy: prefer " + preferEncodeWidth + "x" + preferEncodeHeight + ", select " + selectWidth + "x" + selectHeight + ", norm " + normWidth + "x" + normHeight);
        List<Pair<String, String>> params = new ArrayList<>();
        Pair<String, String> key = new Pair<>(KEY, Key.KEY_MODIFIED_VIEW_SIZE_SELECTOR);
        Pair<String, String> prefer = new Pair<>(Field.PREFER_ENCODE_WH, preferEncodeWidth + "x" + preferEncodeHeight);
        Pair<String, String> select = new Pair<>(Field.SELECT_WH, selectWidth + "x" + selectHeight);
        Pair<String, String> norm = new Pair<>(Field.NORM_WH, normWidth + "x" + normHeight);
        params.add(key);
        params.add(prefer);
        params.add(select);
        params.add(norm);
        report(params);
    }

}