package sg.bigo.libvideo.cam.metering;

import android.graphics.Point;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.hardware.Camera;

import java.util.ArrayList;
import java.util.List;

import sg.bigo.libvideo.cam.abs.Log;
import sg.bigo.libvideo.cam.abs.VcCharacteristics;
import sg.bigo.libvideo.cam.abs.VcProperties;
import sg.bigo.libvideo.cam.report.HEMediaReporter;

/**
 * @author wilbert
 * @Date 2020/12/22 21:18
 * @email jiangwang.wilbert@bigo.sg
 **/
public class MeteringDelegate implements MeteringCallback {
    private final String TAG = "MeteringDelegate";
    public static boolean sDebug = true;
    private VcCharacteristics mCharacteristics;
    private MeteringController mMeteringController;
    private MeteringCallback mMeteringCallback;
    private boolean mMeterConfigInited = false;
    private ManualType mMeteringConfig = ManualType.TYPE_MANUAL_CLOSE;
    private int mTouchForFocusTimes = 0;
    private int mCurrentSwitchTimes = 0;
    private int mFocusTimes = 0;
    private int mResetFocusTimes = 0;

    public MeteringDelegate(VcCharacteristics vcCharacteristics) {
        mCharacteristics = vcCharacteristics;
        setMeteringCallback(this);
    }
    public synchronized void setMeteringCallback(MeteringCallback callback) {
        this.mMeteringCallback = callback;
        if (mMeteringController != null) {
            mMeteringController.setMeteringCallback(mMeteringCallback);
        }
    }

    public synchronized void setMeteringConfig(ManualType config) {
        if (!mMeterConfigInited) {
            mMeteringConfig = config;
            mMeterConfigInited = true;
        }
    }

    public synchronized boolean hitOnManualFocusMetering() {
        return mMeteringConfig.ordinal() > ManualType.TYPE_MANUAL_CLOSE.ordinal();
    }

    public synchronized boolean startFaceDetect(Camera camera) {
        boolean result = false;
        if (hitOnManualFocusMetering()) {
            try {
                getMeteringController().reset();
                Camera.Parameters parameters = camera.getParameters();
                if (parameters != null
                        && parameters.getMaxNumDetectedFaces() > 0) {
                    camera.setFaceDetectionListener(new Camera.FaceDetectionListener() {
                        @Override
                        public void onFaceDetection(Camera.Face[] faces, Camera camera) {
                            boolean hasFace = faces != null && faces.length > 0;
                            Rect rect = new Rect(0, 0, 0, 0);
                            if (hasFace) {
                                RectF faceRect = new RectF(faces[0].rect);
                                rect.set(CameraHelper.clamp((int) faceRect.left, -1000, 1000),
                                        CameraHelper.clamp((int) faceRect.top, -1000, 1000),
                                        CameraHelper.clamp((int) faceRect.right, -1000, 1000),
                                        CameraHelper.clamp((int) faceRect.bottom, -1000, 1000));
                            }
                            getMeteringController().onFaceEvent(hasFace, rect);
                        }
                    });
                    camera.startFaceDetection();
                }
            } catch (Exception e) {
                Log.w(TAG, "[onCameraStartPreview] face detection failed", e);
            }
            result = true;
        }
        return result;
    }

    public synchronized boolean onFrameAvailable(byte[] yuv420, int width, int height) {
        boolean result = false;
        if (hitOnManualFocusMetering()) {
            getMeteringController().onFrameAvailable(yuv420, width, height);
            result = true;
        }
        return result;
    }

    private synchronized boolean onManualFocus(Rect meteringRect, Rect focusRect, Point manualCenter) {
        boolean result = false;
        if (hitOnManualFocusMetering()) {
            getMeteringController().onManualEvent(mMeteringConfig, meteringRect, focusRect, manualCenter);
            result = true;
        }
        return result;
    }

    private MeteringController getMeteringController() {
        if (mMeteringController == null) {
            mMeteringController = new MeteringController(mMeteringConfig);
            mMeteringController.setMeteringCallback(mMeteringCallback);
        }
        return mMeteringController;
    }

    @Override
    public void onMeteringChanged(Metering oldMetering, Metering newMetering) {
        if (newMetering != null) {
            Rect meterRect = newMetering.getMeterRect();
            Metering.ExposureStatus status = newMetering.getState();
            if (MeteringDelegate.sDebug) {
                Log.e(TAG, "onMeteringChanged:" + oldMetering.getState().name() + ";" + newMetering.getState().name());
            }
            try {
                boolean meterResult = false;
                boolean focusResult = false;
                switch (status) {
                    case MANUAL:
                        mTouchForFocusTimes++;
                    case FACE_EXIST: {
                        meterResult = mCharacteristics.setMeteringArea(meterRect, 1000) > VcProperties.Constant.FALSE;
                        List<Integer> autoFocusSupported = new ArrayList<>();
                        mCharacteristics.isAutoFocusSupported(autoFocusSupported);
                        if (autoFocusSupported.size()> 0 && autoFocusSupported.get(0) > VcProperties.Constant.FALSE) {
                            Log.e(TAG, "requestFocus!");
                            focusResult = mCharacteristics.setFocusAreas(meterRect, 1000) > VcProperties.Constant.FALSE;
                            if (mMeteringConfig != ManualType.TYPE_MANUAL_FACE) {
                                HEMediaReporter.reportFocusEvent(0, focusResult, mFocusTimes++, mTouchForFocusTimes);
                            }
                        }
                    }
                    break;
                    // 目前大部分安卓设备底层实现了 中心测光-人脸测光 状态自动切换，部分老旧设备为单一的中心测光，
                    // 设置为 null 时将测光托管给系统，设置为 FOCUS_MODE_CONTINUOUS_VIDEO，时将 AF 托管给系统
                    case DEFAULT:
                    case CENTER_METERING:
                    default: {
                        mCharacteristics.resetMeteringState();
                        mCharacteristics.resetFocusState();
                        HEMediaReporter.reportFocusEvent(1, true, mResetFocusTimes++, mTouchForFocusTimes);
                    }
                    break;
                }
                if (oldMetering instanceof FaceMetering && newMetering instanceof FaceMetering) {
                    // 自研人脸测光策略，人脸位置改变无需统计
                    return;
                }
                reportSwitchEvent(oldMetering, newMetering, getCameraStatResult(meterResult, focusResult));
            } catch (Exception e) {
                Log.e(TAG, e.getMessage() + ";changeState:" + status.name() + ";rect:" + meterRect);
                e.printStackTrace();
            }
        }
    }

    @Override
    public void onMessage(String msg) {
        if (MeteringDelegate.sDebug) {
            Log.e(TAG, "onMessage:" + msg);
        }
    }

    public int requestExposureFocus(int touchX, int touchY, int viewWidth, int viewHeight,
                                    int orientation, int captureWidth, int captureHeight,
                                    boolean isFacingFront) {
        Log.e(TAG, "[requestExposureFocus]" + touchX + touchY + viewWidth + viewHeight);
        float areaMultiple = CameraHelper.DEFAULT_AREA_MULTIPLE;
        PointF centerPoint = CameraHelper.transFormPoint(new PointF(touchX, touchY), orientation,
                isFacingFront, viewWidth, viewHeight, captureWidth, captureHeight);
        if (centerPoint == null) {
            Log.e(TAG, "touchX:" + touchX + ";touchY:" + touchY + ";viewWidth:" + viewWidth + ";viewHeight:" + viewHeight + ";captureWidth:" + captureWidth + ";captureHeight:" + captureHeight);
            HEMediaReporter.reportFocusError(touchX, touchY, viewWidth, viewHeight, captureWidth, captureHeight);
            return VcProperties.Constant.UNAVAILABLE;
        }
        Rect meterRect = CameraHelper.calcMeterRect(centerPoint, captureWidth, captureHeight, areaMultiple);
        Rect focusRect = meterRect;

        boolean manualFocus = onManualFocus(meterRect, focusRect, new Point((int) centerPoint.x, (int) centerPoint.y));
        Log.e(TAG, "requestExposureFocus:" + manualFocus + ",touchX:" + touchX + ";touchY:" +
                touchY + ";viewWidth:" + viewWidth + ";viewHeight:" + viewHeight + ";areaMultiple:" + areaMultiple);
        return VcProperties.Constant.TRUE;
    }

    private int getCameraStatResult(boolean meterResult, boolean focusResult) {
        int camResult = 0;
        camResult = meterResult ? camResult + 1 : camResult;
        camResult = focusResult ? camResult + 1 : camResult;
        return camResult;
    }

    private void reportSwitchEvent(Metering oldMetering, Metering metering, int meterResult) {
        mCurrentSwitchTimes++;
        if (metering == null || oldMetering == null || oldMetering.getState() != metering.getState() || metering.getState() == Metering.ExposureStatus.MANUAL) {
            if (metering.getState() == Metering.ExposureStatus.DEFAULT) {
                //任何状态切换到Default状态都无需上报
                return;
            }
            //手动测光或者测光模式发生改变时上报
            int oldM = oldMetering == null ? Metering.ExposureStatus.CENTER_METERING.ordinal() : oldMetering.getState().ordinal();
            int newM = metering == null ? Metering.ExposureStatus.CENTER_METERING.ordinal() : metering.getState().ordinal();
            int manual = oldMetering != null ? oldMetering.getManualType().ordinal() :
                    (metering != null ? metering.getManualType().ordinal() : Metering.ExposureStatus.CENTER_METERING.ordinal());
                HEMediaReporter.reportSwitchMeter(oldM, newM, manual, mCurrentSwitchTimes, meterResult);
            mCurrentSwitchTimes = 0;
        }
    }
}