package sg.bigo.libvideo.cam.metering;

import android.graphics.Matrix;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.hardware.Camera;
import android.os.Build;

import sg.bigo.libvideo.cam.abs.Log;

public class CameraHelper {
    private static final String TAG = "CameraHelper";
    public static float DEFAULT_AREA_MULTIPLE = 0.15f;

    /**
     * @param point             控件点击位置
     * @param cameraOrientation 只包含了camera自身orientation
     * @param facingFront       是否前置
     * @param viewWidth
     * @param viewHeight
     * @param captureWidth
     * @param captureHeight
     * @return
     */
    public static PointF transFormPoint(PointF point, int cameraOrientation, boolean facingFront, int viewWidth, int viewHeight, int captureWidth, int captureHeight) {
        if (point == null || viewWidth <= 0 || viewHeight <= 0 || captureWidth <= 0 || captureHeight <= 0) {
            return null;
        }
        float ratioView = 1.0f * viewWidth / viewHeight;
        if (cameraOrientation == 90 || cameraOrientation == 270) {
            int temp = captureWidth;
            captureWidth = captureHeight;
            captureHeight = temp;
        }
        float ratioCapture = 1.0f * captureWidth / captureHeight;
        float ratio = 1.0f;
        if (ratioCapture > ratioView) {
            ratio = 1.0f * captureHeight / viewHeight;
            float w = viewWidth * ratio;
            float leftOffset = (captureWidth - w) / 2.0f;
            point.x = leftOffset + point.x * ratio;
            point.y = point.y * ratio;
        } else {
            ratio = 1.0f * captureWidth / viewWidth;
            float h = viewHeight * ratio;
            float topOffset = (captureHeight - h) / 2.0f;
            point.y = point.y * ratio + topOffset;
            point.x = point.x * ratio;
        }
        float points[] = new float[]{point.x, point.y};//计算过画面裁剪的point

        Matrix matrix = new Matrix();
        matrix.setScale(facingFront ? -1 : 1, 1, captureWidth / 2, captureHeight / 2);

        int rotation = 360 - cameraOrientation;//逆转camera旋转角度
        matrix.postRotate(rotation);

        int transX = captureWidth;//画面左上角移动到原点
        int transY = captureHeight;
        switch (rotation) {
            case 90: {
                transX = captureHeight;
                transY = 0;
            }
            break;
            case 180: {
                transX = captureWidth;
                transY = captureHeight;
            }
            break;
            case 270: {
                transX = 0;
                transY = captureWidth;
            }
            break;
        }
        matrix.postTranslate(transX, transY);

        matrix.mapPoints(points);
        PointF result = new PointF(points[0], points[1]);
        return result;
    }

    /**
     * 根据手动测光的点在camera回调画面上的对应位置计算矩形(比例未缩放到-1000~1000)
     *
     * @param manualCapturePoint 手动测光的点在camera回调画面上的对应位置
     * @param captureWidth       camera回调画面宽度
     * @param captureHeight      camera回调画面高度
     * @param areaMultiple       测光矩形最大边占画面的比例
     * @return
     */
    public static RectF calcTapRect(PointF manualCapturePoint, int captureWidth, int captureHeight, float areaMultiple) {
        float compensate = 1.5f;
        int rectWidth = captureWidth < captureHeight ? (int) (captureWidth * compensate * areaMultiple) : (int) (captureWidth * areaMultiple);
        int rectHeight = captureWidth < captureHeight ? (int) (captureHeight * areaMultiple) : (int) (captureHeight * compensate * areaMultiple);
        float halfWidth = rectWidth / 2.0f;
        float halfHeight = rectHeight / 2.0f;
        RectF rectF = new RectF(clamp(manualCapturePoint.x - halfWidth, 0, captureWidth),
                (int) clamp(manualCapturePoint.y - halfHeight, 0, captureHeight),
                (int) clamp(manualCapturePoint.x + halfWidth, 0, captureWidth),
                (int) clamp(manualCapturePoint.y + halfHeight, 0, captureHeight));
        return rectF;
    }

    /**
     * 根据手动测光的点在camera回调画面上的对应位置计算矩形
     *
     * @param manualCapturePoint 手动测光的点在camera回调画面上的对应位置
     * @param captureWidth       camera回调画面宽度
     * @param captureHeight      camera回调画面高度
     * @param areaMultiple       测光矩形最大边占画面的比例
     * @return
     */
    public static Rect calcMeterRect(PointF manualCapturePoint, int captureWidth, int captureHeight, float areaMultiple) {
        if (manualCapturePoint == null || captureWidth <= 0 || captureHeight <= 0 || areaMultiple <= 0) {
            return null;
        }
        RectF captureRect = calcTapRect(manualCapturePoint, captureWidth, captureHeight, areaMultiple);
        int left = (int) (2000 * captureRect.left / captureWidth - 1000);
        int top = (int) (2000 * captureRect.top / captureHeight - 1000);
        int right = (int) (2000 * captureRect.right / captureWidth - 1000);
        int bottom = (int) (2000 * captureRect.bottom / captureHeight - 1000);
        Rect meterRect = new Rect(left, top, right, bottom);
        Log.e(TAG, "calcMeterRect:" + meterRect);
        return meterRect;
    }

    /**
     *
     */
    public static int clamp(int value, int min, int max) {
        if (value > max) {
            return max;
        }
        if (value < min) {
            return min;
        }
        return value;
    }

    /**
     *
     */
    public static float clamp(float value, float min, float max) {
        if (value > max) {
            return max;
        }
        if (value < min) {
            return min;
        }
        return value;
    }

    /**
     * https://issuetracker.googdetectCamera2Supportle.com/issues/37014456
     * @return whether the os version is 5.1.1 or up
     * 是否支持 camera api2 (使用 ImageReader 的时候，使用此方法判断)
     */
    public static boolean detectCamera2Support() {
        Log.e(TAG, "detectCamera2Support");
        final int[] baseOsVersion = {5, 1, 0};
        String[] osVersion = Build.VERSION.RELEASE.split("\\.");
        boolean osQualified = false;

        try {
            for (int i = 0; i < osVersion.length; i++) {
                int val = Integer.parseInt(osVersion[i]);
                if (val > baseOsVersion[i]) {
                    osQualified = true;
                    break;
                } else if (val < baseOsVersion[i]) {
                    break;
                }
            }
        } catch (RuntimeException e) {
            Log.e(TAG, "failed to parse Build.VERSION.RELEASE:\t" + Build.VERSION.RELEASE, e);
            osQualified = Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1;
        }

        return osQualified;
    }

    // 获取前、后置相机数量，
    public static int getCameraCount(int cameraFacingFront) {
        int frontCameraNum = 0;
        int backCameraNum = 0;
        int numberOfCameras = Camera.getNumberOfCameras();
        Camera.CameraInfo info = new Camera.CameraInfo();
        for (int i = 0; i < numberOfCameras; i++) {
            Camera.getCameraInfo(i, info);
            if (info.facing == Camera.CameraInfo.CAMERA_FACING_FRONT) {
                frontCameraNum++;
            } else if (info.facing == Camera.CameraInfo.CAMERA_FACING_BACK) {
                backCameraNum++;
            }
        }
        return cameraFacingFront == Camera.CameraInfo.CAMERA_FACING_FRONT ? frontCameraNum : backCameraNum;
    }
}