package sg.bigo.libvideo.cam.camera2;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.ImageFormat;
import android.graphics.SurfaceTexture;
import android.hardware.camera2.CameraAccessException;
import android.hardware.camera2.CameraCaptureSession;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraDevice;
import android.hardware.camera2.CameraManager;
import android.hardware.camera2.CaptureRequest;
import android.hardware.camera2.params.StreamConfigurationMap;
import android.media.Image;
import android.media.ImageReader;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.util.Size;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import java.lang.ref.SoftReference;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import sg.bigo.libvideo.cam.abs.VcStatusType;
import sg.bigo.libvideo.cam.runtime.VcContext;
import sg.bigo.libvideo.cam.abs.VcCamera;
import sg.bigo.libvideo.cam.abs.VcCharacteristics;
import sg.bigo.libvideo.cam.abs.VcListener;
import sg.bigo.libvideo.cam.abs.VcProperties;
import sg.bigo.libvideo.cam.abs.VcDeviceParam;
import sg.bigo.libvideo.cam.runtime.VcRuntime;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class VcCamera2Impl implements VcCamera, ImageReader.OnImageAvailableListener {
    private final String TAG = "VcCamera2Impl";
    private VcListener mListener;
    private CameraHandler mHandler;
    private VcCharacteristics mCharacteristics;
    private String mCameraIndex;
    private ImageReader mImageReader;
    private CameraDevice mCameraDevice;
    private CameraCaptureSession mCaptureSession;
    private CameraCharacteristics mDevCharacteristics;
    private CaptureRequest.Builder mPreviewRequestBuilder;
    private Semaphore mSemaPhore = new Semaphore(0);
    private ByteBuffer mFrameBuffer;
    private int mWidth;
    private int mHeight;
    private int mStatus;
    private int mOrientation;
    private int mIsFacingFront;
    private int mPreferWidth;
    private int mPreferHeight;
    private int mPreferBufferType;

    private Object mLock = new Object();

    public VcCamera2Impl() {
    }

    @Override
    public void setListener(VcListener listener) {
        mListener = listener;
    }

    @SuppressLint("MissingPermission")
    @Override
    public int prepare(VcCharacteristics characteristics, int preferWidth, int preferHeight, int bufferType) {
        synchronized (mLock) {
            mCharacteristics = characteristics;
            mPreferWidth = preferWidth;
            mPreferHeight = preferHeight;
            mPreferBufferType = bufferType;
        }
        return 0;
    }

    @SuppressLint("MissingPermission")
    @Override
    public int open() {
        synchronized (mLock) {
            if (mPreferWidth <= 0 || mPreferHeight <= 0 || mListener == null) {
                return VcProperties.Constant.FALSE;
            }

            CameraManager manager = (CameraManager) VcContext.appContext().getSystemService(Context.CAMERA_SERVICE);
            if (mHandler == null) {
                HandlerThread thread = new HandlerThread(TAG);
                thread.start();
                mHandler = new CameraHandler(this, thread.getLooper());
            }
            mCameraIndex = mCharacteristics.getCameraIndex();
            try {
                manager.openCamera(mCameraIndex, stateCallback, mHandler);
                mDevCharacteristics = manager.getCameraCharacteristics(mCameraIndex);
                ((VcCamera2Characteristics) mCharacteristics).init(mDevCharacteristics);
                mIsFacingFront = isFacingFront();
                mOrientation = getOrientation();
            } catch (CameraAccessException e) {
                e.printStackTrace();
            }
            if (mImageReader == null) {
                int[] previewSize = mListener.onChoosePreviewSize(mPreferWidth, mPreferHeight, packSupportedPreviewSize());
                mWidth = previewSize[0];
                mHeight = previewSize[1];
                mImageReader = ImageReader.newInstance(mWidth, mHeight, ImageFormat.YUV_420_888, 2);
                mImageReader.setOnImageAvailableListener(this, mHandler);
                try {
                    mSemaPhore.tryAcquire(4000, TimeUnit.MILLISECONDS);
                    mPreviewRequestBuilder = mCameraDevice.createCaptureRequest(CameraDevice.TEMPLATE_PREVIEW);
                    mPreviewRequestBuilder.addTarget(mImageReader.getSurface());
                    mCameraDevice.createCaptureSession(Arrays.asList(mImageReader.getSurface()), sessionStateCallback, mHandler);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                } catch (CameraAccessException e) {
                    e.printStackTrace();
                }
            }
        }
        return 0;
    }

    private int isFacingFront() {
        List<Integer> facingFront = new ArrayList<>();
        mCharacteristics.isFacingFront(facingFront);
        if (facingFront.size() <= 0) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        return facingFront.get(0);
    }

    private int getOrientation() {
        List<Integer> orientations = new ArrayList<>();
        mCharacteristics.getOrientation(orientations);
        if (orientations.size() <= 0) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        return orientations.get(0);
    }

    @Override
    public int close() {
        synchronized (mLock) {
            if (mCaptureSession != null) {
                mCaptureSession.close();
                mCaptureSession = null;
            }
            if (mCameraDevice != null) {
                mCameraDevice.close();
                mCameraDevice = null;
            }
            if (mImageReader != null) {
                mImageReader.close();
                mImageReader = null;
            }
            if (mHandler != null) {
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLock) {
                            mHandler.getLooper().quit();
                            mHandler = null;
                        }
                    }
                });
            }
        }
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int applyParam(int key, int[] paramsInt) {
        return 0;
    }

    @Override
    public int queryParam(int key, List<Integer> result) {
        return 0;
    }


    @Override
    public VcCharacteristics getCharacteristics() {
        return null;
    }

    @Override
    public SurfaceTexture getSurfaceTexture() {
        return null;
    }

    @Override
    public String getCameraIndex() {
        return null;
    }

    private int[] packSupportedPreviewSize() {
        if (mDevCharacteristics == null) {
            return null;
        }
        StreamConfigurationMap map = mDevCharacteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
        Size[] sizes = map.getOutputSizes(SurfaceTexture.class);
        if (sizes != null && sizes.length > 0) {
            int[] packedSizes = new int[sizes.length * 2];
            for (int i = 0; i < sizes.length; i++) {
                packedSizes[i * 2] = sizes[i].getWidth();
                packedSizes[i * 2 + 1] = sizes[i].getHeight();
            }
            return packedSizes;
        }
        return null;
    }

    @Override
    public void onImageAvailable(ImageReader reader) {
        Image image = reader.acquireLatestImage();
        try {
            if (image != null && !VcRuntime.instance().needDropFrame()) {
                Image.Plane[] planes = image.getPlanes();
                int width = image.getWidth();
                int heigth = image.getHeight();
                if (mFrameBuffer == null) {
                    mFrameBuffer = ByteBuffer.allocateDirect(width * heigth * 3 / 2);
                }
                mFrameBuffer.position(0);//暂时只处理y分量
                mFrameBuffer.put(planes[0].getBuffer());
                mListener.onFrameAvailable(mFrameBuffer.array());
            }
        } finally {
            image.close();
        }
    }

    static class CameraHandler extends Handler {
        final SoftReference<VcCamera2Impl> reference;

        public CameraHandler(VcCamera2Impl camera, Looper looper) {
            super(looper);
            reference = new SoftReference<>(camera);
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            super.handleMessage(msg);
            VcCamera2Impl camera = reference.get();
            if (camera != null) {
                camera.handleMessage(msg);
            }
        }
    }

    private void handleMessage(Message msg) {
    }

    CameraDevice.StateCallback stateCallback = new CameraDevice.StateCallback() {
        @Override
        public void onOpened(@NonNull CameraDevice camera) {
            synchronized (mLock) {
                mCameraDevice = camera;
                mSemaPhore.release();
            }
        }

        @Override
        public void onDisconnected(@NonNull CameraDevice camera) {
            VcCamera2Impl.this.close();
        }

        @Override
        public void onError(@NonNull CameraDevice camera, int error) {
            VcCamera2Impl.this.close();
        }
    };

    CameraCaptureSession.StateCallback sessionStateCallback = new CameraCaptureSession.StateCallback() {
        @Override
        public void onConfigured(@NonNull CameraCaptureSession session) {
            try {
                mCaptureSession = session;
                mCaptureSession.setRepeatingRequest(mPreviewRequestBuilder.build(), null, mHandler);
                mStatus = VcProperties.CameraStatus.kStatus_Opened;
                notifyStatusChanged();
            } catch (CameraAccessException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onConfigureFailed(@NonNull CameraCaptureSession session) {
        }
    };

    private void notifyStatusChanged() {
        if (mListener != null) {
//            int[] params = new int[10];
//            params[VcDeviceParam.CAMERA_STATUS] = mStatus;
//            params[VcDeviceParam.PREVIEW_WIDTH] = mWidth;
//            params[VcDeviceParam.PREVIEW_HEIGHT] = mHeight;
//            params[VcDeviceParam.ORIENTATION] = mOrientation;
//            params[VcDeviceParam.FACING_FRONT] = mIsFacingFront;
//            params[VcDeviceParam.FLASH_SUPPORTED] = VcProperties.Constant.UNAVAILABLE;
//            params[VcDeviceParam.FLASH_MODE_TORCH_SUPPORTED] = VcProperties.Constant.UNAVAILABLE;
//            params[VcDeviceParam.ZOOM_SUPPORTED] = VcProperties.Constant.UNAVAILABLE;
//            params[VcDeviceParam.FOCUS_METERING_AVAILABLE] = VcProperties.Constant.UNAVAILABLE;
//            mListener.onCameraStatusChanged(mCameraIndex, VcStatusType.TypeStatus, params, null);
        }
    }
}
