package sg.bigo.libvideo.cam.camera1;

import android.content.Context;
import android.graphics.ImageFormat;
import android.graphics.SurfaceTexture;
import android.hardware.Camera;
import android.os.SystemClock;
import android.text.TextUtils;
import android.view.Surface;
import android.view.WindowManager;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import sg.bigo.libvideo.cam.abs.Log;
import sg.bigo.libvideo.cam.abs.VcCameraKey;
import sg.bigo.libvideo.cam.abs.VcErrorIndex;
import sg.bigo.libvideo.cam.abs.VcStatusType;
import sg.bigo.libvideo.cam.report.HEMediaReporter;
import sg.bigo.libvideo.cam.runtime.VcContext;
import sg.bigo.libvideo.cam.abs.VcCamera;
import sg.bigo.libvideo.cam.abs.VcCharacteristics;
import sg.bigo.libvideo.VcABConfig;
import sg.bigo.libvideo.cam.metering.ManualType;
import sg.bigo.libvideo.cam.abs.VcListener;
import sg.bigo.libvideo.cam.abs.VcProperties;
import sg.bigo.libvideo.cam.metering.MeteringDelegate;
import sg.bigo.libvideo.cam.runtime.VcRuntime;

public class VcCameraImpl implements VcCamera, Camera.PreviewCallback {

    private static final String TAG = "VcCameraImpl";
    private static final int DEFAULT_BUFFER_COUNT = 2;
    private Camera mCamera;
    private Camera.Parameters mParameters;

    private VcCharacteristics mCharacteristics;
    private SurfaceTexture mSurfaceTexture;
    private VcListener mListener;
    private String mCameraIndex;
    private int mOrientation = 0;
    private LinkedList<byte[]> mFrameBufferList = new LinkedList();
    private int mCaptureWidth = 1280;
    private int mCaptureHeight = 720;
    private MeteringDelegate mMeteringDelegate;
    private ReentrantReadWriteLock mLock = new ReentrantReadWriteLock();
    private HashMap<Integer, VcCameraKey.IParam> mCameraParams = new HashMap<>();
    private boolean isMeteringStateInited = false;
    private ManualType mMeteringType = ManualType.TYPE_MANUAL_CLOSE;
    private long mCameraOpenTS = Long.MIN_VALUE;
    private long mCameraOpenSuccessedTS = Long.MIN_VALUE;
    private boolean mIsFirstFrame = true;

    public VcCameraImpl() {
    }

    @Override
    public void setListener(VcListener listener) {
        mLock.writeLock().lock();
        mListener = listener;
        mLock.writeLock().unlock();
    }

    @Override
    public int prepare(VcCharacteristics characteristics, int preferWidth, int preferHeight, int bufferType) {
        mLock.writeLock().lock();
        try {
            mCharacteristics = characteristics;
            mCameraIndex = mCharacteristics.getCameraIndex();
            Log.e(TAG, "prepare cameraIndex:" + mCameraIndex + ";[" + preferWidth + "," + preferHeight + "],bufferType:" + bufferType);
            mCharacteristics.prepare(preferWidth, preferHeight, bufferType);
        } finally {
            mLock.writeLock().unlock();
        }
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int open() {
        mLock.readLock().lock();
        try {
            if (mCharacteristics == null) {
                notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true,
                        getErrorCameraState("openUnprepared", null));
                return VcProperties.Constant.UNAVAILABLE;
            }
            List<Integer> preparedData = new ArrayList<>();
            mCharacteristics.getPreferSize(preparedData);
            mCharacteristics.getBufferType(preparedData);
            int preferWidth = preparedData.get(0);
            int preferHeight = preparedData.get(1);
            int bufferType = preparedData.get(2);
            Log.e(TAG, "open cameraIndex:" + mCameraIndex + ";preferSize[" + preferWidth + "," + preferHeight + "],bufferType:" + bufferType);
            if (mCharacteristics instanceof VcCameraCharacteristics) {
                int index = -1;
                if (!TextUtils.isEmpty(mCameraIndex) && (index = Integer.parseInt(mCameraIndex)) >= 0) {
                    Log.e(TAG, "index:" + mCameraIndex + ";" + VcCameraImpl.this.hashCode());
                    mCameraOpenTS = SystemClock.elapsedRealtime();
                    mCamera = Camera.open(index);
                    reportOpenDelay();
                    mCamera.setErrorCallback(mErrorCallback);
                    if (mCamera == null) {
                        notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true, getErrorCameraState("openFailed index:" + index, null));
                        return VcProperties.Constant.FALSE;
                    }
                    HEMediaReporter.reportCameraOperations(HEMediaReporter.Field.CAMERA_OPEN_STATUS,
                            VcProperties.Constant.TRUE + "", VcProperties.API.CAMERA1);
                    mParameters = mCamera.getParameters();
                    reportDefaultFpsRange();
                    initAEAFStrategy();
                    ((VcCameraCharacteristics) mCharacteristics).initCharacteristics(this);
                } else {
                    notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true, getErrorCameraState("wrongIndex:" + mCameraIndex, null));
                    return VcProperties.Constant.FALSE;
                }
            } else {
                notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true, getErrorCameraState(mCharacteristics == null ? "null characteristics" : "unSupported characteristics", null));
                return VcProperties.Constant.FALSE;
            }
            return startPreview(preferWidth, preferHeight, bufferType);
        } catch (Exception e) {
            notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true, getErrorCameraState("open", e));
            return VcProperties.Constant.FALSE;
        } finally {
            mLock.readLock().unlock();
        }
    }

    private void clearPreviewCallbacks() {
        try {
            mCamera.setFaceDetectionListener(null);
            mCamera.setErrorCallback(null);
            mCamera.setPreviewCallback(null);
            mCamera.stopPreview();
            if (mSurfaceTexture != null) {
                mSurfaceTexture.release();
                mSurfaceTexture = null;
            }
        } catch (Exception e) {
        }
    }

    @Override
    public int close() {
        if (mCamera == null || mParameters == null) {
            return VcProperties.Constant.FALSE;
        }
        Log.e(TAG, "close");
        try {
            clearPreviewCallbacks();
            mCamera.release();
            if (mCharacteristics != null) {
                mCharacteristics.setCameraStatus(VcProperties.CameraStatus.kStatus_Closed);
            }
            if (mCharacteristics instanceof VcCameraCharacteristics) {
                ((VcCameraCharacteristics) mCharacteristics).releaseCharacteristics();
            }
            notifyStatusChanged();
            mCamera = null;
            mParameters = null;
            mMeteringDelegate = null;
            mIsFirstFrame = true;
            HEMediaReporter.reportCameraOperations(HEMediaReporter.Field.CAMERA_CLOSE_STATUS,
                    VcProperties.Constant.TRUE + "", VcProperties.API.CAMERA1);
        } catch (Exception e) {
            notifyError(VcProperties.ErrCode.kErr_CloseCamera, 0, true, getErrorCameraState("close", e));
        }
        Log.e(TAG, "close end");
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int applyParam(int key, int[] paramsInt) {
        if (mCamera == null || mParameters == null) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        VcCameraKey.IParam param = mCameraParams.get(key);
        if (param == null) {
            param = VcCameraKey.newParamInstance(key);
            if (param == null) {
                Log.e(TAG, "applyParam newParamInstance failed");
                return VcProperties.Constant.FALSE;
            }
            mCameraParams.put(key, param);
        }
        return param.applyParam(mCharacteristics, paramsInt);
    }

    @Override
    public int queryParam(int key, List<Integer> result) {
        if (mCamera == null || mParameters == null) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        VcCameraKey.IParam param = mCameraParams.get(key);
        if (param == null) {
            param = VcCameraKey.newParamInstance(key);
            if (param == null) {
                Log.e(TAG, "queryParam newParamInstance failed");
                return VcProperties.Constant.FALSE;
            }
            mCameraParams.put(key, param);
        }
        return param.queryParam(mCharacteristics, result);
    }

    @Override
    public VcCharacteristics getCharacteristics() {
        return mCharacteristics;
    }

    @Override
    public SurfaceTexture getSurfaceTexture() {
        return mSurfaceTexture;
    }

    @Override
    public String getCameraIndex() {
        return mCameraIndex;
    }

    @Override
    public void onPreviewFrame(byte[] data, Camera camera) {
        reportOpenedToFirstFrameDelay();
        if (data != null && !VcRuntime.instance().needDropFrame()) {
            if (mListener != null) {
                mListener.onFrameAvailable(data);
                if (mMeteringDelegate != null && isMeteringStateInited
                        && mMeteringType.ordinal() > ManualType.TYPE_MANUAL_CLOSE.ordinal()) {
                    mMeteringDelegate.onFrameAvailable(data, mCaptureWidth, mCaptureHeight);
                }
            }
        }
        camera.addCallbackBuffer(data);
    }

    protected Camera getCamera() {
        return mCamera;
    }

    protected Camera.Parameters getParameters() {
        return mParameters;
    }

    protected MeteringDelegate getMeteringDelegate() {
        return mMeteringDelegate;
    }

    private void notifyError(int errorType, int code, boolean needReport, String message) {
        mLock.readLock().lock();
        try {
            if (mListener == null) {
                return;
            }
            int[] errorCode = new int[VcErrorIndex.SIZE];
            errorCode[VcErrorIndex.ERROR_CODE] = errorType;
            errorCode[VcErrorIndex.ERROR_SUBCODE] = code;
            errorCode[VcErrorIndex.REPORT_NEEDED] = needReport ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
            mListener.onCameraStatusChanged(mCameraIndex, VcStatusType.TypeErr, errorCode, message);
        } finally {
            mLock.readLock().unlock();
        }
    }

    private void reportOpenDelay() {
        if (mCameraOpenTS == Long.MIN_VALUE) {
            return;
        }
        mCameraOpenSuccessedTS = SystemClock.elapsedRealtime();
        long timeCost = mCameraOpenSuccessedTS - mCameraOpenTS;
        HEMediaReporter.reportCameraOperations(HEMediaReporter.Field.CAMERA_OPEN_DELAY,
                timeCost + "", VcProperties.API.CAMERA1);
        mCameraOpenTS = Long.MIN_VALUE;
    }

    private void reportOpenedToFirstFrameDelay() {
        if (mIsFirstFrame == false || mCameraOpenSuccessedTS == Long.MIN_VALUE) {
            return;
        }
        long timeCost = SystemClock.elapsedRealtime() - mCameraOpenSuccessedTS;
        HEMediaReporter.reportCameraOperations(HEMediaReporter.Field.OPENED_TO_FIRST_FRAME_DELAY,
                timeCost + "", VcProperties.API.CAMERA1);
        mIsFirstFrame = false;
        mCameraOpenSuccessedTS = Long.MIN_VALUE;
    }

    private void reportPreviewSize() {
        Map<String, String> previewSize = new HashMap<>();
        previewSize.put(HEMediaReporter.Field.CAMERA_RAW_WIDTH, mCaptureWidth + "");
        previewSize.put(HEMediaReporter.Field.CAMERA_RAW_HEIGHT, mCaptureHeight + "");
        HEMediaReporter.reportCameraAttributes(previewSize);
    }

    // 上报默认采集帧率区间
    private void reportDefaultFpsRange() {
        int[] fpsRange = new int[2];
        Map<String, String> defaultFPSRange = new HashMap<>();
        mParameters.getPreviewFpsRange(fpsRange);
        defaultFPSRange.put(HEMediaReporter.Field.PREVIEW_FPS_DEFAULT,
                "(" + fpsRange[Camera.Parameters.PREVIEW_FPS_MIN_INDEX] + ","
                        + fpsRange[Camera.Parameters.PREVIEW_FPS_MAX_INDEX] + ")");
        HEMediaReporter.reportCameraAttributes(defaultFPSRange);
    }

    private String getErrorCameraState(String message, Throwable e) {
        return HEMediaReporter.getExceptionString(message, e, mCameraIndex, isFacingFront(),
                mCaptureWidth, mCaptureHeight, mCharacteristics.getCameraStatus());
    }

    private void notifyStatusChanged() {
        mLock.readLock().lock();
        try {
            if (mListener != null && mCharacteristics != null) {
                Object ext = null;
                int status = mCharacteristics.getCameraStatus();
                if (status == VcProperties.CameraStatus.kStatus_Opened) {
                    ext = mSurfaceTexture;
                }
                mListener.onCameraStatusChanged(mCameraIndex, VcStatusType.TypeStatus, getDeviceParams(), ext);
            }
        } finally {
            mLock.readLock().unlock();
        }
    }

    private int isFacingFront() {
        List<Integer> facingFront = new ArrayList<>();
        mCharacteristics.isFacingFront(facingFront);
        if (facingFront.size() <= 0) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        return facingFront.get(0);
    }

    private int getOrientation() {
        List<Integer> orientations = new ArrayList<>();
        mCharacteristics.getOrientation(orientations);
        if (orientations.size() <= 0) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        return orientations.get(0);
    }

    private int calcPreviewOrientation(int displayRotation) {
        int rotation = displayRotation;
        mOrientation = getOrientation();
        int result;
        if (isFacingFront() == VcProperties.Constant.TRUE) {
            result = (mOrientation + rotation) % 360;
            result = (360 - result) % 360;
        } else if (isFacingFront() == VcProperties.Constant.FALSE) {
            result = (mOrientation - rotation + 360) % 360;
        } else {
            result = -1;
        }
        return result;
    }

    private int getDisplayRotate() {
        Context context = VcContext.appContext();
        if (context == null) {
            return 270;
        }
        int m_Orientation = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE))
                .getDefaultDisplay()
                .getRotation();
        int degrees = 0;
        switch (m_Orientation) {
            case Surface.ROTATION_0:
                degrees = 0;
                break;
            case Surface.ROTATION_90:
                degrees = 90;
                break;
            case Surface.ROTATION_180:
                degrees = 180;
                break;
            case Surface.ROTATION_270:
                degrees = 270;
                break;
        }
        return degrees;
    }

    private void initAEAFStrategy() {
        // 手动测光对焦策略 AB 实验
        if (isMeteringStateInited == false) {
            mMeteringType = VcABConfig.getMeteringFocusConfig();
            isMeteringStateInited = true;
        }
        if (isMeteringStateInited && mMeteringType.ordinal() > ManualType.TYPE_MANUAL_CLOSE.ordinal()) {
            mMeteringDelegate = new MeteringDelegate(mCharacteristics);
            // 1 lock, 2 smart, 3 face
            mMeteringDelegate.setMeteringConfig(mMeteringType);
        }
    }

    private StringBuilder checkValid() {
        if (mCamera == null || mParameters == null || mCharacteristics == null || mListener == null) {
            StringBuilder errorMessage = new StringBuilder();
            if (mCamera == null) {
                errorMessage.append("mCamera == null;");
            }
            if (mParameters == null) {
                errorMessage.append("mParameters == null;");
            }
            if (mCharacteristics == null) {
                errorMessage.append("mCharacteristics == null;");
            }
            if (mListener == null) {
                errorMessage.append("mListener == null;");
            }
            return errorMessage;
        }
        return null;
    }

    private int startPreview(int preferWidth, int preferHeight, int preferBufferType) throws IOException {
        StringBuilder errorMessage = null;
        if ((errorMessage = checkValid()) != null || preferWidth <= 0 || preferHeight <= 0 || preferBufferType <= 0) {
            if (errorMessage == null) {
                errorMessage = new StringBuilder();
            }
            errorMessage.append("[").append(preferWidth).append("*").append(preferHeight).append("];").append(preferBufferType);
            notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true, getErrorCameraState(errorMessage.toString(), null));
            return VcProperties.Constant.FALSE;
        }
        Log.e(TAG, "_startPreview,w:" + preferWidth + ";h:" + preferHeight + ";bufferType:" + preferBufferType + ";mCameraIndex:" + mCameraIndex);
        int rotation = calcPreviewOrientation(getDisplayRotate());
        mCamera.setDisplayOrientation(rotation);
        int[] previewSize = null;
        mLock.readLock().lock();
        try {
            previewSize = mListener.onChoosePreviewSize(preferWidth, preferHeight, packSortedSupportedPreviewSize());
        } finally {
            mLock.readLock().unlock();
        }
        if (previewSize != null && previewSize.length == 2) {
            mCaptureWidth = previewSize[0];
            mCaptureHeight = previewSize[1];
            reportPreviewSize();
            mCharacteristics.setPreviewSize(previewSize);
            mSurfaceTexture = new SurfaceTexture(0);
            mParameters.setPreviewSize(mCaptureWidth, mCaptureHeight);
            int outputType = preferBufferType;
            if (outputType == VcProperties.OutputType.NV21 || outputType == VcProperties.OutputType.YUV420P) {
                mParameters.setPreviewFormat(ImageFormat.NV21);
                int frameSize = mCaptureWidth * mCaptureHeight * 3 / 2;
                mFrameBufferList.clear();
                for (int i = 0; i < DEFAULT_BUFFER_COUNT; i++) {
                    mFrameBufferList.offer(new byte[frameSize]);
                    mCamera.addCallbackBuffer(mFrameBufferList.poll());
                }
                mCamera.setPreviewCallbackWithBuffer(this);
            } else if (outputType == VcProperties.OutputType.SURFACE_TEXTURE) {
                //SurfaceTexture数据流camera=>gl，要想获取帧数据需要在gl环境中读图
            } else {
                notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true, getErrorCameraState("unsupported outputType", null));
                return VcProperties.Constant.FALSE;
            }
        } else {
            notifyError(VcProperties.ErrCode.kErr_OpenCamera, 0, true,
                    getErrorCameraState((previewSize == null ? "previewSizeNull" : "previewSizeLength:" + previewSize.length)
                            + "[" + preferWidth + "," + preferHeight + "]", null));
            return VcProperties.Constant.FALSE;
        }
        mCamera.setParameters(mParameters);
        mCamera.setPreviewTexture(mSurfaceTexture);
        mCamera.startPreview();
        mCharacteristics.setCameraStatus(VcProperties.CameraStatus.kStatus_Opened);
        notifyStatusChanged();
        // 手动测光对焦策略 AB 实验?
        if (isMeteringStateInited
                && mMeteringType.ordinal() > ManualType.TYPE_MANUAL_CLOSE.ordinal()) {
            mMeteringDelegate.startFaceDetect(mCamera);
        }
        return VcProperties.Constant.TRUE;
    }

    private int[] getDeviceParams() {
        List<Integer> params = new ArrayList<>();
        queryParam(VcCameraKey.Device.key(), params);
        int paramSize = params.size();
        if (paramSize <= 0) {
            return new int[]{VcProperties.CameraStatus.kStatus_Closed};
        }
        int[] result = new int[paramSize];
        for (int i = 0; i < paramSize; i++) {
            result[i] = params.get(i);
        }
        return result;
    }

    private int[] packSortedSupportedPreviewSize() {
        if (mParameters == null) {
            Log.e(TAG, "packSortedSupportedPreviewSize mParameters == null");
            return null;
        }
        Camera.Size[] sortedPreviewSize = sortPreviewSize(mParameters.getSupportedPreviewSizes());
        if (sortedPreviewSize != null && sortedPreviewSize.length > 0) {
            int length = sortedPreviewSize.length;
            int[] packedSizes = new int[length * 2];
            for (int i = 0; i < length; i++) {
                packedSizes[i * 2] = sortedPreviewSize[i].width;
                packedSizes[i * 2 + 1] = sortedPreviewSize[i].height;
            }
            return packedSizes;
        }
        Log.e(TAG, "packSortedSupportedPreviewSize sortedPreviewSize empty");
        return null;
    }

    private Camera.Size[] sortPreviewSize(List<Camera.Size> supportedSizes) {
        Camera.Size[] sizes = new Camera.Size[supportedSizes.size()];
        for (int index = 0; index < supportedSizes.size(); index++) {
            sizes[index] = supportedSizes.get(index);
        }
        Arrays.sort(sizes, new Comparator<Camera.Size>() {
            @Override
            public int compare(Camera.Size lhs, Camera.Size rhs) {
                // 按照视频的宽排序，宽一样的按照高排序
                if (lhs.width != rhs.width) {
                    return lhs.width - rhs.width;
                } else {
                    return lhs.height - rhs.height;
                }
            }
        });
        for (Camera.Size size : sizes) {
            Log.e(TAG, "camera size: (" + size.width + ", " + size.height + ")");
        }
        return sizes;
    }

    Camera.ErrorCallback mErrorCallback = new Camera.ErrorCallback() {
        @Override
        public void onError(int error, Camera camera) {
            notifyError(VcProperties.ErrCode.kErr_SystemErr, error, true, getErrorCameraState("system_err", null));
            close();
        }
    };
}
