package sg.bigo.libvideo.cam.camera1;

import android.graphics.Rect;
import android.hardware.Camera;
import android.text.TextUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import sg.bigo.libvideo.cam.abs.Log;
import sg.bigo.libvideo.cam.abs.VcCharacteristics;
import sg.bigo.libvideo.cam.abs.VcProperties;
import sg.bigo.libvideo.cam.abs.VcRectAngle;
import sg.bigo.libvideo.cam.abs.VcSize;
import sg.bigo.libvideo.cam.metering.MeteringDelegate;

/**
 * camera参数查询与设置，为了避免线程同步问题查询/设置操作需要在camera线程中执行
 */
public class VcCameraCharacteristics implements VcCharacteristics {
    private final String TAG = "VcCameraCharacteristics";
    private Camera mCamera;
    private Camera.Parameters mParameter;
    private Camera.CameraInfo mCameraInfo;
    private VcCameraImpl mVcCameraImpl;
    private String mCameraIndex = "-1";
    private VcSize mPreferSize = new VcSize(1280, 720);
    private VcSize mPreviewSize = new VcSize(1280, 720);
    private int mDefaultZoom = 0;
    private int mBufferType = 0;
    private AtomicInteger mStatus = new AtomicInteger(VcProperties.CameraStatus.kStatus_UnKnown);

    public VcCameraCharacteristics(String cameraIndex) {
        int index = -1;
        if (!TextUtils.isEmpty(cameraIndex) && (index = Integer.parseInt(cameraIndex)) >= 0) {
            mCameraInfo = new Camera.CameraInfo();
            Camera.getCameraInfo(index, mCameraInfo);
            mCameraIndex = cameraIndex;
        } else {
            throw new RuntimeException(TAG + " error construct cameraIndex:" + cameraIndex);
        }
    }

    public void initCharacteristics(VcCameraImpl vcCameraImpl) {
        mVcCameraImpl = vcCameraImpl;
        mCamera = mVcCameraImpl.getCamera();
        mParameter = mVcCameraImpl.getParameters();
        if (mParameter != null && mDefaultZoom == 0) {
            mDefaultZoom = mParameter.getZoom();
        }
    }

    public void releaseCharacteristics() {
        mVcCameraImpl = null;
        mCamera = null;
        mParameter = null;
        mDefaultZoom = 0;
    }

    @Override
    public String getCameraIndex() {
        return mCameraIndex;
    }

    @Override
    public int getAPILevel() {
        return VcProperties.API.CAMERA1;
    }

    @Override
    public void setCameraStatus(int status) {
        mStatus.set(status);
    }

    @Override
    public int getCameraStatus() {
        return mStatus.get();
    }

    @Override
    public int isFacingFront(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int facingFront = VcProperties.Constant.UNAVAILABLE;
        if (mCameraInfo != null) {
            facingFront = mCameraInfo.facing == Camera.CameraInfo.CAMERA_FACING_FRONT ?
                    VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
            result = VcProperties.Constant.TRUE;
        }
        outData.add(facingFront);
        return result;
    }

    public boolean isFacingFront() {
        List<Integer> isFacingFront = new ArrayList<>();
        isFacingFront(isFacingFront);
        if (isFacingFront.size() <= 0 || isFacingFront.get(0) <= 0) {
            return false;
        }
        return true;
    }

    @Override
    public int getOrientation(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int orientation = VcProperties.Constant.UNAVAILABLE;
        if (mCameraInfo != null) {
            orientation = mCameraInfo.orientation;
            result = VcProperties.Constant.TRUE;
        }
        outData.add(orientation);
        return result;
    }

    public int getOrientation() {
        List<Integer> orientations = new ArrayList<>();
        getOrientation(orientations);
        if (orientations.size() <= 0 || orientations.get(0) < 0) {
            return 270;
        }
        return orientations.get(0);
    }

    @Override
    public void prepare(int preferWidth, int preferHeight, int bufferType) {
        mPreferSize.setWidth(preferHeight);//来自编码模块的分辨率是宽高对调的
        mPreferSize.setHeight(preferWidth);
        mBufferType = bufferType;
        mStatus.set(VcProperties.CameraStatus.kStatus_Prepared);
    }

    @Override
    public int isZoomSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int zoomSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            zoomSupported = mParameter.isZoomSupported() ? VcProperties.Constant.TRUE
                    : VcProperties.Constant.FALSE;
        }
        outData.add(zoomSupported);
        return result;
    }

    @Override
    public int isSmoothZoomSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int smoothZoomSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            smoothZoomSupported = mParameter.isSmoothZoomSupported() ? VcProperties.Constant.TRUE
                    : VcProperties.Constant.FALSE;
        }
        outData.add(smoothZoomSupported);
        return result;
    }

    @Override
    public int isVideoStabilizationSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int videoStabilizationSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            videoStabilizationSupported = mParameter.isVideoStabilizationSupported() ? VcProperties.Constant.TRUE
                    : VcProperties.Constant.FALSE;
        }
        outData.add(videoStabilizationSupported);
        return result;
    }

    public boolean isVideoStabilizationSupported() {
        List<Integer> result = new ArrayList<>();
        isVideoStabilizationSupported(result);
        if (result.size() <= 0 || result.get(0) <= 0) {
            return false;
        }
        return true;
    }

    @Override
    public int getMaxZoom(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int maxZoom = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            maxZoom = mParameter.getMaxZoom();
        }
        outData.add(maxZoom);
        return result;
    }

    @Override
    public int getDefaultZoom(List<Integer> outData) {
        outData.add(mDefaultZoom);
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int isFocusSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int focusSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            focusSupported = mParameter.getMaxNumFocusAreas() > 0 ? VcProperties.Constant.TRUE
                    : VcProperties.Constant.FALSE;
        }
        outData.add(focusSupported);
        return result;
    }

    @Override
    public int isAutoFocusSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int autoFocusSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            autoFocusSupported = mParameter.getSupportedFocusModes()
                    .contains(Camera.Parameters.FOCUS_MODE_AUTO)
                    ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
        }
        outData.add(autoFocusSupported);
        return result;
    }

    public boolean isAutoFocusSupported() {
        List<Integer> autoFocusSupported = new ArrayList<>();
        isAutoFocusSupported(autoFocusSupported);
        if (autoFocusSupported.size() <= 0 || autoFocusSupported.get(0) <= 0) {
            return false;
        }
        return true;
    }

    @Override
    public int setAutoFocus() {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mCamera == null || !isAutoFocusSupported()) {
            return result;
        }
        mCamera.autoFocus(new Camera.AutoFocusCallback() {
            @Override
            public void onAutoFocus(boolean success, Camera camera) {
                Log.i(TAG, "[onAutoFocus] " + success + " done !");
                try {
                    camera.cancelAutoFocus();
                    if (mParameter == null) {
                        Log.e(TAG, "[onAutoFocus] mParameter is null");
                        return;
                    }
                    List<String> focusModes = mParameter.getSupportedFocusModes();
                    if (focusModes.contains(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO)) {
                        mParameter.setFocusMode(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO);
                        camera.setParameters(mParameter);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
        result = VcProperties.Constant.TRUE;
        return result;
    }

    @Override
    public int isVideoContinuousFocusSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int videoContinuousFocusSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            videoContinuousFocusSupported = mParameter.getSupportedFocusModes()
                    .contains(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO)
                    ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
        }
        outData.add(videoContinuousFocusSupported);
        return result;
    }

    public boolean isVideoContinuousFocusSupported() {
        List<Integer> videoContinuousFocusSupported = new ArrayList<>();
        isVideoContinuousFocusSupported(videoContinuousFocusSupported);
        if (videoContinuousFocusSupported.size() <= 0 || videoContinuousFocusSupported.get(0) <= 0) {
            return false;
        }
        return true;
    }

    @Override
    public int setFocusMode(int focusMode) {
        if (mParameter != null && mCamera != null) {
            mParameter.setFocusMode(VcProperties.getFocusModeName(focusMode));
            mCamera.setParameters(mParameter);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int getFocusMode(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int focusMode = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            focusMode = VcProperties.getFocusMode(mParameter.getFocusMode());
        }
        outData.add(focusMode);
        return result;
    }

    @Override
    public int isMeteringSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int meteringSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            meteringSupported = mParameter.getMaxNumMeteringAreas() > 0 ? VcProperties.Constant.TRUE
                    : VcProperties.Constant.FALSE;
        }
        outData.add(meteringSupported);
        return result;
    }

    @Override
    public int setMeteringArea(Rect rect, int weight) {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mParameter == null || rect == null || mCamera == null) {
            return result;
        }
        if (mParameter.getMaxNumMeteringAreas() > 0) {
            List<Camera.Area> areas = new ArrayList<>();
            areas.add(new Camera.Area(rect, weight));
            mParameter.setMeteringAreas(areas);
            result = VcProperties.Constant.TRUE;
        }
        if (mParameter.isAutoExposureLockSupported()) {
            mParameter.setAutoExposureLock(false);
        }
        if (mParameter.isAutoWhiteBalanceLockSupported()) {
            mParameter.setAutoWhiteBalanceLock(false);
        }
        mCamera.setParameters(mParameter);
        return result;
    }

    @Override
    public int resetMeteringState() {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mParameter == null || mCamera == null) {
            return result;
        }
        if (mParameter.getMaxNumMeteringAreas() > 0) {
            mParameter.setMeteringAreas(null);
            mCamera.setParameters(mParameter);
            result = VcProperties.Constant.TRUE;
        }
        return result;
    }

    @Override
    public int getSupportedPreviewSizes(List<Integer> outSizes) {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            List<Camera.Size> sizes = mParameter.getSupportedPreviewSizes();
            if (sizes != null) {
                int length = sizes.size();
                for (int i = 0; i < length; i++) {
                    Camera.Size size = sizes.get(i);
                    outSizes.add(size.width);
                    outSizes.add(size.height);
                }
            }
        }
        return result;
    }

    @Override
    public int getSupportedFlashModes(List<Integer> modes) {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            List<Integer> flashMode = new ArrayList<>();
            List<String> flashModes = mParameter.getSupportedFlashModes();
            if (flashModes != null) {
                for (String mode : flashModes) {
                    Integer m = VcProperties.getFlashMode(mode);
                    if (m != null) {
                        modes.add(m);
                        flashMode.add(m);
                    }
                }
            }
        }
        return result;
    }

    @Override
    public int getFocusAreas(ArrayList<VcRectAngle> areas) {
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int setFocusAreas(Rect rect, int weight) {
        int result = VcProperties.Constant.FALSE;
        if (mParameter == null || rect == null || mCamera == null) {
            return result;
        }
        if (mParameter.getMaxNumFocusAreas() > 0) {
            List<Camera.Area> focusAreas = new ArrayList<Camera.Area>();
            focusAreas.add(new Camera.Area(rect, weight));
            mParameter.setFocusAreas(focusAreas);
            result = VcProperties.Constant.TRUE;
        }
        List<String> focusModes = mParameter.getSupportedFocusModes();
        if (focusModes.contains(Camera.Parameters.FOCUS_MODE_AUTO)) {
            mCamera.cancelAutoFocus();
            mParameter.setFocusMode(Camera.Parameters.FOCUS_MODE_AUTO);
            setAutoFocus();
            mCamera.setParameters(mParameter);
        }
        return result;
    }

    @Override
    public int resetFocusState() {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mParameter == null || mCamera == null) {
            return result;
        }
        if (isAutoFocusSupported() && isVideoContinuousFocusSupported()) {
            Log.e(TAG, "resetFocus Area!");
            mCamera.cancelAutoFocus();
            mParameter.setFocusMode(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO);
            mCamera.setParameters(mParameter);
            result = VcProperties.Constant.TRUE;
        }
        return result;
    }

    @Override
    public int getMeteringAreas(ArrayList<VcRectAngle> areas) {
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int requestExposureFocus(int touchX, int touchY, int viewWidth, int viewHeight) {
        Log.e(TAG, "[requestExposureFocus]" + touchX + touchY + viewWidth + viewHeight);
        int orientation = getOrientation();
        int captureWidth = mPreviewSize.getWidth();
        int captureHeight = mPreviewSize.getHeight();
        boolean isFacingFront = isFacingFront();
        if (mVcCameraImpl == null) {
            Log.e(TAG, "[requestExposureFocus] failed, mVcCameraImpl is null");
            return VcProperties.Constant.UNAVAILABLE;
        }
        MeteringDelegate meteringDelegate = mVcCameraImpl.getMeteringDelegate();
        if (meteringDelegate == null) {
            Log.e(TAG, "[requestExposureFocus] failed, meteringDelegate is null");
            return VcProperties.Constant.UNAVAILABLE;
        }
        return meteringDelegate.requestExposureFocus(touchX, touchY, viewWidth, viewHeight,
                orientation, captureWidth, captureHeight, isFacingFront);
    }

    @Override
    public int setZoom(int zoom) {
        if (mParameter != null && mCamera != null) {
            mParameter.setZoom(zoom);
            mCamera.setParameters(mParameter);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int getZoom(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int zoom = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            zoom = mParameter.getZoom();
        }
        outData.add(zoom);
        return result;
    }

    @Override
    public int getPreferSize(List<Integer> size) {
        int result = VcProperties.Constant.UNAVAILABLE;
        if (mPreferSize != null) {
            result = VcProperties.Constant.TRUE;
            size.add(mPreferSize.getWidth());
            size.add(mPreferSize.getHeight());
        } else {
            size.add(VcProperties.Constant.UNAVAILABLE);
            size.add(VcProperties.Constant.UNAVAILABLE);
        }
        return result;
    }

    @Override
    public int setPreviewSize(int[] size) {
        mPreviewSize.setWidth(size[0]);
        mPreviewSize.setHeight(size[1]);
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int setBufferType(int bufferType) {
        mBufferType = bufferType;
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int getBufferType(List<Integer> outData) {
        outData.add(mBufferType);
        return VcProperties.Constant.TRUE;
    }

    @Override
    public int getPreviewSize(List<Integer> outSize) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int previewWidth = VcProperties.Constant.UNAVAILABLE;
        int previewHeight = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            previewWidth = mPreviewSize.getWidth();
            previewHeight = mPreviewSize.getHeight();
        }
        outSize.add(previewWidth);
        outSize.add(previewHeight);
        return result;
    }

    @Override
    public int setFlashMode(int mode) {
        if (mParameter != null && mCamera != null) {
            mParameter.setFlashMode(VcProperties.getFlashModeName(mode));
            mCamera.setParameters(mParameter);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int isFlashSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int flashSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            flashSupported = mParameter.getSupportedFlashModes() != null
                    ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
        }
        outData.add(flashSupported);
        return result;
    }

    @Override
    public int isFlashModeTorchSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int flashModeTorchSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            if (mParameter.getSupportedFlashModes() != null) {
                flashModeTorchSupported = mParameter.getSupportedFlashModes().contains(Camera.Parameters.FLASH_MODE_TORCH)
                        ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
            }
        }
        outData.add(flashModeTorchSupported);
        return result;
    }

    @Override
    public int getFlashMode(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int flashMode = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            flashMode = VcProperties.getFlashMode(mParameter.getFlashMode());
        }
        outData.add(flashMode);
        return result;
    }

    @Override
    public int setStabilization(int mode) {
        // camera1 可以设置，并且设置完状态也会改变，但是防抖没有效果，需要 camera2
        if (mCamera != null && mParameter != null && isVideoStabilizationSupported()) {
            mParameter.setVideoStabilization(mode == VcProperties.Constant.TRUE);
            mCamera.setParameters(mParameter);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int getStabilization(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int stabilization = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            stabilization = mParameter.getVideoStabilization() ? VcProperties.Constant.TRUE :
                    VcProperties.Constant.FALSE;
        }
        outData.add(stabilization);
        return result;
    }

    @Override
    public int getSupportedPreviewFpsRange(List<Integer> outSupprotedRangeList) {
        if (mParameter != null) {
            List<int[]> supprotedRangeList = mParameter.getSupportedPreviewFpsRange();
            int length = supprotedRangeList.size();
            for (int i = 0; i < length; i++) {
                int[] tempRange = supprotedRangeList.get(i);
                outSupprotedRangeList.add(tempRange[Camera.Parameters.PREVIEW_FPS_MIN_INDEX]);
                outSupprotedRangeList.add(tempRange[Camera.Parameters.PREVIEW_FPS_MAX_INDEX]);
            }
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int getPreviewFpsRange(List<Integer> range) {
        if (mParameter != null) {
            int[] previewFpsRange = new int[2];
            mParameter.getPreviewFpsRange(previewFpsRange);
            range.add(previewFpsRange[0]);
            range.add(previewFpsRange[1]);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int setPreviewFpsRange(int[] range) {
        if (mCamera != null && mParameter != null) {
            mParameter.setPreviewFpsRange(range[Camera.Parameters.PREVIEW_FPS_MIN_INDEX],
                    range[Camera.Parameters.PREVIEW_FPS_MAX_INDEX]);
            Log.e(TAG, "set final  min fps = " + range[Camera.Parameters.PREVIEW_FPS_MIN_INDEX]
                    + "max fps = " + range[Camera.Parameters.PREVIEW_FPS_MAX_INDEX]);
            mCamera.setParameters(mParameter);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int isAntibandingsSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int antiBandingSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            antiBandingSupported = mParameter.getSupportedAntibanding() != null ? VcProperties.Constant.TRUE :
                    VcProperties.Constant.FALSE;
        }
        outData.add(antiBandingSupported);
        return result;
    }

    @Override
    public int isAntibandingAutoSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int antiBandingAutoSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            if (mParameter.getSupportedAntibanding() != null) {
                antiBandingAutoSupported = mParameter.getSupportedAntibanding().contains(Camera.Parameters.ANTIBANDING_AUTO)
                        ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
            }
        }
        outData.add(antiBandingAutoSupported);
        return result;
    }

    @Override
    public int setAntibandingMode(int mode) {
        if (mCamera != null && mParameter != null) {
            mParameter.setAntibanding(VcProperties.getAntibandingModeName(mode));
            mCamera.setParameters(mParameter);
            return VcProperties.Constant.TRUE;
        }
        return VcProperties.Constant.UNAVAILABLE;
    }

    @Override
    public int getAntiBandingMode(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int antiBandingMode = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            antiBandingMode = VcProperties.getAntibandingMode(mParameter.getAntibanding());
        }
        outData.add(antiBandingMode);
        return result;
    }

    @Override
    public int isFaceDetectionSupported(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int faceDetectionSupported = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            faceDetectionSupported = mParameter.getMaxNumDetectedFaces() > 0
                    ? VcProperties.Constant.TRUE : VcProperties.Constant.FALSE;
        }
        outData.add(faceDetectionSupported);
        return result;
    }

    @Override
    public int getMaxNumDetectedFaces(List<Integer> outData) {
        int result = VcProperties.Constant.UNAVAILABLE;
        int maxNumDetectedFaces = VcProperties.Constant.UNAVAILABLE;
        if (mParameter != null) {
            result = VcProperties.Constant.TRUE;
            maxNumDetectedFaces = mParameter.getMaxNumDetectedFaces();
        }
        outData.add(maxNumDetectedFaces);
        return result;
    }
}
