package sg.bigo.libvideo.cam.abs;

import androidx.annotation.Keep;

import java.util.ArrayList;
import java.util.List;
import sg.bigo.libvideo.cam.runtime.VcProvenance;
import sg.bigo.libvideo.cam.runtime.VcHandler;
import sg.bigo.libvideo.cam.runtime.VcRuntime;

@Keep
public class VcDevice {
    private final String TAG = "VcDevice";
    private long mNativeListener = 0;
    private long mFrameIndex = 0;
    private VcApi mApi;
    private VcCamera mCameraImpl;
    private VcCharacteristics mCharacteristics;
    private VcListener mJavaListener;

    @Keep
    public VcDevice(int api, String cameraId) {
        Log.e(TAG, "VcDevice construct,api:" + api + ";cameraId:" + cameraId);
        mApi = VcApiFactory.factoryApi(api);
        mCameraImpl = mApi.createCamera();
        mCameraImpl.setListener(mCameraListener);
        mCharacteristics = mApi.createCharacteristics(cameraId);
    }

    @Keep
    public void setListener(long nativeListener) {
        Log.e(TAG, "setListener:" + nativeListener);
        mNativeListener = nativeListener;
    }

    @Keep
    public int open(int width, int height, int bufferType) {
        Log.e(TAG, "open width:" + width + ";height:" + height);
        mCameraImpl.prepare(mCharacteristics, width, height, bufferType);
        VcHandler runtime = VcRuntime.instance().runtime();
        runtime.open(mCameraImpl, mCharacterListener);
        return VcProperties.Constant.TRUE;
    }

    @Keep
    public int reOpen() {
        Log.e(TAG, "reOpen");
        VcRuntime.instance().runtime().open(mCameraImpl, mCharacterListener);
        return VcProperties.Constant.TRUE;
    }

    @Keep
    public int close() {
        VcRuntime.instance().runtime().close();
        Log.e(TAG, "close");
        return VcProperties.Constant.TRUE;
    }

    @Keep
    public int getCameraApi() {
        return mApi.getCameraApi();
    }

    @Keep
    public Object getCameraIndex() {
        String result = mCharacteristics.getCameraIndex();
        Log.e(TAG, "getCameraIndex:" + result);
        return result;
    }

    @Keep
    public int isFacingFront() {
        List<Integer> result = new ArrayList<>();
        int res = mCharacteristics.isFacingFront(result);
        if (res < 0 || result.size() <= 0) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        Log.e(TAG, "isFacingFront:" + result.get(0) + ";res:" + res);
        return result.get(0);
    }

    @Keep
    public int getOrientation() {
        List<Integer> result = new ArrayList<>();
        int res = mCharacteristics.getOrientation(result);
        if (res < 0 || result.size() <= 0) {
            return VcProperties.Constant.UNAVAILABLE;
        }
        Log.e(TAG, "getOrientation:" + result.get(0) + ";res:" + res);
        return result.get(0);
    }

    @Keep
    public int queryParam(int requestCode, int key) {
        VcRuntime.instance().runtime().queryParam(requestCode, key);
        return VcProperties.Constant.TRUE;
    }

    @Keep
    public int applyParam(int requestCode, int key, int[] paramsInt) {
        VcRuntime.instance().runtime().applyParam(requestCode, key, paramsInt);
        return VcProperties.Constant.TRUE;
    }

    VcCharacterListener mCharacterListener = new VcCharacterListener() {
        @Override
        public void onApplyResult(int requestCode, int key, int result) {
            if (mNativeListener != 0) {
                VcDevice.this.onApplyResult(mNativeListener, requestCode, key, result);
            }
        }

        @Override
        public void onQueryResult(int requestCode, int key, int[] result) {
            if (mNativeListener != 0) {
                VcDevice.this.onQueryResult(mNativeListener, requestCode, key, result);
            }
        }
    };

    VcListener mCameraListener = new VcListener() {
        @Override
        public void onCameraStatusChanged(String cameraIndex, int statusType, int[] intParams, Object ext) {
            if (intParams != null && intParams.length > 1) {
                int status = intParams[VcDeviceParam.CAMERA_STATUS];
                if (status == VcProperties.CameraStatus.kStatus_Closed) {
                    mFrameIndex = 0;
                }
            }
            if (mNativeListener != 0) {
                VcDevice.this.OnCameraStatusChanged(mNativeListener, cameraIndex, statusType, intParams, ext);
            }
            if (mJavaListener != null) {
                mJavaListener.onCameraStatusChanged(cameraIndex, statusType, intParams, ext);
            }
        }

        @Override
        public void onFrameAvailable(byte[] frameBuffer) {
            if (mNativeListener != 0) {
                ++mFrameIndex;
                VcRuntime.instance().onFrameMileStone(mFrameIndex, VcProvenance.CamCTX, "start");
                VcDevice.this.OnFrameAvailable(mNativeListener, mFrameIndex, frameBuffer);
            }
            if (mJavaListener != null) {
                mJavaListener.onFrameAvailable(frameBuffer);
            }
        }

        @Override
        public int[] onChoosePreviewSize(int preferWidth, int preferHeight, int[] supportedPreviewSize) {
            return VcDevice.this.onChoosePreviewSize(mNativeListener, preferWidth, preferHeight, supportedPreviewSize);
        }
    };

    @Keep
    private native void OnFrameAvailable(long addr, long frameIndex, byte[] frameBuffer);

    @Keep
    private native void OnCameraStatusChanged(long addr, String cameraIndex, int statusType, int[] intParams, Object ext);

    @Keep
    private native void onQueryResult(long addr, int requestCode, int key, int[] resultInt);

    @Keep
    private native void onApplyResult(long addr, int requestCode, int key, int resultInt);

    @Keep
    private native int[] onChoosePreviewSize(long addr, int preferWidth, int preferHeight, int[] supportedPreviewSize);
}
