package sg.bigo.libvideo.cam.abs;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

public enum VcCameraKey {

    Unknown(0, null),
    FrameRate(1, VcFrameRate.class),
    Stabilization(2, VcStabilization.class),
    Zoom(3, VcZoom.class),
    Flash(4, VcFlash.class),
    FocusMode(5, VcFocusMode.class),
    FocusTouch(6, VcFocusTouch.class),
    ExposureMode(7, VcExposureMode.class),
    ExposureTouch(8, VcExposureTouch.class),
    AntiBanding(10, VcAntiBanding.class),
    Device(11, VcDevice.class),
    SupportedPreviewSize(12, VcSupportedPreviewSizes.class),
    ExposureFocusTouch(13, VcExposureFocusTouch.class),
    SupportedPreviewFpsRanges(14, VcSupportedFpsRanges.class);

    private int mKey = 0;
    private Class<? extends IParam> mParamClass = null;

    VcCameraKey(int key, Class<? extends IParam> clazz) {
        this.mKey = key;
        this.mParamClass = clazz;
    }

    public int key() {
        return mKey;
    }

    public Class<? extends IParam> clazz() {
        return mParamClass;
    }

    public static IParam newParamInstance(int keyValue) {
        IParam param = null;
        VcCameraKey[] cameraKeys = VcCameraKey.values();
        for (VcCameraKey key : cameraKeys) {
            if (key.mKey == keyValue) {
                Class<? extends VcCameraKey.IParam> clazz = key.clazz();
                if (clazz == null) {
                    return null;
                }
                try {
                    Constructor<? extends IParam> constructor = clazz.getDeclaredConstructor();
                    param = constructor.newInstance();
                    return param;
                } catch (NoSuchMethodException e) {
                    e.printStackTrace();
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                } catch (InstantiationException e) {
                    e.printStackTrace();
                } catch (InvocationTargetException e) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    public interface IParam {
        int applyParam(VcCharacteristics characteristics, int[] paramsInt);

        int queryParam(VcCharacteristics characteristics, List<Integer> result);
    }

    static class VcFrameRate implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            int minFps = 0;
            int maxFps = 0;
            if (paramsInt.length != 2) {
                List<Integer> supportedFpsRanges = new ArrayList<>();
                characteristics.getSupportedPreviewFpsRange(supportedFpsRanges);
                if(supportedFpsRanges.size() <= 0){
                    return VcProperties.Constant.FALSE;
                }
                if(paramsInt.length != 1){
                    minFps = supportedFpsRanges.get(0);
                    maxFps = supportedFpsRanges.get(1);
                } else {
                    for (int i = 0; i < supportedFpsRanges.size(); i = i + 2) {
                        minFps = supportedFpsRanges.get(i);
                        maxFps = supportedFpsRanges.get(i + 1);
                        if (minFps <= paramsInt[0] && maxFps >= paramsInt[0]) {
                            break;
                        }
                    }
                }
            } else {
                minFps = paramsInt[0];
                maxFps = paramsInt[1];
            }
            return characteristics.setPreviewFpsRange(new int[]{minFps, maxFps});
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getPreviewFpsRange(result);
        }
    }

    static class VcStabilization implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.setStabilization(paramsInt[0]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getStabilization(result);
        }
    }

    static class VcZoom implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.setZoom(paramsInt[0]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getZoom(result);
        }
    }

    static class VcFlash implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.setFlashMode(paramsInt[0]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getFlashMode(result);
        }
    }

    static class VcFlashMode implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.setFlashMode(paramsInt[0]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getFlashMode(result);
        }
    }

    static class VcFocusMode implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.setFocusMode(paramsInt[0]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getFocusMode(result);
        }
    }

    static class VcFocusTouch implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return VcProperties.Constant.UNAVAILABLE;
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            ArrayList<VcRectAngle> vcRectAngles = new ArrayList<>();
            int res = characteristics.getFocusAreas(vcRectAngles);
            int size = vcRectAngles.size();
            if (size > 0) {
                for (VcRectAngle rectAngle : vcRectAngles) {
                    rectAngle.packData(result);
                }
            }
            return res;
        }
    }

    static class VcExposureMode implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return VcProperties.Constant.UNAVAILABLE;
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return VcProperties.Constant.UNAVAILABLE;
        }
    }

    static class VcExposureTouch implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return VcProperties.Constant.UNAVAILABLE;
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            ArrayList<VcRectAngle> vcRectAngles = new ArrayList<>();
            int res = characteristics.getMeteringAreas(vcRectAngles);
            int size = vcRectAngles.size();
            if (size > 0) {
                for (VcRectAngle rectAngle : vcRectAngles) {
                    rectAngle.packData(result);
                }
            }
            return res;
        }
    }

    static class VcAntiBanding implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.setAntibandingMode(paramsInt[0]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getAntiBandingMode(result);
        }
    }

    static class VcExposureFocusTouch implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return characteristics.requestExposureFocus(paramsInt[0], paramsInt[1], paramsInt[2], paramsInt[3]);
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return VcProperties.Constant.UNAVAILABLE;
        }
    }

    static class VcDevice implements IParam {

        public VcDevice() {
        }

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return VcProperties.Constant.UNAVAILABLE;
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            result.add(characteristics.getCameraStatus());
            characteristics.getPreviewSize(result);
            characteristics.getOrientation(result);
            characteristics.isFacingFront(result);
            result.add(characteristics.getAPILevel());
            characteristics.getMaxZoom(result);
            characteristics.isFlashSupported(result);
            characteristics.isFlashModeTorchSupported(result);
            characteristics.isZoomSupported(result);
            characteristics.isMeteringSupported(result);
            characteristics.isFocusSupported(result);
            characteristics.isAutoFocusSupported(result);
            characteristics.isVideoContinuousFocusSupported(result);
            characteristics.isFaceDetectionSupported(result);
            characteristics.getMaxNumDetectedFaces(result);
            characteristics.isAntibandingsSupported(result);
            characteristics.isAntibandingAutoSupported(result);
            characteristics.isVideoStabilizationSupported(result);
            return VcProperties.Constant.TRUE;
        }
    }

    static class VcSupportedPreviewSizes implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return VcProperties.Constant.UNAVAILABLE;
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getSupportedPreviewSizes(result);
        }
    }

    static class VcSupportedFpsRanges implements IParam {

        @Override
        public int applyParam(VcCharacteristics characteristics, int[] paramsInt) {
            return VcProperties.Constant.UNAVAILABLE;
        }

        @Override
        public int queryParam(VcCharacteristics characteristics, List<Integer> result) {
            return characteristics.getSupportedPreviewFpsRange(result);
        }
    }
}
