local venuscore = require "libvenuscore"
local BlueNode = require "bluecore.bluenode"
local Types = require "venuscore.rtti.types"
local BlueSelf = require "bluecore.blueself"
local EventDispatcher = require "bluecore.eventdispatcher"
local BD = require "bluecore.bluedefined"
local AR = require "augmentedreality"
local BU = require "bluecore.blueutility"
local ARPlaneExist = BlueNode:extend();

local BREAK = 0
local TRIGGER = 1
local RELEASE = 2

function ARPlaneExist:new(...)
  ARPlaneExist.super.new(self, ...)
  self:_OnReset();
end

-- for BD.EventType.RegisterCallback's BlueNode, OnUpdate's argument is event.params
function ARPlaneExist:_OnUpdate(planeType)
  self.type = planeType
  self.nextExecIdx = BREAK
  local detected = false

  if self.type >= AR.ARPlaneComponent.arHorizontalObject then
    detected = true
  end

  if detected then 
    local edge = false
    if self.lastDetect == false then
      self.lastDetect = true
      edge = true
    end
    if self.isEdge and edge then
      self.nextExecIdx = TRIGGER
    elseif not self.isEdge then
      self.nextExecIdx = TRIGGER
    end
  else
    if self.lastDetect then
      self.lastDetect = false
      self.nextExecIdx = RELEASE
    end
  end

  return detected;
end

function ARPlaneExist:_OnReset()
  self.type = AR.ARPlaneComponent.arUnknownType
  self.lastDetect = false
  self.nextExecIdx = BREAK
end

function ARPlaneExist:_OnNextBranch()
  return self.nextExecIdx
end

function ARPlaneExist:RegisterTriggerFunc(func, bluePrintSelf)
  self:_GetInput()

  EventDispatcher:RegisterCallback(
          self.inputArgs[1]:GetContentPath(),
          BD.Event.VirtualEvent.HasPlane,
          func,
          bluePrintSelf)

  self.isEdge = self.inputArgs[2] or false
end

function ARPlaneExist:_OnCompile()
  self.compileResult = nil
  if ARPlaneExist.super._OnCompile(self) then
    -- Step.1 确保Component引脚有连接
    if #self.inputs[1].links ~= 0 then
      local bluePinLink = self.inputs[1].links[1] -- 输入引脚只允许一个连接
      local otherNode = self.graph:GetNode(bluePinLink.nodeUid);

      -- Step.2 确保上一个节点是 引用组件类型 的节点
      if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp == nil then
        self.compileResult = BD.EHINT.NOT_LINK_TO_COMP
        return false
      end
      if venuscore.isNil(otherNode.comp) then
        self.compileResult = BD.EHINT.COMP_LOST
        return false
      end

      -- Step.3 RecognitionComponent没有激活当前的检测类型
      local hasType = self:_HasType(otherNode.comp);
      if not hasType then
        self.compileResult = BD.EHINT.NOT_ENABLE;
        return false
      end
    else
      self.compileResult = BD.EHINT.PIN_1_EMPTY;
      return false
    end
    return true
  end
  return false
end

function ARPlaneExist:_HasType(comp)
  return comp.PlaneType > AR.ARPlaneComponent.arUnknownType
end

function ARPlaneExist:_OnDeserialize()
  ARPlaneExist.super._OnDeserialize(self);
end

---- 编辑器模式
if _KRATOSEDITOR then

function ARPlaneExist:_OnLink(selfPin, otherNode, otherPin)
  if BU:IsInputPin(selfPin) and selfPin.argId == 1 then
    if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp then
      if venuscore.isNil(otherNode.comp) then
        ERROR(BD.EHINT.COMP_LOST)
      else
        local hasType = self:_HasType(otherNode.comp);
        if not hasType then
          ERROR(BD.EHINT.NOT_ENABLE)
        end
      end
    else
      ERROR(BD.EHINT.NOT_LINK_TO_COMP)
    end
  end
end

function ARPlaneExist:_OnUnLink(unlinkPin)
  if unlinkPin.argId == 1 then
     ERROR(BD.EHINT.PIN_1_EMPTY)
  end
end

function ARPlaneExist:_OnCreateWithEditor()
  ARPlaneExist.super._OnCreateWithEditor(self);
end

function ARPlaneExist:_OnUpdateByEditor()
  ARPlaneExist.super._OnUpdateByEditor(self);
  self:_OnCompile();
end

end

ARPlaneExist:RegisterInput(1, AR.ARPlaneComponent:RTTI(), "component", "监听组件", BlueSelf());
ARPlaneExist:RegisterInput(2, Types.BoolType, "isEdge", "从无到有触发一次(OnTrigger)", false);

ARPlaneExist:RegisterOutput(1, Types.BoolType, "hasPlane", "有任意平面?");

ARPlaneExist:RegisterExecOutput(TRIGGER, "OnTrigger", "execute");
ARPlaneExist:RegisterExecOutput(RELEASE, "OnRelease", "execute");

ARPlaneExist:SetFunctionName("Has any plane?");
ARPlaneExist:SetFunctionType(BD.EVENT_FUNCTION_NODE);
ARPlaneExist:SetEventType(BD.EventType.RegisterCallback);

return ARPlaneExist;