local venuscore = require "venuscore"
local apollocore = require "apollocore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local FaceEditor = venuscore.VenusBehavior:extend("FaceEditor");
local liquifyconfig = require "behavior3.liquify_config"
local liquifyfunction = require "behavior3.liquify_function"
local cv = require "computervisionfunction"

function FaceEditor:new()
  LOG("Face Editor new is executed!");
  self.Texture = "";
  FaceEditor.Subject = "cat";
end

function FaceEditor:_OnStart()
  LOG("face editor On start is executed!");
end

function FaceEditor:_OnAwake()
  LOG("face editor on awake!");
  self.rendercomponent = nil;
  self.inited = false;
end

function FaceEditor:Init()
  LOG("face editor script initialized!");
  if _KRATOSEDITOR then
    local faceliquifyManager = require "window.editor.widget.face_editor.faceliquify_manager"
    faceliquifyManager:InitState();
  end
  local node = self.Node;
  self.rendercomponent = node:GetComponent(apollocore.Node.CT_RENDER);
  self.cameracomponent = node:GetComponent(apollocore.Node.CT_CAMERA);
  local scriptcomponent = node:GetComponent(apollocore.Node.CT_SCRIPT);
  if (self.rendercomponent~=nil) then
    local resolution = apolloengine.Framework:GetResolution();
    local aspect_ratio = resolution:y() / resolution:x();
    local gridX = 60;
    local gridY = math.floor(gridX * aspect_ratio);
    local vertexstream, indexstream = self:TesslateGivenPart(gridX, gridY, false, true,{-0.25,0.25}, {-0.25,0.25});
    self.rendercomponent:ChangeIndexBuffer(indexstream);
    self.rendercomponent:ChangeVertexBuffer(vertexstream);
    local matpath = "comm:documents/material/liquifyFaceEditor.mat";
    local meta = apolloengine.MaterialMetadata(matpath);
    local materialEntity = apolloengine.MaterialEntity();
    materialEntity:PushMetadata(meta);
    materialEntity:CreateResource();
    materialEntity:ChangeShaderPath("comm:documents/editor_shaders/2d/faceliquify2.material");
    self.rendercomponent:ChangeMaterial(materialEntity, 1);

    aspect_ratio = mathfunction.vector1(aspect_ratio);
    local zoomscale = mathfunction.Matrix33(
      4, 0, 0,
      0, 4, 0,
      0, 0, 0
    );

    self.rendercomponent:SetParameter("UNIFORM_ZOOMSCALE",zoomscale);
    self.rendercomponent:SetParameter("UNIFORM_ASPECTRATIO", aspect_ratio);
    -- local entity
    if (self.Texture ~= "") then
      local ext = self.Texture:match("^.+(%..+)$");
      if (ext == ".fbo") then
        local fbo = apolloengine.RenderTargetEntity();
        local meta = apolloengine.RenderTargetFileMetadata(venuscore.IFileSystem:PathAssembly(self.Texture));
        fbo:PushMetadata(meta);
        fbo:CreateResource();
        self.rendercomponent:SetParameter("TEXTURE_DIFFUSE",fbo);
      else
        local entity= apolloengine.TextureEntity();
        entity:PushMetadata(apolloengine.TextureReferenceMetadata(self.Texture));
        entity:CreateResource();
        self.rendercomponent:SetParameter("TEXTURE_DIFFUSE",entity);
      end
    else
      self.Texture = "DEVICE_CAPTURE";
      local entity= apolloengine.TextureEntity();
      entity:PushMetadata(apolloengine.TextureReferenceMetadata(self.Texture));
      entity:CreateResource();
      self.rendercomponent:SetParameter("TEXTURE_DIFFUSE",entity);
    end
    -- local entity= apolloengine.TextureEntity();
    -- entity:PushMetadata(apolloengine.TextureReferenceMetadata("DEVICE_CAPTURE"));
    -- entity:CreateResource();
  end
end

function FaceEditor:_OnUpdate(node,def)
  LOG("on updating")
  if not self.inited then
    self:Init();
    self.inited = true;
  else
    local resolution = self.cameracomponent.CameraResolution;
    local half_width = resolution:x() / 2;
    local half_height = resolution:y() / 2;
    local aspect_ratio = half_height / half_width;
    local ms_aspectRatio = mathfunction.vector1(aspect_ratio);
    local allRecogResult = {};

    local recognitioncomponent = self.Node:GetComponent(apollocore.Node.CT_CV_RECOGNITION);
    local catfacecomponent = self.Node:GetComponent(apollocore.Node.CT_CV_CATFACE);
    if (recognitioncomponent~=nil) then
      local recogResult = self:GetHumanRecogResult(recognitioncomponent);
      allRecogResult["human"] = recogResult;
      LOG("HAVE HUMAN RECOG COM");
    end
    if (catfacecomponent~=nil) then
      local catfaceResult = self:GetCatRecogResult(catfacecomponent);
      allRecogResult["cat"] = catfaceResult;
      LOG("HAVE CAT RECOG COM");
    end

    self.rendercomponent:SetParameter("UNIFORM_ASPECTRATIO",ms_aspectRatio);

    local childs = self.Node:GetChildrens();
    for i = 1, #childs do
      local liquify = childs[i]:GetComponent(apollocore.Node.CT_FACELIQUIFY);
      if liquify then
        local face_type = liquify.FaceType;
        if face_type == 0 then
          FaceEditor.Subject = "human";
        elseif face_type == 1 then
          FaceEditor.Subject = "cat";
        end
        break;
      end
    end

    local recogResult = allRecogResult[FaceEditor.Subject];
    local numFace;
    if (recogResult~=nil) then
      numFace = recogResult.num_face;
    else
      numFace = 0;
    end
    LOG("LENS NUM: ", #childs)
    for i=1, #childs do
      local liquify = childs[i]:GetComponent(apollocore.Node.CT_FACELIQUIFY);
      if liquify then
        if childs[i].Active then
          if liquify.CoorMode == "cartesian" then
            local ms_radiusRatio;
            if aspect_ratio>1 then
              ms_radiusRatio = mathfunction.vector2(liquify.CoefX/2,liquify.CoefY/2);
            else
              ms_radiusRatio = mathfunction.vector2(liquify.CoefX*aspect_ratio/2,liquify.CoefY*aspect_ratio/2)
            end
            
            local ms_faceDirX = mathfunction.vector2(1,0);
            local cartesianCoor = liquify.CartesianCoor;
            local ms_point = mathfunction.vector2(cartesianCoor[1],-cartesianCoor[2]);
            local ms_scaleRatioX = mathfunction.vector1(liquify.IntensityX);
            local ms_scaleRatioY = mathfunction.vector1(liquify.IntensityY);
            local ms_radius = mathfunction.vector2(1,0);
            self.rendercomponent:SetParameter("UNIFORM_RADIUS_RATIO"..liquify.Index,ms_radiusRatio);
            self.rendercomponent:SetParameter("UNIFORM_FACEDIR_X"..liquify.Index,ms_faceDirX);
            self.rendercomponent:SetParameter("UNIFORM_POINT"..liquify.Index,ms_point);
            self.rendercomponent:SetParameter("UNIFORM_SCALERATIOX"..liquify.Index,ms_scaleRatioX);
            self.rendercomponent:SetParameter("UNIFORM_SCALERATIOY"..liquify.Index,ms_scaleRatioY);
            self.rendercomponent:SetParameter("UNIFORM_RADIUS"..liquify.Index,ms_radius);
            if (liquify.IsMirror) then
              ms_point.mx = -ms_point.mx;
              self.rendercomponent:SetParameter("UNIFORM_RADIUS_RATIO"..(liquify.Index+liquifyconfig.max_lens),ms_radiusRatio);
              self.rendercomponent:SetParameter("UNIFORM_FACEDIR_X"..(liquify.Index+liquifyconfig.max_lens),ms_faceDirX);
              self.rendercomponent:SetParameter("UNIFORM_POINT"..(liquify.Index+liquifyconfig.max_lens),ms_point);
              self.rendercomponent:SetParameter("UNIFORM_SCALERATIOX"..(liquify.Index+liquifyconfig.max_lens),ms_scaleRatioX);
              self.rendercomponent:SetParameter("UNIFORM_SCALERATIOY"..(liquify.Index+liquifyconfig.max_lens),ms_scaleRatioY);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS"..(liquify.Index+liquifyconfig.max_lens),ms_radius);
            end
          else
            if (recogResult==nil) then
              local radius = mathfunction.vector2(0,0);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS"..liquify.Index,radius);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS"..(liquify.Index+liquifyconfig.max_lens),radius);
              return;
            end
            local faceIndex = liquify.FaceIndex;
            if (faceIndex<numFace) then
              local ms_faceDirX, ms_radiusRatio = self:GetFaceDirXAndRadiusRatio(FaceEditor.Subject,faceIndex, liquify.CoefX, liquify.CoefY, half_width, recogResult);
              local ms_scaleRatioX, ms_scaleRatioY = self:GetScaleRatios(FaceEditor.Subject, faceIndex, liquify.IntensityX, liquify.IntensityY, recogResult);
              local point;
              if liquify.CoorMode == "triangle" then
                point = self:GetBarycentricCoor(FaceEditor.Subject, faceIndex, recogResult.all_lmks, liquify.CurrentTriangle, liquify.BaryCoef);
              elseif liquify.CoorMode == "plane" then
                point = self:GetPlaneCoor(FaceEditor.Subject,faceIndex,recogResult.all_lmks,liquify.PlaneCoef)
              end
              local ms_point = mathfunction.vector2((point.mx-half_width)/half_width,(half_height-point.my)/half_height);
              local ms_radius = mathfunction.vector2(liquify.Radius,0);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS_RATIO"..liquify.Index,ms_radiusRatio);
              self.rendercomponent:SetParameter("UNIFORM_FACEDIR_X"..liquify.Index,ms_faceDirX);
              self.rendercomponent:SetParameter("UNIFORM_POINT"..liquify.Index,ms_point);
              self.rendercomponent:SetParameter("UNIFORM_SCALERATIOX"..liquify.Index,ms_scaleRatioX);
              self.rendercomponent:SetParameter("UNIFORM_SCALERATIOY"..liquify.Index,ms_scaleRatioY);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS"..liquify.Index,ms_radius);
              if (liquify.IsMirror) then
                if liquify.CoorMode == "triangle" then
                  local triangle = liquify.CurrentTriangle;
                  local point_mirror = liquifyconfig.subject[FaceEditor.Subject].point_mirror;
                  local mirrorTriangle = {point_mirror[triangle[3]],point_mirror[triangle[2]],point_mirror[triangle[1]]};
                  local mirrorBaryCoef = {liquify.BaryCoef[3],liquify.BaryCoef[2],liquify.BaryCoef[1]};
                  point = self:GetBarycentricCoor(FaceEditor.Subject, faceIndex, recogResult.all_lmks, mirrorTriangle, mirrorBaryCoef);
                elseif liquify.CoorMode == "plane" then
                  local mirrorPlaneCoef = {-liquify.PlaneCoef[1],liquify.PlaneCoef[2]};
                  point = self:GetPlaneCoor(FaceEditor.Subject,faceIndex,recogResult.all_lmks,mirrorPlaneCoef)
                end
                ms_point = mathfunction.vector2((point.mx-half_width)/half_width,(half_height-point.my)/half_height);
                self.rendercomponent:SetParameter("UNIFORM_RADIUS_RATIO"..(liquify.Index+liquifyconfig.max_lens),ms_radiusRatio);
                self.rendercomponent:SetParameter("UNIFORM_FACEDIR_X"..(liquify.Index+liquifyconfig.max_lens),ms_faceDirX);
                self.rendercomponent:SetParameter("UNIFORM_POINT"..(liquify.Index+liquifyconfig.max_lens),ms_point);
                self.rendercomponent:SetParameter("UNIFORM_SCALERATIOX"..(liquify.Index+liquifyconfig.max_lens),ms_scaleRatioX);
                self.rendercomponent:SetParameter("UNIFORM_SCALERATIOY"..(liquify.Index+liquifyconfig.max_lens),ms_scaleRatioY);
                self.rendercomponent:SetParameter("UNIFORM_RADIUS"..(liquify.Index+liquifyconfig.max_lens),ms_radius);
              else
                ms_radius.mx = 0;
                self.rendercomponent:SetParameter("UNIFORM_RADIUS"..(liquify.Index+liquifyconfig.max_lens),ms_radius);
              end
            else
              local ms_point = mathfunction.vector2(200,200);
              local ms_radius = mathfunction.vector2(0,0);
              self.rendercomponent:SetParameter("UNIFORM_POINT"..liquify.Index,ms_point);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS"..liquify.Index,ms_radius);
              self.rendercomponent:SetParameter("UNIFORM_POINT"..(liquify.Index+liquifyconfig.max_lens),ms_point);
              self.rendercomponent:SetParameter("UNIFORM_RADIUS"..(liquify.Index+liquifyconfig.max_lens),ms_radius);
            end
          end
        else
          local radius = mathfunction.vector2(0,0);
          self.rendercomponent:SetParameter("UNIFORM_RADIUS"..liquify.Index,radius);
          self.rendercomponent:SetParameter("UNIFORM_RADIUS"..(liquify.Index+liquifyconfig.max_lens),radius);
        end
      end
    end
  end
end

function FaceEditor:TesslateGivenPart(splitw, splith,flat,flip,xrange,yrange)
  local t1,t2,t3,t4;
  local p1 = {xrange[1],yrange[1]};
  local p2 = {xrange[2],yrange[1]};
  local p3 = {xrange[1],yrange[2]}; 
  local p4 = {xrange[2],yrange[2]};
  local vertexArray = {};
  local texcoordArray = {};
  
  local texrangex = {(xrange[1]+1)*0.5,(xrange[2]+1)*0.5};
  local texrangey = {(yrange[1]+1)*0.5,(xrange[2]+1)*0.5};
  if flip == true then
    t1 = {texrangex[1],texrangey[1]};
    t2 = {texrangex[2],texrangey[1]};
    t3 = {texrangex[1],texrangey[2]};
    t4 = {texrangex[2],texrangey[2]};
  else
    t1 = {texrangex[1],texrangey[2]};
    t2 = {texrangex[2],texrangey[2]};
    t3 = {texrangex[1],texrangey[1]};
    t4 = {texrangex[2],texrangey[1]};
  end

  --tesslate
  local vectorLeftEdge = { (p3[1] - p1[1]), (p3[2] - p1[2]) };
  local vectorRightEdge = { (p4[1] - p2[1]), (p4[2] - p2[2]) };
  for i = 0, splith do
    local pointLeft = { (p1[1] + (i / splith) * vectorLeftEdge[1]), (p1[2] + (i / splith) * vectorLeftEdge[2]) };
    local pointRight = { (p2[1] + (i / splith) * vectorRightEdge[1]), (p2[2] + (i / splith) * vectorRightEdge[2]) };
    for j = 0, splitw do
      local vectorX = { (pointRight[1] - pointLeft[1]), (pointRight[2] - pointLeft[2]) };
      local point = { (pointLeft[1] + (j / splitw) * vectorX[1]), (pointLeft[2] + (j / splitw) * vectorX[2]) };
      local texcoord = { j / splitw, t1[2] + (t3[2] - t1[2]) * i / splith }; --(t3[2] - t1[2]) + i / splith };
      --分割信息
      table.insert(vertexArray,point);
      table.insert(texcoordArray,texcoord);
    end
  end
  
  
  local vertexstream = apolloengine.VertexStream();
  vertexstream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
    apolloengine.VertexBufferEntity.DT_FLOAT,
    apolloengine.VertexBufferEntity.DT_HALF_FLOAT,4);
    vertexstream:SetVertexType(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
        apolloengine.VertexBufferEntity.DT_FLOAT,
        apolloengine.VertexBufferEntity.DT_HALF_FLOAT,
        2);

        vertexstream:ReserveBuffer((splitw + 1) * (splith + 1));

  local indexstream = apolloengine.IndicesStream();
  indexstream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
  indexstream:ReserveBuffer(splitw * splith * 6);
  
  --vertex data
  local newpoint =  mathfunction.vector4(0, 0, 0, 1);
  local index_vertex = vertexstream:GetAttributeIndex(apolloengine.ShaderEntity.ATTRIBUTE_POSITION);
  for i = 1, #vertexArray do
    local point = vertexArray[i];
    if flat == true then
      newpoint:Set(point[1],0,point[2],1);
    else
      newpoint:Set(point[1],point[2],0,1);
    end
    vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_POSITION,newpoint);
  end

  --texcoord data
  local newtexcoord = mathfunction.vector2(0,0);
  local index_texcoord = vertexstream:GetAttributeIndex(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0);
  for i = 1, #texcoordArray do
    local texcoord = texcoordArray[i];
    newtexcoord:Set(texcoord[1],texcoord[2]);
    vertexstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,newtexcoord);
  end

  --index data
  for i = 1, splith do
    for j = 1, splitw do
      indexstream:PushIndicesData((i - 1) * (splitw + 1) + (j - 1));  --0
      indexstream:PushIndicesData(i * (splitw + 1) + (j - 1));  --2
      indexstream:PushIndicesData(i * (splitw + 1) + j); --3
      indexstream:PushIndicesData((i - 1) * (splitw + 1) + (j - 1)); --0
      indexstream:PushIndicesData(i * (splitw + 1) + j); --3
      indexstream:PushIndicesData((i - 1) * (splitw + 1) + j); --1
    end
  end

  return vertexstream, indexstream
  
end

function FaceEditor:GetHumanRecogResult(recognitioncomponent)
  local recogResult = {};
  local all_lmks = {};
  local all_pry = {};
  local results = recognitioncomponent:GetResult();
  if results==nil or results[cv.RecognitionComponent.cvFace]==nil then
      recogResult.num_face = 0;
      recogResult.all_lmks = all_lmks;
      recogResult.all_pry = all_pry;
      return recogResult;
  end
  local numFace = #results[cv.RecognitionComponent.cvFace];
  if numFace>liquifyconfig.max_face_id then
      numFace = liquifyconfig.max_face_id;
  end
  for i = 1, numFace do
      local face_lmks = results[cv.RecognitionComponent.cvFace][i];
      local num_lmks = #face_lmks/2;
      for id = 1, 106 do
          local x = face_lmks[2*id-1];
          local y = face_lmks[2*id];
          table.insert(all_lmks, mathfunction.vector2(x, y));
      end
      table.insert(all_pry, face_lmks[109 * 2 - 1]);
      table.insert(all_pry, face_lmks[111 * 2 - 1]);
      table.insert(all_pry, face_lmks[110 * 2 - 1]);
  end
  recogResult.num_face = numFace;
  recogResult.all_lmks = all_lmks;
  recogResult.all_pry = all_pry;
  return recogResult;
end

function FaceEditor:GetCatRecogResult(catfacecomponent)
  -- TODO
  local recogResult = {};
  local all_lmks = {};
  local all_pry = {};
  local results = catfacecomponent:GetResult();
  if results==nil or results[cv.CatFaceComponent.cvCatface]==nil then
    recogResult.num_face = 0;
    recogResult.all_lmks = all_lmks;
    recogResult.all_pry = all_pry;
    return recogResult;
  end
  local numFace = #results[cv.CatFaceComponent.cvCatface];
  if numFace>liquifyconfig.max_face_id then
      numFace = liquifyconfig.max_face_id;
  end
  for i = 1, numFace do
      local face_lmks = results[cv.CatFaceComponent.cvCatface][i];
      local num_lmks = #face_lmks/2;
      for id = 1, 32 do
          local x = face_lmks[2*id-1];
          local y = face_lmks[2*id];
          table.insert(all_lmks, mathfunction.vector2(x, y));
      end
  end
  recogResult.num_face = numFace;
  recogResult.all_lmks = all_lmks;
  recogResult.all_pry = all_pry;
  return recogResult;
end

function FaceEditor:GetFaceDirXAndRadiusRatio(subject, faceIndex, coefX, coefY, half_width, recogResult)
  return liquifyfunction.GetFaceDirXAndRadiusRatio[subject](faceIndex, coefX, coefY, half_width, recogResult);
end

function FaceEditor:GetScaleRatios(subject, faceIndex, intensityX, intensityY, recogResult)
  return liquifyfunction.GetScaleRatios[subject](faceIndex, intensityX, intensityY, recogResult);
end

function FaceEditor:GetBarycentricCoor(subject, faceIndex, all_lmks, triangle, ws)
  local offset = faceIndex * liquifyconfig.subject[subject].num_lmks;
  local point = all_lmks[offset+triangle[1]+1]*ws[1]+all_lmks[offset+triangle[2]+1]*ws[2]
                +all_lmks[offset+triangle[3]+1]*ws[3];
  return point;
end

function FaceEditor:GetPlaneCoor(subject, faceIndex, all_lmks, ws)
  return liquifyfunction.GetPlaneCoor[subject](faceIndex, all_lmks, ws);
end

function FaceEditor:GetTexture()
  return self.Texture;
end

function FaceEditor:SetTexture(value)
  self.Texture = value;
  if (self.rendercomponent ~= nil) then
    local ext = self.Texture:match("^.+(%..+)$");
    if (ext == ".fbo") then
      local fbo = apolloengine.RenderTargetEntity();
      local meta = apolloengine.RenderTargetFileMetadata(venuscore.IFileSystem:PathAssembly(self.Texture));
      fbo:PushMetadata(meta);
      fbo:CreateResource();
      self.rendercomponent:SetParameter("TEXTURE_DIFFUSE",fbo);
    else
      local entity= apolloengine.TextureEntity();
      entity:PushMetadata(apolloengine.TextureReferenceMetadata(self.Texture));
      entity:CreateResource();
      self.rendercomponent:SetParameter("TEXTURE_DIFFUSE",entity);
    end
  end
end

function FaceEditor:GetFaceType()
  return FaceEditor.Subject;
end

function FaceEditor:SetFaceType(value)
  if _KRATOSEDITOR then
    local faceliquifyManager = require "window.editor.widget.face_editor.faceliquify_manager"
    local defined = require "window.editor.system.defined"
    FaceEditor.Subject = value;
    faceliquifyManager.lens_type = value;
    faceliquifyManager.landmarks = liquifyconfig.subject[faceliquifyManager.lens_type].landmarks;
    faceliquifyManager.triangles = liquifyconfig.subject[faceliquifyManager.lens_type].triangles;
    faceliquifyManager.point_mirror = liquifyconfig.subject[faceliquifyManager.lens_type].point_mirror;
    faceliquifyManager.image_size = liquifyconfig.subject[faceliquifyManager.lens_type].size;
    faceliquifyManager.default_landmarks = faceliquifyManager:DefaultLandmarks();
    faceliquifyManager:ChangeSubject();

    if faceliquifyManager.rendercomponent ~= nil then
      local gridX = 50;
      local vs, is;
      local image_remap_param;
      if value == "human" then
        vs,is = faceliquifyManager:TesslateGivenPart(gridX, math.floor(gridX * 1.5), false, true,{-0.1,0.1}, {-0.12,0.18});
        image_remap_param = mathfunction.vector4(5.0, 3.333, 0.5, 0.4);
      elseif value == "cat" then
        vs, is = faceliquifyManager:TesslateGivenPart(gridX, math.floor(gridX * 1.28), false, true, {-0.128, 0.128}, {-0.133, 0.191});
        image_remap_param = mathfunction.vector4(3.906, 3.086, 0.5, 0.41);
      end


      faceliquifyManager.rendercomponent:ChangeIndexBuffer(is);
      faceliquifyManager.rendercomponent:ChangeVertexBuffer(vs);
      faceliquifyManager.rendercomponent:SetParameter("IMAGE_REMAP_PARAMS", image_remap_param);

      local tex = apolloengine.TextureEntity();
      local tex_pic;
      if value == "human" then
          tex_pic = defined.StadardFace;
      elseif value == "cat" then
          tex_pic = defined.StadardCat;
      end
      tex:PushMetadata(apolloengine.TextureFileMetadata(
              apolloengine.TextureEntity.TU_STATIC,
              apolloengine.TextureEntity.PF_AUTO,1, false,
              apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
              apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
              apolloengine.TextureEntity.TF_NEAREST,
              apolloengine.TextureEntity.TF_NEAREST,
              tex_pic));
      tex:SetJobType(venuscore.IJob.JT_SYNCHRONOUS);
      tex:CreateResource();
      
      local materialEntity = faceliquifyManager.rendercomponent.MaterialEntities[1];
      materialEntity:SetParameter("TEXTURE_DIFFUSE", tex);
    end
  end
end

FaceEditor:MemberRegister("Texture",
  venuscore.ScriptTypes.FilePathType(
      apolloengine.TextureEntity:RTTI(),
      FaceEditor.GetTexture,
      FaceEditor.SetTexture
));

FaceEditor:MemberRegister("Subject",
  venuscore.ScriptTypes.ComboType(
    {
      {
        key = "human",
        value = "human"
      },
      {
        key = "cat",
        value = "cat"
      }
    },
    FaceEditor.GetFaceType,
    FaceEditor.SetFaceType
  )
);

return FaceEditor;