local Object        = require "classic"
local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local mathfunction  = require "mathfunction"
local venuscore     = require "venuscore"
local likeapp       = require "likeapp"
local defined       = require "liveportrait.defined"
local fmdefined     = require "facemorph.defined"
local utility       = require "liveportrait.utility"

local WrinkleRenderer = Object:extend();

function WrinkleRenderer:new(wrinkle_coords, wrinkle_tex, forehead_mask, ldecree_mask, rdecree_mask)
  self.wrinkle_coords = wrinkle_coords; 
  self.wrinkle_texs = { wrinkle_tex, forehead_mask, ldecree_mask, rdecree_mask };
  
  self.wrinkle_nodes = {}
  self.vtx_streams = {}
  
  for i = 1, #self.wrinkle_texs do
    local vtx_stream, render_node = self:_GenerateRenderContext(self.wrinkle_texs[i]);
    table.insert(self.wrinkle_nodes, render_node);
    table.insert(self.vtx_streams, vtx_stream);
  end
  
  self.render_cams = {}
  self.warped_wrinkle_texs = {}
end

function WrinkleRenderer:Clear()
  self.warped_wrinkle_texs = {}
  self.wrinkle_texs = {}
  
  --目前node需要自己删除
  for i = 1, #self.wrinkle_nodes do
    local render_node = self.wrinkle_nodes[i]
    render_node:Destroy();
  end
  for i = 1, #self.render_cams do
    local cam = self.render_cams[i];
    cam:Destroy()
  end
end

function WrinkleRenderer:SetRenderSequence(seq)
  for i = 1, #self.wrinkle_nodes do
    local render_node = self.wrinkle_nodes[i];
    render_node:SetSequence(seq - i)
  end
  
  for i = 1, #self.render_cams do
    local cam = self.render_cams[i];
    cam:SetSequence(seq - i)
  end
  
  self.sequence = seq;
end

function WrinkleRenderer:WarpImage(imagesize, facelmks, fhlmks)
  self.render_cams = {}
  self.warped_wrinkle_texs = {}
  
  local normalized_lmks = self:_GenerateVertices(imagesize, facelmks, fhlmks);
  for i = 1, #self.wrinkle_nodes do
    local render_node = self.wrinkle_nodes[i]
    local vtx_stream = self.vtx_streams[i];
    
    local camera, warpped_tex = utility.CreateRenderTarget(imagesize, false, true);
    camera:SetClearColor(mathfunction.Color(0.0, 0.0, 0.0, 1.0));
    
    if self.sequence ~= nil then
      camera:SetSequence(self.sequence - i)
    end
    
    table.insert(self.render_cams    , camera);
    table.insert(self.warped_wrinkle_texs, warpped_tex);
    
    utility.UpdateVertexStream( render_node, 
                                vtx_stream, 
                                apolloengine.ShaderEntity.ATTRIBUTE_POSITION, 
                                normalized_lmks);
                        
    render_node:SetShow(true)
  end 
end

function WrinkleRenderer:GetWrinkleTex(index)
  
  if index < 0 or index > #self.wrinkle_texs then
    return nil
  end
  
  if self.warped_wrinkle_texs ~= nil and next(self.warped_wrinkle_texs) ~= nil then
    return self.warped_wrinkle_texs[index]
  else
    return self.wrinkle_texs[index]
  end
  
end

function WrinkleRenderer:_GenerateRenderContext(tex)
  local material       = "docs:liveportrait/material/facewarp.material";
  local render_mode    =  apolloengine.RenderComponent.RM_TRIANGLES;
  
  local vtx_stream     =  apolloengine.VertexStream();  
  --position
  vtx_stream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                             apolloengine.VertexBufferEntity.DT_FLOAT,
                             apolloengine.VertexBufferEntity.DT_FLOAT,
                             2);
  --tex coords
  vtx_stream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
                             apolloengine.VertexBufferEntity.DT_FLOAT,
                             apolloengine.VertexBufferEntity.DT_FLOAT,
                             2); 

  for i = 1, fmdefined.NUM_DELAUNAY_POINTS do 
    vtx_stream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_POSITION, mathfunction.vector2(0, 0));
    local coord = self.wrinkle_coords:Get(i);
    vtx_stream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, self.wrinkle_coords:Get(i));
  end

  local idx_stream = apolloengine.IndicesStream(); 
  idx_stream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
  
  local triangles = fmdefined.DELAUNAY_INDICE_WITH_MOUTH;
  idx_stream:ReserveBuffer(#triangles);

  for i = 1, #triangles do
    idx_stream:PushIndicesData(triangles[i]);
  end  
  
  local texSlot = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE_DIFFUSE");  
  
  local render_node       =  apollonode.RenderNode(); 
  render_node:CreateResource(material, render_mode, vtx_stream, idx_stream);    
  render_node:SetParameter(texSlot, tex);
  render_node:SetShow(false);  
  return vtx_stream, render_node;
end

function WrinkleRenderer:_GenerateVertices(size, facelmks, fhlmks)  
  
  local points = mathfunction.vector2array();
  for i = 1, #facelmks, 2 do
    points:PushBack(mathfunction.vector2(facelmks[i], facelmks[i+1]));
  end
  
  for i = 1, #fhlmks, 2 do
    points:PushBack(mathfunction.vector2(fhlmks[i], fhlmks[i+1]));
  end

  local col, row = 0, 0;
  local left, bot, width, height = 0, 0, size:x(), size:y();
  
  local NUM_DIVISION = 10;
  
  for cx = 1, NUM_DIVISION + 1 do
    col = math.min(left + width * (cx - 1) / NUM_DIVISION, left + width);
    for cy = 1, NUM_DIVISION + 1 do
      row =  math.min(bot + height * (cy - 1) / NUM_DIVISION, bot + height);
      if not (
        col ~= left and 
        col ~= (left + width) and
        row ~= bot and 
        row ~= (bot + height) ) then
        points:PushBack(mathfunction.vector2(col, row));
      end 
    end
  end
  
  self:_NormalizeLmk(size, points);

  return points;
end

function WrinkleRenderer:_NormalizeLmk(size, vertices)
  for i = 1, fmdefined.NUM_DELAUNAY_POINTS do    
    local point = (vertices:Get(i) / size) * 2 - 1;
    vertices:Set(i, point);
  end
end

return WrinkleRenderer;
