local venuscore = require "venuscore"
local apollocore = require "apollocore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local optimization = require "optimization"
local collisionhandler = require "avatar.retarget.collisionhandler"
local cv = require "computervisionfunction"
local util = require "behavior.avatar_behavior.util"
local modelpose = require "behavior.avatar_behavior.model_pose"
local skelScriptPath = "scrs:behavior/avatar_behavior/skeleton_behavior.lua"
local BundleSystem = require "venuscore.bundle.bundlesystem"

local SkeCubes = {}

table.insert( SkeCubes, "RARM" );
table.insert( SkeCubes, "RFOREARM" );
table.insert( SkeCubes, "LARM" );
table.insert( SkeCubes, "LFOREARM" );
table.insert( SkeCubes, "RLEG" );
table.insert( SkeCubes, "RFORELEG" );
table.insert( SkeCubes, "LLEG" );
table.insert( SkeCubes, "LFORELEG" );
table.insert( SkeCubes, "NECKANDHEAD" );

local AvatarTest = modelpose:extend("AvatarTest"); 

function AvatarTest:new()
    
    AvatarTest.super.new(self);
    self.boneoffset = {0,0.3,0,0.3,0,0,0,0,0.1,0,0.0};
    self.boneMargin = {{1.8,1.3,1.0},{1.8,1.3,1.0},{1.8,1.3,1.0},{1.8,1.3,1.0},{1.8,1.3,1.0},{1.8,1.3,1.0},{1.8,1.3,1.0},{1.8,1.3,1.0},{2.3,1.3,1.0}};

    self.RARM = nil;
    self.RFOREARM = nil;
    self.LARM = nil;
    self.LFOREARM = nil;
    self.RLEG = nil;
    self.RFORELEG = nil;
    self.LLEG = nil;
    self.LFORELEG = nil;
    self.NECKANDHEAD = nil;

    self.BODYUPPER = nil;
    self.BODYLOWER = nil;
    self.BODYAVG = nil;

    self.BodyUpperMargin = mathfunction.vector3(1.5,1.6,1.0);
    self.BodyLowerMargin = mathfunction.vector3(2.0,1.2,1.0);
    self.BoxScale = 10;


    self.camera=nil;

    self.trans = {};
    self.attachers = {};
    
    self.lateralErr = 0;
    self.headTrans = nil;
    self.cvOriginType = cv.RecognitionComponent.cvBodyPose3D;
    self.offset = mathfunction.vector3(0.0, -0.4, 1.0);
    
    self.usereproj = true;

end

function AvatarTest:InitSkeleton()
  --[==[]==]
  self.skChildIdx = {2,3,0,5,6,0,8,9,0,11,12,0,14,0}
  self.skTBoneLength = {
    0.26239383220672607, 0.26926156878471375, 0,
    0.25683808326721191, 0.26611471176147461, 0,
    0.38614886999130249, 0.42189562320709229, 0,
    0.38898751139640808, 0.42876723408699036, 0,
    0.10272922366857529, 0,
  } -- self -> child, from smpl model pose
  self.skBodyIdx = {1, 4, 7, 10, 13}
  self.skBodyModelPose = {
    mathfunction.vector3(-0.2022, 0.5604, 0.0311),
    mathfunction.vector3(0.1887, 0.5641, 0.0253),
    mathfunction.vector3(-0.0594, 0.0041, 0.0021),
    mathfunction.vector3(0.0594, -0.0041, -0.0021),
    mathfunction.vector3(0.0059, 0.6164, 0.0266),
  } -- body keypoints are treated as rigid body
  self.skBodyStartRotMatTranspose = mathfunction.Matrix33(
    0.9970,    0.0650,   -0.0340,
    0.0670,   -0.9970,    0.0490,
   -0.0310,   -0.0510,   -0.9980);
  self.skBodyStartRotMatTranspose:Transpose()
  self.skTBoneLengthCoeffEMA = 0.99
  self.skInited = true
end

function AvatarTest:GetBodyRotateMatrix(pos3d, score, cutoff)
  local rotationmat = mathfunction.Matrix33();
  if score==nil or (score~=nil and score[1]>cutoff and score[4]>cutoff and score[7]>cutoff and score[10]>cutoff) then
    tempppair = {{1,4},{7,10},{1,7},{4,10}}
    tempdir = {}
    for i=1,#tempppair do
      local fpos = mathfunction.vector3(pos3d[tempppair[i][1]][1],pos3d[tempppair[i][1]][2],pos3d[tempppair[i][1]][3]);
      local spos = mathfunction.vector3(pos3d[tempppair[i][2]][1],pos3d[tempppair[i][2]][2],pos3d[tempppair[i][2]][3]);
      local dir = (spos - fpos);
      dir:NormalizeSelf();
      table.insert(tempdir,dir);
    end
    tempdir[3] = (tempdir[3]+tempdir[4])/2
    tempdir[3]:NormalizeSelf();

    local currentBodyRotMat = util:ConstractRotationFromVector(tempdir[2],tempdir[3]);
    rotationmat =  self.skBodyStartRotMatTranspose * currentBodyRotMat;
  end
  return rotationmat;
end

function AvatarTest:UpdateSkeleton(modelPose, score, cutoff)
  --LOG('******** Update Skeleton ********')
  if self.skInited == nil or not self.skInited then
    self:InitSkeleton();
    return modelPose;
  end
  local newModelPose = {};
  local bodyRotateMat = self:GetBodyRotateMatrix(modelPose, score, cutoff);
  for i, idx in ipairs(self.skBodyIdx) do
    local curPose = self.skBodyModelPose[i] * bodyRotateMat;
    newModelPose[idx] = {curPose:x(), curPose:y(), curPose:z()};
  end
  for idx, idxChild in ipairs(self.skChildIdx) do
    if idxChild ~= 0 then
      local selfPose = mathfunction.vector3(unpack(modelPose[idx]));
      local childPose = mathfunction.vector3(unpack(modelPose[idxChild]));
      local orientation = childPose - selfPose;
      local bonelength = orientation:Length();
      orientation:NormalizeSelf();
      local boneOffset = orientation * self.skTBoneLength[idx];
      newModelPose[idxChild] = {
        newModelPose[idx][1] + boneOffset:x(),
        newModelPose[idx][2] + boneOffset:y(),
        newModelPose[idx][3] + boneOffset:z(),
      };
      if score[idx] > cutoff and score[idxChild] > cutoff then
        if idx == 13 then -- special care for neck-head
          bonelength = bonelength / 3;
        end
        self.skTBoneLength[idx] = self.skTBoneLengthCoeffEMA * self.skTBoneLength[idx] +
          (1-self.skTBoneLengthCoeffEMA) * bonelength;
      end
    end
  end
  return newModelPose;
end

function AvatarTest:_OnAwake()
    if self.recognition ~= nil then
        self.cvOriginType = self.recognition.Type;
    end
    
    if _KRATOSEDITOR then
      self._malePath = venuscore.IFileSystem:PathAssembly("proj:assets/sticker_resource/humanpose/male.prefab");
      self._deserializeMode = BundleSystem.DeserializeMode.Prefab;
      
      local malefile = io.open(self._malePath, "rb");
      local malestr = malefile:read("*a");
      local hostscene = self.Node:GetHostScene();
      local malenodeList = hostscene:Deserialize(malestr,self._deserializeMode);
      
      local maleroot = malenodeList[#malenodeList];
      maleroot:SetLayer(apollocore.LayerMask.MC_MASK_EDITOR_SCENE_LAYER);
      self.editormaleroot = maleroot;
    end
end

function AvatarTest:GetEditorTransform(index)
  if _KRATOSEDITOR then
    if index == 1 then 
      local trans = self.editormaleroot:GetNodeByName("m_avg_R_Shoulder"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==2 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_R_Elbow"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==3 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_R_Wrist"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==4 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_L_Shoulder"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==5 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_L_Elbow"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==6 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_L_Wrist"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==7 then 
      local trans = self.editormaleroot:GetNodeByName("m_avg_R_Hip"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==8 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_R_Knee"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==9 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_R_Ankle"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==10 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_L_Hip"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==11 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_L_Knee"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==12 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_L_Ankle"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    elseif index==13 then
      local trans = self.editormaleroot:GetNodeByName("m_avg_Pelvis"):GetComponent(apolloengine.Node.CT_TRANSFORM);
      return trans:GetWorldPosition();
    else
      return mathfunction.vector3();
    end
  end
end

function AvatarTest:GetCamera()
    return  self.camera;
end

function AvatarTest:SetCamera(camera)
    self.camera = camera;
end

function AvatarTest:GetRecognitionComponent()
    return  self.recognition;
end

function AvatarTest:SetRecognitionComponent(recognition)
    self.recognition = recognition;
end

function AvatarTest:GetHumanPose()

    local  pose2d_ret, score_ret, pose3d_ret = AvatarTest.super.GetHumanPose(self);
    if pose2d_ret == nil or
       score_ret == nil or
       pose3d_ret == nil then  
      return nil, nil, nil
    end
    
    pose3d_ret = self:UpdateSkeleton(pose3d_ret, score_ret, self.cutoffscore);
    
    if self.lastroot==nil then
        self.lastroot = {0,0,-4.3}
    end

    if self.lastroot[1]>20 or self.lastroot[2]>20 or self.lastroot[3]>20 or self.lastroot[1]<-20 or self.lastroot[2]<-20 or self.lastroot[3]<-20 then
        self.lastroot = {0,0,-4.3};
        self.RootEstimate:SetContinueWeight(0.0);
        self.RootEstimate:SetLastRootPose(mathfunction.vector3(self.lastroot[1],self.lastroot[2],self.lastroot[3]));
    end
    
    if self.first==false or self.first==nil  then
        self.RootEstimate:SetLastRootPose(mathfunction.vector3(self.lastroot[1],self.lastroot[2],self.lastroot[3]));
    end
    
    self.RootEstimate:SetRootPose(mathfunction.vector3(self.lastroot[1],self.lastroot[2],self.lastroot[3]));
    self.RootEstimate:SetPose3D(pose3d_ret);
    self.RootEstimate:SetPose2D(pose2d_ret);
    self.lastroot = self.RootEstimate:Minimize();
    --LOG("ROOT POSITION IS "..self.lastroot[1].." "..self.lastroot[2].." "..self.lastroot[3]);

    for i = 1, 14 do
        pose3d_ret[i][1] =  pose3d_ret[i][1]+self.lastroot[1];
        pose3d_ret[i][2] =  pose3d_ret[i][2]+self.lastroot[2];
        pose3d_ret[i][3] =  pose3d_ret[i][3]+self.lastroot[3];
    end

    if self.usereproj then
      if self.first==false or self.first==nil  then
          self.ReprojCorrect:SetLastPose3D(pose3d_ret);
      end
      self.first = true;
      self.ReprojCorrect:SetPose3D(pose3d_ret);
      self.ReprojCorrect:SetPose2D(pose2d_ret);
      
      pose3d_ret_tmp = self.ReprojCorrect:Minimize();
      pose3d_ret = pose3d_ret_tmp;
      
      self.ReprojCorrect:SetLastPose3D(pose3d_ret);
    end
    return pose2d_ret, score_ret, pose3d_ret
end

function AvatarTest:_OnUpdate(def)
    self.trans[1] = self.RARM ;
    self.trans[2] = self.RFOREARM ;
    self.trans[3] = self.LARM ;
    self.trans[4] = self.LFOREARM ;
    self.trans[5] = self.RLEG ;
    self.trans[6] = self.RFORELEG ;
    self.trans[7] = self.LLEG ;
    self.trans[8] = self.LFORELEG ;
    self.trans[9] = self.NECKANDHEAD ;

    local pos2d, scores, pos3d = self:GetHumanPose();
    
    if pos2d == nil or pos3d == nil then
       return;
    end
--[[
    if  self:IsLessScore() then
        self.recognition.Type = bit.bor(self.recognition.Type, cv.RecognitionComponent.cvFace);
    else
        self.recognition.Type = self.cvOriginType;
    end]]
    
    local transidx = 1;
    for i =1 , #self.parents do
        if self.parents[i] ~= 0 then
          
            local paridx = self.parents[i];
            if scores[i]<self.cutoffscore or scores[paridx]<self.cutoffscore then
              if self.trans[transidx] ~= nil then
                self.trans[transidx]:GetHostNode().Active =false;
                transidx= transidx+1;
              end
            else
              local mypos = mathfunction.vector3((pos3d[i][1]),(pos3d[i][2]),pos3d[i][3]);
              
              local parentpos = mathfunction.vector3((pos3d[paridx][1]),(pos3d[paridx][2]),pos3d[paridx][3]);
              local cubepos = (mypos-parentpos)*(0.5+ self.boneoffset[transidx])+parentpos;

              local dir = parentpos-mypos;
              local length = dir:Length()*(1+ self.boneoffset[transidx]*2);
              local otherscale = length/4;
              dir:NormalizeSelf();
              local rotation = mathfunction.Quaternion();
              rotation:AxisToAxis(mathfunction.vector3(0,0,1),dir);
              
              if self.trans[transidx] ~= nil then
                self.trans[transidx]:GetHostNode().Active =true;
                local margin = self.boneMargin[transidx];
                if margin~=nil then
                  otherscale = otherscale*margin[1];
                  length = length*margin[3]
                end
                self.trans[transidx]:SetLocalScale(mathfunction.vector3(otherscale*self.BoxScale,otherscale*self.BoxScale,length*self.BoxScale));
                self.trans[transidx]:SetLocalPosition(cubepos);
                self.trans[transidx]:SetLocalRotation(rotation);
                transidx= transidx+1
              end
            end
        end
    end

    if self.BODYUPPER~=nil and  self.BODYLOWER~=nil and   self.BODYAVG~=nil then
        if scores[1]<self.cutoffscore or scores[4]<self.cutoffscore then
          self.BODYUPPER:GetHostNode().Active =false;
        else
          self.BODYUPPER:GetHostNode().Active =true;
        end
        if scores[7]<self.cutoffscore or scores[10]<self.cutoffscore then
          self.BODYLOWER:GetHostNode().Active =false;
        else
          self.BODYLOWER:GetHostNode().Active =true;
        end
      
        local RSPos = mathfunction.vector3( pos3d[1][1],  pos3d[1][2],  pos3d[1][3]);
        local LSPos = mathfunction.vector3( pos3d[4][1],  pos3d[4][2],  pos3d[4][3]);
        local RHPos = mathfunction.vector3( pos3d[7][1],  pos3d[7][2],  pos3d[7][3]);
        local LHPos = mathfunction.vector3( pos3d[10][1], pos3d[10][2], pos3d[10][3]);
        
        local shoulderwidth = (RSPos - LSPos):Length();
        local hipwidth = (RHPos - LHPos):Length();

        local bodyvector = (RSPos+LSPos)/2 - (RHPos+LHPos)/2;
        local bodyheight = bodyvector:Length();
        local shouldervector = LSPos-RSPos;
        local hipvector = LHPos-RHPos;


        local avgvector = hipvector:Normalize()+shouldervector:Normalize();

        local hiprotate      =  util:ConstractRotationFromVector(hipvector:Normalize(),bodyvector:Normalize());
        local shoulderrotate =  util:ConstractRotationFromVector(shouldervector:Normalize(),bodyvector:Normalize());
        local avgrotate      =  mathfunction.Mathutility:Slerp(hiprotate:ToQuaternion(),shoulderrotate:ToQuaternion(),0.5);

        local hippos = (RHPos+LHPos)/2;
        local sholderpos = (RSPos+LSPos)/2;
        local modelhippos = self:Lerp(hippos,sholderpos,0.25);
        local modelshoulderpos = self:Lerp(sholderpos,hippos,0.25);
        local modelavgpos = self:Lerp(sholderpos,hippos,0.5);


        self.BODYUPPER:SetLocalScale(mathfunction.vector3(shoulderwidth*self.BodyUpperMargin:x()*self.BoxScale,bodyheight/2*self.BodyUpperMargin:y()*self.BoxScale,shoulderwidth/3*self.BodyUpperMargin:z()*self.BoxScale));
        self.BODYUPPER:SetLocalPosition(modelshoulderpos);
        self.BODYUPPER:SetLocalRotation(shoulderrotate:ToQuaternion());

        self.BODYLOWER:SetLocalScale(mathfunction.vector3(hipwidth*self.BodyLowerMargin:x()*self.BoxScale,bodyheight/2*self.BodyLowerMargin:y()*self.BoxScale,hipwidth/3*self.BodyLowerMargin:z()*self.BoxScale));
        self.BODYLOWER:SetLocalPosition(modelhippos);
        self.BODYLOWER:SetLocalRotation(hiprotate:ToQuaternion());

        self.BODYAVG:SetLocalScale(mathfunction.vector3(shoulderwidth*self.BodyUpperMargin:x()*self.BoxScale,bodyheight*self.BodyUpperMargin:y()*self.BoxScale,shoulderwidth/3*self.BodyUpperMargin:z()*self.BoxScale));
        self.BODYAVG:SetLocalPosition(modelavgpos);
        self.BODYAVG:SetLocalRotation(avgrotate);
       
    end
    
    --self:UpdateAttachers(pos3d,scores);
    self:CaculateTransforms(pos3d,scores);
    self.super.SetCachedPoseForBluePrint(self);
end

for i=1, #SkeCubes do
    AvatarTest:MemberRegister(SkeCubes[i],
    venuscore.ScriptTypes.ReferenceType(
    apolloengine.TransformComponent:RTTI()
  )); 
end

AvatarTest:MemberRegister("usereproj");

AvatarTest:MemberRegister("BODYUPPER",
venuscore.ScriptTypes.ReferenceType(
apolloengine.TransformComponent:RTTI()
)); 

AvatarTest:MemberRegister("BODYLOWER",
venuscore.ScriptTypes.ReferenceType(
apolloengine.TransformComponent:RTTI()
)); 

AvatarTest:MemberRegister("BODYAVG",
venuscore.ScriptTypes.ReferenceType(
apolloengine.TransformComponent:RTTI()
)); 

AvatarTest:MemberRegister("camera",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.CameraComponent:RTTI(),
    AvatarTest.GetCamera,
    AvatarTest.SetCamera 
)); 


AvatarTest:MemberRegister("recognition",
  venuscore.ScriptTypes.ReferenceType(
    cv.RecognitionComponent:RTTI(),
    AvatarTest.GetRecognitionComponent,
    AvatarTest.SetRecognitionComponent 
)); 

AvatarTest:MemberRegister("headTrans",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.TransformComponent:RTTI()
)); 



AvatarTest:MemberRegister("BodyUpperMargin");
AvatarTest:MemberRegister("BodyLowerMargin");
AvatarTest:MemberRegister("cutoffscore");
AvatarTest:MemberRegister("scoreMin");
AvatarTest:MemberRegister("lateralErr");
AvatarTest:MemberRegister("offset");

-- rigister for blueprint
AvatarTest:MemberRegister("pos_R_Shoulder", nil, "none");
AvatarTest:MemberRegister("pos_R_Elbow",    nil, "none");
AvatarTest:MemberRegister("pos_R_Wrist",    nil, "none");
AvatarTest:MemberRegister("pos_L_Shoulder", nil, "none");
AvatarTest:MemberRegister("pos_L_Elbow",    nil, "none");
AvatarTest:MemberRegister("pos_L_Wrist",    nil, "none");
AvatarTest:MemberRegister("pos_R_Hip",      nil, "none");
AvatarTest:MemberRegister("pos_R_Knee",     nil, "none");
AvatarTest:MemberRegister("pos_R_Ankle",    nil, "none");
AvatarTest:MemberRegister("pos_L_Hip",      nil, "none");
AvatarTest:MemberRegister("pos_L_Knee",     nil, "none");
AvatarTest:MemberRegister("pos_L_Ankle",    nil, "none");
AvatarTest:MemberRegister("pos_Pelvis",     nil, "none");
-- rigister for blueprint
AvatarTest:MemberRegister("rot_R_Shoulder", nil, "none");
AvatarTest:MemberRegister("rot_R_Elbow",    nil, "none");
AvatarTest:MemberRegister("rot_R_Wrist",    nil, "none");
AvatarTest:MemberRegister("rot_L_Shoulder", nil, "none");
AvatarTest:MemberRegister("rot_L_Elbow",    nil, "none");
AvatarTest:MemberRegister("rot_L_Wrist",    nil, "none");
AvatarTest:MemberRegister("rot_R_Hip",      nil, "none");
AvatarTest:MemberRegister("rot_R_Knee",     nil, "none");
AvatarTest:MemberRegister("rot_R_Ankle",    nil, "none");
AvatarTest:MemberRegister("rot_L_Hip",      nil, "none");
AvatarTest:MemberRegister("rot_L_Knee",     nil, "none");
AvatarTest:MemberRegister("rot_L_Ankle",    nil, "none");
AvatarTest:MemberRegister("rot_Pelvis",     nil, "none");
AvatarTest:MemberRegister("rot_Spine",      nil, "none");
AvatarTest:MemberRegister("rot_Neck",       nil, "none");

return AvatarTest;