local venuscore = require "libvenuscore"
local Types = require "venuscore.rtti.types"
local BlueNode = require "bluecore.bluenode"
local BlueSelf = require "bluecore.blueself"
local EventDispatcher = require "bluecore.eventdispatcher"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local CV = require "computervisionfunction"
local RecognitionBase = BlueNode:extend();

local BREAK = 0
local TRIGGER = 1
local RELEASE = 2

RecognitionBase.BlueNodeExtend = BlueNode.extend;

-- 蓝图节点的继承,只能增加引脚
function RecognitionBase:extend(...)
  local clazz = RecognitionBase.BlueNodeExtend(self,...)
  for pinType, infos in pairs(RecognitionBase.infoTable) do
    -- shallow copy is ok
    for _, info in pairs(infos) do
      table.insert(clazz.infoTable[pinType], info)
    end
  end
  return clazz
end

---- 游戏运行模式
function RecognitionBase:new(...)
  RecognitionBase.super.new(self, ...)
  self:_OnReset();
  self.faceID = 1
end

function RecognitionBase:_OnUpdate(args)

  self._formatOutputArray = args

  local result = nil

  self.nextExecIdx = BREAK

  if self._formatOutputArray ~= nil and self._formatOutputArray[self.faceID] ~= nil then
    local edge = false
    if self.lastDetect == false then
      self.lastDetect = true
      edge = true
    end
    result = self._formatOutputArray[self.faceID]
    if self.isEdge and edge then
      self.nextExecIdx = TRIGGER
    elseif not self.isEdge then
      self.nextExecIdx = TRIGGER
    end
  else
    if self.lastDetect then
      self.lastDetect = false
      self.nextExecIdx = RELEASE -- 下一个输出执行引脚是RELEASE
    end
  end
  return result;
end

function RecognitionBase:_OnReset()
  self._formatOutputArray = {}
  self.lastDetect = false
  self.nextExecIdx = BREAK
end

function RecognitionBase:_OnNextBranch()
  return self.nextExecIdx
end

function RecognitionBase:RegisterTriggerFunc(func, bluePrintSelf)

  self:_GetInput()

  local comp = self.component or self.inputArgs[1]

  EventDispatcher:RegisterCallback(
          comp:GetContentPath(),
          self:_GetType(),
          func,
          bluePrintSelf)

  -- 调用blueprint reset会导致所有node的inputArgs清空
  self.faceID = self.inputArgs[2]
  if self.faceID <= 0 then
    self.faceID = 1
  end

  self.isEdge = self.inputArgs[3] or false
end

function RecognitionBase:_OnCompile()
  self.compileResult = nil
  if RecognitionBase.super._OnCompile(self) then

    local comp = self.component

    -- Step.0 判断反射面板有override
    if comp ~= nil then
      if venuscore.isNil(comp) then
        self.compileResult = BD.EHINT.COMP_LOST
        return false
      end
    end

    if comp == nil then
      -- Step.1 当反射面板没有override,则Component引脚要有连接
      if #self.inputs[1].links ~= 0 then
        local bluePinLink = self.inputs[1].links[1] -- 输入引脚只允许一个连接
        local otherNode = self.graph:GetNode(bluePinLink.nodeUid);

        -- Step.2 确保上一个节点是 引用组件类型 的节点
        if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp == nil then
          self.compileResult = BD.EHINT.NOT_LINK_TO_COMP
          return false
        end

        if venuscore.isNil(otherNode.comp) then
          self.compileResult = BD.EHINT.COMP_LOST
          return false
        end

        comp = otherNode.comp

      else
        self.compileResult = BD.EHINT.PIN_1_EMPTY;
        return false
      end

    end

    -- Step.3 RecognitionComponent没有激活当前的检测类型
    local hasType = self:_HasType(comp);
    if not hasType then
      self.compileResult = BD.EHINT.NOT_ENABLE;
      return false
    end

    return true ;
  end
  return false;
end

function RecognitionBase:_HasType(comp)
  return bit.band(comp.Type, self:_GetType()) ~= 0
end

function RecognitionBase:_OnDeserialize()

end

-- 蓝图节点反射面板相关
function RecognitionBase:_SetComp(comp)
  self.component = comp ;
end

function RecognitionBase:_GetComp()
  return self.component
end

function RecognitionBase:LiteralMismatchOnCompile(pin, pinInfo)
  if pin.argId ==  1 and BU:isBlueSelf(pin.literal) and  not venuscore.isNil(self.component) then
    return true
  end
  return false
end

function RecognitionBase:ReflectFilterFtnOnPrefab(obj, member, value)
  return member == "component"
end

RecognitionBase:MemberRegister("component",
        Types.ReferenceType(CV.RecognitionComponent:RTTI(),
                RecognitionBase._GetComp, RecognitionBase._SetComp));


---- 编辑器模式
if _KRATOSEDITOR then

function RecognitionBase:_OnLink(selfPin, otherNode, otherPin)
  if BU:IsInputPin(selfPin) and selfPin.argId == 1 then
    if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp then
      if venuscore.isNil(otherNode.comp) then
        ERROR(BD.EHINT.COMP_LOST)
      else
        local hasType = self:_HasType(otherNode.comp);
        if not hasType then
          ERROR(BD.EHINT.NOT_ENABLE)
        end
      end
    else
      ERROR(BD.EHINT.NOT_LINK_TO_COMP)
    end
  end
end

function RecognitionBase:_OnUnLink(unlinkPin)
  if unlinkPin.argId == 1 then -- 断开是第一个引脚
     ERROR(BD.EHINT.PIN_1_EMPTY)
  end
end

function RecognitionBase:_OnCreateWithEditor()
  RecognitionBase.super._OnCreateWithEditor(self);
end

function RecognitionBase:_OnUpdateByEditor()
  RecognitionBase.super._OnUpdateByEditor(self);
  self:_OnCompile();
end

end

RecognitionBase:RegisterInput(1, CV.RecognitionComponent:RTTI(), "component", "监听组件", BlueSelf());
RecognitionBase:RegisterInput(2, Types.IntType, "index", "face index(从1开始)", 1);
RecognitionBase:RegisterInput(3, Types.BoolType, "isEdge", "从无到有触发一次(OnTrigger)", false);


RecognitionBase:RegisterExecOutput(TRIGGER, "OnTrigger", "execute");
RecognitionBase:RegisterExecOutput(RELEASE, "OnRelease", "execute");

RecognitionBase:SetFunctionName("n/a");

RecognitionBase:SetFunctionType(BD.EVENT_FUNCTION_NODE);

RecognitionBase:SetEventType(BD.EventType.RegisterCallback);

return RecognitionBase;

